/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui.wizards;

import java.io.File;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.atf.generator.model.BuilderInfo;
import org.eclipse.atf.generator.model.MrCleanToolingInfo;
import org.eclipse.atf.generator.model.NatureInfo;
import org.eclipse.atf.generator.model.RuntimeInfo;
import org.eclipse.atf.generator.model.RuntimeInstanceInfo;
import org.eclipse.atf.generator.model.Utils;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.WizardResourceImportPage;
import org.eclipse.ui.internal.wizards.datatransfer.DataTransferMessages;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.wizards.datatransfer.FileSystemStructureProvider;
import org.eclipse.ui.wizards.datatransfer.IImportStructureProvider;


public class RuntimeArtifactsWizardPage extends WizardResourceImportPage {

	protected Combo sourceNameField;
	protected Button sourceBrowseButton;
	protected Button selectTypesButton;
	protected Button selectAllButton;
	protected Button deselectAllButton;
	protected Button smallIconBrowseButton;
	protected Button largeIconBrowseButton;
	protected Text smallIconField;
	protected Text largeIconField;
	protected Text projectNameField;
	protected Text projectFolderField;
	protected Text technologyField;
	protected Text pluginPrefixField;
	
	private MrCleanToolingInfo dataModel;
    private boolean entryChanged = false;
    private boolean initted = false;
    private String rootDirectory;
    private String destinationProject;
    private String projectFolder;
    private String techname;
    private String camelCasedTechnologyName;
    private String pluginPrefix;
    private String smallIconPath;
    private String largeIconPath;
    
    private static int MAX_SOURCE_NAME_HISTORY = 5;
    private static String SOURCE_NAME_HISTORY_SETTING = "SourceNameHistorySetting";
    private static String SOURCE_NAME_SETTING = "SourceNameSetting";
    private static String TECH_NAME_SETTING = "TechNameSetting";
    private static String PREFIX_NAME_SETTING = "PrefixNameSetting";
    private static String SMALL_ICON_SETTING = "SmallIconSetting";
    private static String LARGE_ICON_SETTING = "LargeIconSetting";
    private static String PROJECT_NAME_SETTING = "ProjectNameSetting";
    private static String PROJECT_FOLDER_SETTING = "ProjectFolderSetting";
	
	public RuntimeArtifactsWizardPage(IWorkbench aWorkbench, IStructuredSelection selection, MrCleanToolingInfo dataModel) {
		super("RuntimeArtifactsPage", selection);
		setTitle("New AJAX Runtime");
		setDescription("Specify the technology artifacts that are required at runtime.");
		
		
		RuntimeInfo rInfo = dataModel.getRuntimeInfo();
		if(rInfo == null) {
			rInfo = new RuntimeInfo();
			dataModel.setRuntimeInfo(rInfo);
		}
		
		RuntimeInstanceInfo riInfo = dataModel.getRuntimeInstanceInfo();
		if(riInfo == null) {
			riInfo = new RuntimeInstanceInfo();
			dataModel.setRuntimeInstanceInfo(riInfo);
		}
		
		NatureInfo natureInfo = dataModel.getNatureInfo();
		if(natureInfo == null) {
			natureInfo = new NatureInfo(null);
			natureInfo.setContainingPluginId(dataModel.getCoreProjectName());
			dataModel.setNatureInfo(natureInfo);
		}
		BuilderInfo builderInfo = dataModel.getBuilderInfo();
		if(builderInfo == null) {
			builderInfo = new BuilderInfo(null);
			builderInfo.setContainingPluginId(dataModel.getCoreProjectName());
			dataModel.setBuilderInfo(builderInfo);
		}
		
		this.dataModel = dataModel;
	}
	
	/** (non-Javadoc)
     * Method declared on IDialogPage.
     */
    public void createControl(Composite parent) {

        initializeDialogUnits(parent);
        
        Composite composite = new Composite(parent, SWT.NONE);
        composite.setFont(parent.getFont());
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        
		createTopGroup(composite);
		
        Group group = new Group(composite, SWT.NONE);
        group.setText("Runtime Artifacts");
        group.setLayout(new GridLayout());
        group.setLayoutData(new GridData(GridData.FILL_BOTH));
        group.setSize(group.computeSize(SWT.DEFAULT, SWT.DEFAULT));
        group.setFont(composite.getFont());
        		
        createSourceGroup(group);
        createDestinationProjectGroup(group);

        restoreWidgetValues();
        updateWidgetEnablements();
        setPageComplete(determinePageCompletion());

        setControl(composite);
    }
    
    protected void restoreWidgetValues() {
    	IDialogSettings settings = getDialogSettings();
        if (settings != null) {
            String[] sourceNames = settings.getArray(SOURCE_NAME_HISTORY_SETTING);
            if (sourceNames != null)
                sourceNameField.setItems(sourceNames);
            String sourceName = settings.get(SOURCE_NAME_SETTING);
            if(sourceName != null) {
            	sourceNameField.setText(sourceName);
            	updateFromSourceField();
            }
            String techName = settings.get(TECH_NAME_SETTING);
            if(techName != null)
            	technologyField.setText(techName);
            String prefix = settings.get(PREFIX_NAME_SETTING);
            if(prefix != null)
            	pluginPrefixField.setText(prefix);
            String smallIcon = settings.get(SMALL_ICON_SETTING);
            if(smallIcon != null)
            	smallIconField.setText(smallIcon);
            String largeIcon = settings.get(LARGE_ICON_SETTING);
            if(largeIcon!= null)
            	largeIconField.setText(largeIcon);
            String projectName = settings.get(PROJECT_NAME_SETTING);
            if(projectName != null)
            	projectNameField.setText(projectName);
            String projectFolder = settings.get(PROJECT_FOLDER_SETTING);
            if(projectFolder != null)
            	projectFolderField.setText(projectFolder);
        }
    }
    
    public void storeDialogSettings() {
    	IDialogSettings settings = getDialogSettings();
    	if(rootDirectory != null)
    		settings.put(SOURCE_NAME_SETTING, rootDirectory);
    	if(techname != null)
    		settings.put(TECH_NAME_SETTING, techname);
    	if(pluginPrefix != null)
    		settings.put(PREFIX_NAME_SETTING, pluginPrefix);
    	if(smallIconPath != null)
    		settings.put(SMALL_ICON_SETTING, smallIconPath);
    	if(largeIconPath != null)
    		settings.put(LARGE_ICON_SETTING, largeIconPath);
    	if(destinationProject != null)
    		settings.put(PROJECT_NAME_SETTING, destinationProject);
    	if(projectFolder != null)
    		settings.put(PROJECT_FOLDER_SETTING, projectFolder);
    		
    }
    
    public void initializeControlContents() {
		if(!initted) {
			String prefix = dataModel.getPluginProjectPrefix();
			if(prefix != null && projectNameField.getText().equals("")) {
				destinationProject = prefix + ".runtime";
				projectNameField.setText(destinationProject);
			}
			String techname = dataModel.getCamelCasedTechnologyName();
			if(techname != null && projectFolderField.getText().equals("")) {
				projectFolder = techname + "Ajax";
				projectFolderField.setText(projectFolder);
			}
			initted = true;
		}
	}
    
    protected void createTopGroup(Composite parent) {
    	
    	Composite composite = new Composite(parent, SWT.NONE);
    	composite.setFont(parent.getFont());
    	GridLayout layout = new GridLayout();
    	layout.numColumns = 3;
    	composite.setLayout(layout);
    	composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    	
    	Label techLabel = new Label(composite, SWT.NONE);
    	techLabel.setText("AJAX Runtime Name:");
    	
    	technologyField = new Text(composite, SWT.BORDER);
    	GridData gData = new GridData(GridData.FILL_HORIZONTAL);
    	gData.horizontalSpan = 2;
    	technologyField.setLayoutData(gData);
    	technologyField.addListener(SWT.Modify, this);
    	
    	Label prefixLabel = new Label(composite, SWT.NONE);
    	prefixLabel.setText("Generated Plugin(s) Prefix:");
    	
    	pluginPrefixField = new Text(composite, SWT.BORDER);
    	gData = new GridData(GridData.FILL_HORIZONTAL);
    	gData.horizontalSpan = 2;
    	pluginPrefixField.setLayoutData(gData);
    	pluginPrefixField.addListener(SWT.Modify, this);
    	
    	Label smallIconLabel = new Label(composite, SWT.NONE);
    	smallIconLabel.setText("Small Icon:");
    	smallIconField = new Text(composite, SWT.BORDER);
		smallIconField.addListener(SWT.Modify, this);
		smallIconField.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		smallIconBrowseButton = new Button(composite, SWT.PUSH);
		smallIconBrowseButton.setText("Browse...");
		smallIconBrowseButton.addListener(SWT.Selection, this);
		
		Label largeIconLabel = new Label(composite, SWT.NONE);
		largeIconLabel.setText("Large Icon:");
		largeIconField = new Text(composite, SWT.BORDER);
		largeIconField.addListener(SWT.Modify, this);
		largeIconField.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		largeIconBrowseButton = new Button(composite, SWT.PUSH);
		largeIconBrowseButton.setText("Browse...");
		largeIconBrowseButton.addListener(SWT.Selection, this);
		
    	
    }
	
	protected void createSourceGroup(Composite parent) {
		
		createRootDirectoryGroup(parent);
        createFileSelectionGroup(parent);
        createButtonsGroup(parent);

	}
	
	protected void createDestinationProjectGroup(Composite parent) {
        // project specification group
        Composite projectGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        projectGroup.setLayout(layout);
        projectGroup.setLayoutData(new GridData(
                GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL));
        projectGroup.setFont(parent.getFont());

        // project label
        Label projectLabel = new Label(projectGroup, SWT.NONE);
        projectLabel.setText("Runtime artifacts project:");
        projectLabel.setFont(parent.getFont());

        // project name entry field
        projectNameField = new Text(projectGroup, SWT.SINGLE | SWT.BORDER);
        projectNameField.addListener(SWT.Modify, this);
        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL
                | GridData.GRAB_HORIZONTAL);
        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        projectNameField.setLayoutData(data);
        projectNameField.setFont(parent.getFont());
        
        // folder label
        Label folderLabel = new Label(projectGroup, SWT.NONE);
        folderLabel.setText("Project destination folder:");
        folderLabel.setFont(parent.getFont());
        
        // folder name entry field
        projectFolderField = new Text(projectGroup, SWT.SINGLE | SWT.BORDER);
        projectFolderField.addListener(SWT.Modify, this);
        data = new GridData(GridData.HORIZONTAL_ALIGN_FILL
                | GridData.GRAB_HORIZONTAL);
        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        projectFolderField.setLayoutData(data);
        projectFolderField.setFont(parent.getFont());
       
    }
	
	/**
     *	Create the group for creating the root directory
     */
    protected void createRootDirectoryGroup(Composite parent) {
        Composite sourceContainerGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 3;
        sourceContainerGroup.setLayout(layout);
        sourceContainerGroup.setFont(parent.getFont());
        sourceContainerGroup.setLayoutData(new GridData(
                GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL));

        Label groupLabel = new Label(sourceContainerGroup, SWT.NONE);
        groupLabel.setText("Root directory:");
        groupLabel.setFont(parent.getFont());

        // source name entry field
        sourceNameField = new Combo(sourceContainerGroup, SWT.BORDER);
        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL
                | GridData.GRAB_HORIZONTAL);
        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        sourceNameField.setLayoutData(data);
        sourceNameField.setFont(parent.getFont());

        sourceNameField.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                updateFromSourceField();
            }
        });
        
        sourceNameField.addListener(SWT.Modify, this);

        sourceNameField.addKeyListener(new KeyListener() {
            /*
             * @see KeyListener.keyPressed
             */
            public void keyPressed(KeyEvent e) {
                //If there has been a key pressed then mark as dirty
                entryChanged = true;
            }

            /*
             * @see KeyListener.keyReleased
             */
            public void keyReleased(KeyEvent e) {
            }
        });

        sourceNameField.addFocusListener(new FocusListener() {
            /*
             * @see FocusListener.focusGained(FocusEvent)
             */
            public void focusGained(FocusEvent e) {
                //Do nothing when getting focus
            }

            /*
             * @see FocusListener.focusLost(FocusEvent)
             */
            public void focusLost(FocusEvent e) {
                //Clear the flag to prevent constant update
                if (entryChanged) {
                    entryChanged = false;
                    updateFromSourceField();
                }

            }
        });

        // source browse button
        sourceBrowseButton = new Button(sourceContainerGroup, SWT.PUSH);
        sourceBrowseButton.setText(DataTransferMessages.DataTransfer_browse);
        sourceBrowseButton.addListener(SWT.Selection, this);
        sourceBrowseButton.setLayoutData(new GridData(
                GridData.HORIZONTAL_ALIGN_FILL));
        sourceBrowseButton.setFont(parent.getFont());
        setButtonLayoutData(sourceBrowseButton);
    }
    
    /**
     * Creates the buttons for selecting specific types or selecting all or none of the
     * elements.
     *
     * @param parent the parent control
     */
    protected final void createButtonsGroup(Composite parent) {
        // top level group
        Composite buttonComposite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 3;
        layout.makeColumnsEqualWidth = true;
        buttonComposite.setLayout(layout);
        buttonComposite.setFont(parent.getFont());
        GridData buttonData = new GridData(GridData.VERTICAL_ALIGN_FILL
                | GridData.HORIZONTAL_ALIGN_FILL);
        buttonData.horizontalSpan = 2;
        buttonComposite.setLayoutData(buttonData);

        // types edit button
        selectTypesButton = createButton(buttonComposite,
                IDialogConstants.SELECT_TYPES_ID, "Filter Types...", false);

        SelectionListener listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                handleTypesEditButtonPressed();
            }
        };
        selectTypesButton.addSelectionListener(listener);
        setButtonLayoutData(selectTypesButton);

        selectAllButton = createButton(buttonComposite,
                IDialogConstants.SELECT_ALL_ID, "Select All", false);

        listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                setAllSelections(true);
            }
        };
        selectAllButton.addSelectionListener(listener);
        setButtonLayoutData(selectAllButton);

        deselectAllButton = createButton(buttonComposite,
                IDialogConstants.DESELECT_ALL_ID, "Deselect All", false);

        listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                setAllSelections(false);
            }
        };
        deselectAllButton.addSelectionListener(listener);
        setButtonLayoutData(deselectAllButton);

    }
    
    /**
     * Creates a new button with the given id.
     * <p>
     * The <code>Dialog</code> implementation of this framework method
     * creates a standard push button, registers for selection events
     * including button presses and registers
     * default buttons with its shell.
     * The button id is stored as the buttons client data.
     * Note that the parent's layout is assumed to be a GridLayout and 
     * the number of columns in this layout is incremented.
     * Subclasses may override.
     * </p>
     *
     * @param parent the parent composite
     * @param id the id of the button (see
     *  <code>IDialogConstants.*_ID</code> constants 
     *  for standard dialog button ids)
     * @param label the label from the button
     * @param defaultButton <code>true</code> if the button is to be the
     *   default button, and <code>false</code> otherwise
     */
    protected Button createButton(Composite parent, int id, String label,
            boolean defaultButton) {
        // increment the number of columns in the button bar
        ((GridLayout) parent.getLayout()).numColumns++;

        Button button = new Button(parent, SWT.PUSH);
        button.setFont(parent.getFont());

        GridData buttonData = new GridData(GridData.FILL_HORIZONTAL);
        button.setLayoutData(buttonData);

        button.setData(new Integer(id));
        button.setText(label);

        if (defaultButton) {
            Shell shell = parent.getShell();
            if (shell != null) {
                shell.setDefaultButton(button);
            }
            button.setFocus();
        }
        return button;
    }
    
    /**
     * Update the receiver from the source name field.
     */

    private void updateFromSourceField() {

        setSourceName(sourceNameField.getText());
        rootDirectory = sourceNameField.getText();
        //Update enablements when this is selected
        updateWidgetEnablements();
    }
    
    /**
     * Sets the source name of the import to be the supplied path.
     * Adds the name of the path to the list of items in the
     * source combo and selects it.
     *
     * @param path the path to be added
     */
    protected void setSourceName(String path) {

        if (path.length() > 0) {

            String[] currentItems = this.sourceNameField.getItems();
            int selectionIndex = -1;
            for (int i = 0; i < currentItems.length; i++) {
                if (currentItems[i].equals(path))
                    selectionIndex = i;
            }
            if (selectionIndex < 0) {
                int oldLength = currentItems.length;
                String[] newItems = new String[oldLength + 1];
                System.arraycopy(currentItems, 0, newItems, 0, oldLength);
                newItems[oldLength] = path;
                this.sourceNameField.setItems(newItems);
                selectionIndex = oldLength;
            }
            this.sourceNameField.select(selectionIndex);

            resetSelection();
        }
    }
    
    /**
     *	Repopulate the view based on the currently entered directory.
     */
    protected void resetSelection() {

        RuntimeArtifactElement currentRoot = getFileSystemTree();
        this.selectionGroup.setRoot(currentRoot);

    }
    
    /**
     *	Answer the root FileSystemElement that represents the contents of
     *	the currently-specified source.  If this FileSystemElement is not
     *	currently defined then create and return it.
     */
    protected RuntimeArtifactElement getFileSystemTree() {

        File sourceDirectory = getSourceDirectory();
        if (sourceDirectory == null)
            return null;

        return selectFiles(sourceDirectory,
                FileSystemStructureProvider.INSTANCE);
    }
    
    /**
     * Invokes a file selection operation using the specified file system and
     * structure provider.  If the user specifies files to be imported then
     * this selection is cached for later retrieval and is returned.
     */
    protected RuntimeArtifactElement selectFiles(
            final Object rootFileSystemObject,
            final IImportStructureProvider structureProvider) {

        final RuntimeArtifactElement[] results = new RuntimeArtifactElement[1];

        BusyIndicator.showWhile(getShell().getDisplay(), new Runnable() {
            public void run() {
                //Create the root element from the supplied file system object
                results[0] = createRootElement(rootFileSystemObject,
                        structureProvider);
            }
        });

        return results[0];
    }
    
    /**
     * Creates and returns a <code>FileSystemElement</code> if the specified
     * file system object merits one.  The criteria for this are:
     * Also create the children.
     */
    protected RuntimeArtifactElement createRootElement(
            Object fileSystemObject, IImportStructureProvider provider) {
        boolean isContainer = provider.isFolder(fileSystemObject);
        String elementLabel = provider.getLabel(fileSystemObject);

        // Use an empty label so that display of the element's full name
        // doesn't include a confusing label
        RuntimeArtifactElement dummyParent = new RuntimeArtifactElement(
                "", null, true);//$NON-NLS-1$
        dummyParent.setPopulated();
        RuntimeArtifactElement result = new RuntimeArtifactElement(
                elementLabel, dummyParent, isContainer);
        result.setFileSystemObject(fileSystemObject);

        //Get the files for the element so as to build the first level
        result.getFiles(provider);

        return dummyParent;
    }
    
    /**
     * Returns a File object representing the currently-named source directory iff
     * it exists as a valid directory, or <code>null</code> otherwise.
     */
    protected File getSourceDirectory() {
        return getSourceDirectory(this.sourceNameField.getText());
    }
    
    /**
     * Returns a File object representing the currently-named source directory iff
     * it exists as a valid directory, or <code>null</code> otherwise.
     *
     * @param path a String not yet formatted for java.io.File compatability
     */
    private File getSourceDirectory(String path) {
        File sourceDirectory = new File(getSourceDirectoryName(path));
        if (!sourceDirectory.exists() || !sourceDirectory.isDirectory()) {
            return null;
        }

        return sourceDirectory;
    }
    
    /**
     *	Answer the directory name specified as being the import source.
     *	Note that if it ends with a separator then the separator is first
     *	removed so that java treats it as a proper directory
     */
    private String getSourceDirectoryName(String sourceName) {
        IPath result = new Path(sourceName.trim());

        if (result.getDevice() != null && result.segmentCount() == 0) // something like "c:"
            result = result.addTrailingSeparator();
        else
            result = result.removeTrailingSeparator();

        return result.toOSString();
    }

	protected ITreeContentProvider getFileProvider() {
		
        return new WorkbenchContentProvider() {
            public Object[] getChildren(Object o) {
                if (o instanceof RuntimeArtifactElement) {
                    RuntimeArtifactElement element = (RuntimeArtifactElement) o;
                    return element.getFiles(
                            FileSystemStructureProvider.INSTANCE).getChildren(
                            element);
                }
                return new Object[0];
            }
        };
	}

	protected ITreeContentProvider getFolderProvider() {
		return new WorkbenchContentProvider() {
            public Object[] getChildren(Object o) {
                if (o instanceof RuntimeArtifactElement) {
                    RuntimeArtifactElement element = (RuntimeArtifactElement) o;
                    return element.getFolders(
                            FileSystemStructureProvider.INSTANCE).getChildren(
                            element);
                }
                return new Object[0];
            }

            public boolean hasChildren(Object o) {
                if (o instanceof RuntimeArtifactElement) {
                    RuntimeArtifactElement element = (RuntimeArtifactElement) o;
                    if (element.isPopulated())
                        return getChildren(element).length > 0;

                    //If we have not populated then wait until asked
                    return true;
                }
                return false;
            }
        };
	}
	
	/**
     *	Handle all events and enablements for widgets in this dialog
     *
     * @param event Event
     */
    public void handleEvent(Event event) {
        if (event.widget == sourceBrowseButton)
            handleSourceBrowseButtonPressed();
        else if(event.widget == projectNameField && event.type == SWT.Modify)
        	handleProjectNameModified();
        else if(event.widget == projectFolderField)
        	handleProjectFolderModified();
        else if(event.widget == sourceNameField)
        	handleSourceRootModified();
        else if(event.widget == technologyField)
        	handleTechnologyModified();
        else if(event.widget == pluginPrefixField)
        	handlePrefixModified();
        else if(event.widget == smallIconBrowseButton)
        	handleSmallIconBrowseButtonSelected();
        else if(event.widget == largeIconBrowseButton)
        	handleLargeIconBrowseButtonSelected();
        else if(event.widget == smallIconField)
        	handleSmallIconModified();
        else if(event.widget == largeIconField)
        	handleLargeIconModified();
        
        super.handleEvent(event);
        
        setPageComplete(determinePageCompletion());
    }
    
    protected void handleProjectNameModified() {
    	destinationProject = projectNameField.getText();
    	updateDataModel();
    	
    }
    
    protected void handleSourceRootModified() {
    	rootDirectory = sourceNameField.getText();
    	updateDataModel();
    }
    
    protected void handleTechnologyModified() {
    	techname = technologyField.getText();
    	camelCasedTechnologyName = getCamelCasedTechnologyName();
		if(techname.equals("")) {
			pluginPrefix = "";
			projectFolder = "";
		}
		else if(camelCasedTechnologyName != null && !camelCasedTechnologyName.equals("")){
			pluginPrefix = "com."+camelCasedTechnologyName;
			projectFolder = camelCasedTechnologyName + "Ajax";
		}
		pluginPrefixField.setText(pluginPrefix);
		projectFolderField.setText(projectFolder);
    	
		updateDataModel();
    }
    
    protected void handlePrefixModified() {
    	pluginPrefix = pluginPrefixField.getText();
    	if(!pluginPrefix.equals("")) {
    		destinationProject = pluginPrefix + ".runtime";
     	} else {
    		destinationProject = "";
    	}
    	projectNameField.setText(destinationProject);
    	updateDataModel();
    }
    
    protected void handleProjectFolderModified() {
    	projectFolder = projectFolderField.getText();
    	updateDataModel();
    }
    
    protected void handleSmallIconBrowseButtonSelected() {
    	String iconPath = smallIconField.getText();
		FileDialog dialog = new FileDialog(getWizard().getContainer().getShell());
		if(!iconPath.equals(""))
			dialog.setFileName(iconPath);
		dialog.setFilterExtensions(new String[] {"*.gif", "*.jpg", "*.png"});
		dialog.setText("Specify the location of the technology icon (16 x 16 pixels).");
		String newPath = dialog.open();
		if(newPath != null) {
			smallIconField.setText(newPath);
		}
    }
    
    protected void handleLargeIconBrowseButtonSelected() {
    	String iconPath = largeIconField.getText();
		FileDialog dialog = new FileDialog(getWizard().getContainer().getShell());
		if(!iconPath.equals(""))
			dialog.setFileName(iconPath);
		dialog.setFilterExtensions(new String[] {"*.gif", "*.jpg", "*.png"});
		dialog.setText("Specify the location of the technology icon (75 x 66 pixels).");
		String newPath = dialog.open();
		if(newPath != null) {
			largeIconField.setText(newPath);
		}
    }
    
    protected void handleSmallIconModified(){
    	smallIconPath = smallIconField.getText();
    }
    
    protected void handleLargeIconModified() {
    	largeIconPath = largeIconField.getText();
    }
    
    private String getCamelCasedTechnologyName() {
		if(techname != null) {
			String ccTechName = "";
			StringTokenizer strTok = new StringTokenizer(techname, " ");
			int i=0;
			while(strTok.hasMoreTokens()) {
				String token = strTok.nextToken();
				if(i == 0) {
					ccTechName += Utils.uncapitalize(token);
				} else {
					ccTechName += Utils.capitalize(token);
				}
				i++;
			}
			return ccTechName;
		}
		return null;
	}
        
    /**
     *	Open an appropriate source browser so that the user can specify a source
     *	to import from
     */
    protected void handleSourceBrowseButtonPressed() {

        String currentSource = this.sourceNameField.getText();
        DirectoryDialog dialog = new DirectoryDialog(
                sourceNameField.getShell(), SWT.SAVE);
        dialog.setText("Select root folder");
        dialog.setMessage("Select the root folder to start selecting runtime artifacts");
        dialog.setFilterPath(getSourceDirectoryName(currentSource));

        String selectedDirectory = dialog.open();
        if (selectedDirectory != null) {
            //Just quit if the directory is not valid
            if ((getSourceDirectory(selectedDirectory) == null)
                    || selectedDirectory.equals(currentSource))
                return;
            //If it is valid then proceed to populate
            setErrorMessage(null);
            setSourceName(selectedDirectory);
            selectionGroup.setFocus();
            //Save location in history
            IDialogSettings settings = getDialogSettings();
            String[] items = sourceNameField.getItems();
            if (settings != null) {
            	if(items.length > MAX_SOURCE_NAME_HISTORY) {
            		String[] itemsToStore = new String[MAX_SOURCE_NAME_HISTORY];
            		int index = 0;
            		for(int i=items.length-1; i>=(items.length-MAX_SOURCE_NAME_HISTORY); i--) {
            			itemsToStore[index] = items[i];
            			index++;
            		}
            		settings.put(SOURCE_NAME_HISTORY_SETTING, itemsToStore);
            	} else {
            		settings.put(SOURCE_NAME_HISTORY_SETTING, items);
            	}
            }
        }
    }
        
    public void setPageComplete(boolean complete) {
        super.setPageComplete(complete);
        if(complete) {
        	updateDataModel();
		}
    }
    
    /*
     * @see WizardDataTransferPage.determinePageCompletion.
     */
    protected boolean determinePageCompletion() {
    	if(!validateTechname()) {
    		return false;
    	} else if(!validatePluginPrefix()) {
    		return false;
    	} else if(!validateSmallIcon()) {
    		return false;
    	} else if(!validateLargeIcon()) {
    		return false;
    	} else if(!validateRootDirectory()) {
    		return false;
    	} else if(!validateSelectedResources()) {
    		return false;
    	} else if(!validateDestinationProject()) {
    		return false;
    	} else if(!validateProjectDestinationFolder()) {
    		return false;
    	} 
    	    	
    	setErrorMessage(null);
        return true;
    }
        
    protected boolean validateTechname() {
    	if(techname == null || techname.equals("")) {
    		setErrorMessage("Valid technology name must be specified.");
    		return false;
    	}
    	return true;
    }
    
    protected boolean validatePluginPrefix() {
    	if(pluginPrefix == null || pluginPrefix.equals("")) {
    		setErrorMessage("Valid plugin prefix must be specified.");
    		return false;
    	}
    	return true;
    }
    
    protected boolean validateSmallIcon() {
    	if(smallIconPath != null && !smallIconPath.equals("")) {
			File path = new File(smallIconPath);
			if(!path.exists() || !path.isFile()) {
				setErrorMessage("Valid small icon path must be specified.");
				return false;
			}
		}
    	return true;
    }
    
    protected boolean validateLargeIcon() {
    	if(largeIconPath != null && !largeIconPath.equals("")) {
			File path = new File(largeIconPath);
			if(!path.exists() || !path.isFile()) {
				setErrorMessage("Valid large icon path must be specified.");
				return false;
			}
		}
    	return true;
    }
    
    protected boolean validateRootDirectory() {
    	if(rootDirectory != null && !rootDirectory.equals("")) {
    		File root = new File(rootDirectory);
    		if(root.exists() && root.isDirectory())
    			return true;
    	}
    	setErrorMessage("Valid root directory must be specified.");
    	return false;
    }
    
    protected boolean validateDestinationProject() {
    	if(destinationProject != null && !destinationProject.equals("")) {
    		IProject p = ResourcesPlugin.getWorkspace().getRoot().getProject(destinationProject);
    		if(p != null && p.exists()) {
    			setErrorMessage("Specified destination project already exists.");
    			return false;
    		}
    	}
    	setErrorMessage("Valid runtime destination project must be specified.");
    	return true;
    }
    
    protected boolean validateProjectDestinationFolder() {
    	if(projectFolder == null || projectFolder.equals("")) {
    		setErrorMessage("Valid runtime project folder must be specified.");
    		return false;
    	}
    	return true;
    }
    
    protected boolean validateSelectedResources() {
    	if(getSelectedResources() == null || getSelectedResources().size() == 0) {
    		setErrorMessage("At least one runtime artifact must be specified.");
    		return false;
    	}
    	return true;
    }
    
    public void setVisible(boolean visible) {
		if(visible)
			initializeControlContents();
		super.setVisible(visible);
	}
    
    public void updateDataModel() {
		
		List list = getSelectedResources();
		File[] artifacts = new File[list.size()];
		for(int i=0; i<list.size(); i++) {
			RuntimeArtifactElement element = (RuntimeArtifactElement)list.get(i);
			File file = (File)element.getFileSystemObject();
			artifacts[i] = file;
		}
		dataModel.setTechnologyName(techname);
		dataModel.setPluginProjectPrefix(pluginPrefix);
		dataModel.setSmallIconSourcePath(smallIconPath);
		dataModel.setLargeIconSourcePath(largeIconPath);
		
		if( techname != null ){
			
			dataModel.getRuntimeInfo().setRuntimeName( techname );
			dataModel.getRuntimeInfo().setRuntimeId( getCamelCasedTechnologyName() );
			dataModel.getRuntimeInfo().setHandlerClassName( Utils.capitalize(techname) +"RuntimeHandler" );
			dataModel.getRuntimeInfo().setHandlerPackageName( "com."+techname+".runtime" );
			dataModel.getRuntimeInfo().setRuntimeInstallRoot( projectFolder );
			dataModel.getRuntimeInfo().setNature( dataModel.getCoreProjectName() + "." + dataModel.getNatureInfo().getNatureID() );
					
			dataModel.getRuntimeInstanceInfo().setRuntimeType( dataModel.getRuntimeInfo().getRuntimeId() );
			dataModel.getRuntimeInstanceInfo().setRuntimeName( dataModel.getRuntimeInfo().getRuntimeId()+"_embed" );
			dataModel.getRuntimeInstanceInfo().setRuntimeVersion("0.0.0" );
			dataModel.getRuntimeInstanceInfo().setRuntimeLocation( "platform:/plugin/"+destinationProject+"/resources" );
			dataModel.getRuntimeInstanceInfo().setArtifacts(artifacts);
			dataModel.getRuntimeInstanceInfo().setRootArtifactsLocation(rootDirectory);
			dataModel.getRuntimeInstanceInfo().setArtifactsPlugin(destinationProject);
		
		}
		
		dataModel.getNatureInfo().setArtifactsPluginId(destinationProject);
		
	}
	

}
