/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.generator.ui.wizards;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.atf.generator.model.MrCleanToolingInfo;
import org.eclipse.atf.generator.model.SnippetInfo;
import org.eclipse.atf.generator.model.VariableInfo;
import org.eclipse.atf.generator.ui.Snippet;
import org.eclipse.atf.generator.ui.SnippetsTableViewerPanel;
import org.eclipse.atf.generator.ui.TableInputListener;
import org.eclipse.atf.generator.ui.Variable;
import org.eclipse.atf.generator.ui.VariableList;
import org.eclipse.atf.generator.ui.VariablesTableViewerPanel;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;


public class SnippetsWizardPage extends WizardPage implements ModifyListener, SelectionListener {

	private SnippetsTableViewerPanel snippetsTablePanel;
	private VariablesTableViewerPanel variablesTablePanel;
	private Button newSnippetButton;
	private Button removeSnippetButton;
	private Button newVariableButton;
	private Button removeVariableButton;
	private Text snippetContentText;
	private Hashtable snippetVariablesHash = new Hashtable();
	boolean initted = false;
	
	private MrCleanToolingInfo dataModel;
	
	public SnippetsWizardPage(MrCleanToolingInfo dataModel) {
		super("New Snippets Wizard");
		setTitle("Code Snippet generation"); //$NON-NLS-1$
	    setDescription("Specify information to create code snippets");
		
	    this.dataModel = dataModel;
	        
	}
	
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setFont(parent.getFont());
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
				
		createSnippetWizardControls(composite);
				
		setControl(composite);
		Dialog.applyDialogFont(composite);

	}
	
	public void initializeControlContents() {
		if(!initted) {
			initted = true;
		}
	}
	
	protected void createSnippetWizardControls(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Composite labelComposite = new Composite(composite, SWT.NONE);
		labelComposite.setFont(parent.getFont());
		labelComposite.setLayout(new GridLayout());
		labelComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel = new Label(labelComposite, SWT.NONE);
		descLabel.setText("Define code snippets to be offered within the resulting development environment.");
		
		Composite buttonComposite = new Composite(composite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		buttonComposite.setLayout(gridLayout);
		buttonComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		newSnippetButton = new Button(buttonComposite, SWT.PUSH);
		newSnippetButton.setText("New Snippet");
		newSnippetButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		newSnippetButton.addSelectionListener(this);
		
		removeSnippetButton = new Button(buttonComposite, SWT.PUSH);
		removeSnippetButton.setText("Remove Snippet");
		removeSnippetButton.setEnabled(false);
		removeSnippetButton.setLayoutData(new GridData (GridData.HORIZONTAL_ALIGN_BEGINNING));
		removeSnippetButton.addSelectionListener(this);
		
		snippetsTablePanel = new SnippetsTableViewerPanel(composite, SWT.NONE);
		GridData gData = new GridData (GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH);
		snippetsTablePanel.setLayoutData (gData);
		snippetsTablePanel.addTableSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				snippetSelectionChanged();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
			
		});
		
		snippetsTablePanel.addTableInputListener(new TableInputListener() {

			public void tableInputModified(Object input) {
				setPageComplete(validatePage());			
			}
		
		});
		
		Composite labelComposite2 = new Composite(composite, SWT.NONE);
		labelComposite2.setFont(parent.getFont());
		labelComposite2.setLayout(new GridLayout());
		labelComposite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel2 = new Label(labelComposite2, SWT.NONE);
		descLabel2.setText("Define the variables that will be used to create the selected snippet");
		
		Composite buttonComposite2 = new Composite(composite, SWT.NONE);
		gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		buttonComposite2.setLayout(gridLayout);
		buttonComposite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		newVariableButton = new Button(buttonComposite2, SWT.PUSH);
		newVariableButton.setText("New Variable");
		newVariableButton.setEnabled(false);
		newVariableButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
		newVariableButton.addSelectionListener(this);
		
		removeVariableButton = new Button(buttonComposite2, SWT.PUSH);
		removeVariableButton.setText("Remove Variable");
		removeVariableButton.setEnabled(false);
		removeVariableButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
		removeVariableButton.addSelectionListener(this);
		
		variablesTablePanel = new VariablesTableViewerPanel(composite, SWT.NONE);
		gData = new GridData (GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH);
		variablesTablePanel.setLayoutData(gData);
		variablesTablePanel.addTableSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				variableSelectionChanged();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		variablesTablePanel.addTableInputListener(new TableInputListener() {

			public void tableInputModified(Object input) {
				setPageComplete(validatePage());			
			}
		
		});
		
		Composite labelComposite3 = new Composite(composite, SWT.NONE);
		labelComposite3.setFont(parent.getFont());
		labelComposite3.setLayout(new GridLayout());
		labelComposite3.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label descLabel3 = new Label(labelComposite3, SWT.NONE);
		descLabel3.setText("Enter the snippet content below. You may reference predefined variables using the ${varName} notation.");
		
		snippetContentText = new Text(composite, SWT.BORDER | SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
		gData = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH);
		gData.heightHint = 100;
		snippetContentText.setLayoutData(gData);
		snippetContentText.addModifyListener(this);
		snippetContentText.setEnabled(false);
	}
	
	protected void snippetSelectionChanged() {
		
		Snippet snippet = snippetsTablePanel.getSelectedSnippet();
		showSnippetInfo(snippet);
		if(snippet != null)
			removeSnippetButton.setEnabled(true);
		else
			removeSnippetButton.setEnabled(false);
	}
	
	protected void variableSelectionChanged() {
		
		Variable var = variablesTablePanel.getSelectedVariable();
		if(var != null && !var.isRequired())
			removeVariableButton.setEnabled(true);
		else
			removeVariableButton.setEnabled(false);
	}
	
	protected void showSnippetInfo(Snippet snippet) {
		if(snippet != null) {
			VariableList varList = (VariableList)snippetVariablesHash.get(snippet);
			if(varList == null) {
				varList = new VariableList();
				snippetVariablesHash.put(snippet, varList);
			}
			
			variablesTablePanel.setInput(varList);
			snippetContentText.setEnabled(true);
			snippetContentText.setText(snippet.getContent());
		}
	}
	
	public void setVisible(boolean visible) {
		if(visible)
			initializeControlContents();
		super.setVisible(visible);
	}
	
	protected boolean validatePage() {
		
		if(!validateSnippets())
			return false;
				
		setErrorMessage(null);
		return true;
	}
	
	protected boolean validateSnippets() {
		Snippet[] snippets = snippetsTablePanel.getSnippets();
		ArrayList snippetNames = new ArrayList();
		for(int i=0; i<snippets.length; i++) {
			//Check the snippet attributes
			if(snippets[i].getName() == null || snippets[i].getName().equals("")) {
				setErrorMessage("All snippets must have a name defined.");
				return false;
			} else if(snippetNames.contains(snippets[i].getName())) {
				setErrorMessage("Duplicate snippet name definition \""+snippets[i].getName()+"\".");
				return false;
			} 
			snippetNames.add(snippets[i].getName());
			if(snippets[i].getDisplayName() == null || snippets[i].getDisplayName().equals("")) {
				setErrorMessage("Snippet, \""+snippets[i].getName()+"\", does not have a display name defined.");
				return false;
			} else if(snippets[i].getContent().equals("")) {
				setErrorMessage("Snippet, \""+snippets[i].getName()+"\", does not have any content.");
				return false;
			}
			//Check the snippet variables
			VariableList varList = (VariableList)snippetVariablesHash.get(snippets[i]);
			if(varList == null)
				continue;
			ArrayList vars = varList.getVariables();
			ArrayList varNames = new ArrayList();
			for(int j=0; j<vars.size(); j++) {
				Variable var = (Variable)vars.get(j);
				if(var.getName() == null || var.getName().equals("")) {
					setErrorMessage("All variables for snippet \""+snippets[i].getName()+"\" must have a name defined.");
					return false;
				} else if(varNames.contains(var.getName())) {
					setErrorMessage("Duplicate variable name definition, \""+var.getName()+"\", for snippet \""+snippets[i].getName()+"\".");
					return false;
				} 
				varNames.add(var.getName());
				if(var.getType() == null || var.getType().equals("")) {
					setErrorMessage("Variable \""+var.getName()+"\", for snippet \""+snippets[i].getName()+"\", does not have a type defined.");
					return false;
				} else if(var.getDisplayName() == null || var.getDisplayName().equals("")) {
					setErrorMessage("Variable \""+var.getName()+"\", for snippet \""+snippets[i].getName()+"\", does not have a display name defined.");
					return false;
				}
			}
			
		}
		return true;
	}
		
	public void setPageComplete(boolean complete) {
        super.setPageComplete(complete);
        if(complete) {
        	updateDataModel();
		}
    }
	
	public void updateDataModel() {
		if(!initted)
			return;
		Snippet[] snippets = snippetsTablePanel.getSnippets();
		ArrayList snippetList = new ArrayList();
		for(int i=0; i<snippets.length; i++) {
			SnippetInfo snippetInfo = new SnippetInfo();
			snippetInfo.setName(snippets[i].getName());
			snippetInfo.setDisplayName(snippets[i].getDisplayName());
			snippetInfo.setTemplate(snippets[i].getContent());
			//create VariableInfo
			VariableList varList = (VariableList)snippetVariablesHash.get(snippets[i]);
			if(varList != null) {
				ArrayList vars = varList.getVariables();
				ArrayList varInfos = new ArrayList();
				for(int j=0; j<vars.size(); j++) {
					Variable var = (Variable)vars.get(j);
					VariableInfo varInfo = new VariableInfo();
					varInfo.setName(var.getName());
					varInfo.setDefaultValue(var.getDefaultValue());
					varInfo.setDisplayName(var.getDisplayName());
					varInfo.setType(var.getType());
					varInfo.setOptions(var.getOptions());
					varInfos.add(varInfo);
				}
				VariableInfo[] infos = new VariableInfo[varInfos.size()];
				infos = (VariableInfo[])varInfos.toArray(infos);
				snippetInfo.setVariables(infos);
			}
			snippetList.add(snippetInfo);
		}
		SnippetInfo[] snippetInfos = new SnippetInfo[snippetList.size()];
		snippetInfos = (SnippetInfo[])snippetList.toArray(snippetInfos);
		dataModel.setSnippetInfo(snippetInfos);
	}
	
	public void modifyText(ModifyEvent e) {
		Widget w = e.widget;
		if(w.equals(snippetContentText)) {
			Snippet selected = snippetsTablePanel.getSelectedSnippet();
			if(selected != null)
				selected.setContent(((Text)w).getText());
		}
		setPageComplete(validatePage());

	}

	public void widgetSelected(SelectionEvent e) {
		Widget w = e.widget;
		if(w.equals(newVariableButton)) {
			addNewVariable();
		} else if(w.equals(removeVariableButton)) {
			removeSelectedVariable();
		} else if(w.equals(newSnippetButton)) {
			addNewSnippet();
		} else if(w.equals(removeSnippetButton)) {
			removeSelectedSnippet();
		} 

	}
	
	protected void addNewVariable() {
		variablesTablePanel.addNewVariable();
		
	}
	
	protected void removeSelectedVariable() {
		variablesTablePanel.removeSelectedVariable();
		if(variablesTablePanel.getVariables().length == 0)
			removeVariableButton.setEnabled(false);
	}
	
	protected void addNewSnippet() {
		snippetsTablePanel.addNewSnippet();
		newVariableButton.setEnabled(true);
	}
	
	protected void removeSelectedSnippet() {
		Snippet snippet = snippetsTablePanel.getSelectedSnippet();
		//Remove snippet from table
		snippetsTablePanel.removeSelectedSnippet();
		//Remove associated variables from hash
		snippetVariablesHash.remove(snippet);
		//Clear the variables table
		variablesTablePanel.clear();
		
		//Clear the content text
		snippetContentText.setText("");
		snippetContentText.setEnabled(false);
		
		if(snippetsTablePanel.getSnippets().length == 0) {
			removeSnippetButton.setEnabled(false);
			newVariableButton.setEnabled(false);
			removeVariableButton.setEnabled(false);
		}
		
		/*Snippet[] remaining = snippetsTablePanel.getSnippets();
		if(remaining != null && remaining.length > 0) {
			showSnippetInfo(remaining[remaining.length-1]);
		}
		else {
			//Clear the content text
			snippetContentText.setText("");
			snippetContentText.setEnabled(false);
		}*/
						
	}

	public void widgetDefaultSelected(SelectionEvent e) {}

	

}
