/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.javascript.internal.validation;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;

import org.eclipse.atf.javascript.validator.JavaScriptValidatorPlugin;
import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.content.IContentDescription;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.core.runtime.content.IContentTypeManager;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentExtension3;
import org.eclipse.jface.text.IDocumentPartitioner;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.osgi.util.NLS;
import org.eclipse.wst.html.core.text.IHTMLPartitions;
import org.eclipse.wst.sse.core.internal.provisional.IModelManager;
import org.eclipse.wst.sse.core.internal.provisional.IStructuredModel;
import org.eclipse.wst.sse.core.internal.provisional.StructuredModelManager;
import org.eclipse.wst.sse.core.internal.provisional.text.IStructuredDocument;
import org.eclipse.wst.sse.core.internal.provisional.text.IStructuredDocumentRegion;
import org.eclipse.wst.sse.core.internal.provisional.text.IStructuredPartitioning;
import org.eclipse.wst.sse.ui.internal.reconcile.validator.ISourceValidator;
import org.eclipse.wst.validation.internal.core.Message;
import org.eclipse.wst.validation.internal.core.ValidationException;
import org.eclipse.wst.validation.internal.operations.IWorkbenchContext;
import org.eclipse.wst.validation.internal.operations.LocalizedMessage;
import org.eclipse.wst.validation.internal.operations.WorkbenchReporter;
import org.eclipse.wst.validation.internal.provisional.core.IMessage;
import org.eclipse.wst.validation.internal.provisional.core.IReporter;
import org.eclipse.wst.validation.internal.provisional.core.IValidationContext;
import org.eclipse.wst.validation.internal.provisional.core.IValidator;
import org.eclipse.wst.validation.internal.provisional.core.IValidatorJob;
import org.eclipse.wst.xml.core.internal.document.DocumentTypeAdapter;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel;
import org.osgi.framework.Bundle;

public abstract class JSAbstractValidator implements IValidator, ISourceValidator, IValidatorJob {

	private String _validationMessageKey;
	private boolean _onlyJavaScript;
	private static boolean noJSDT = !checkForJSDT();
	private static String [] jsdtValidator = {"org.eclipse.wst.jsdt.web.core.internal.validation.JsBatchValidator"};

	public JSAbstractValidator(String validationMessageKey) {
		_validationMessageKey = validationMessageKey;
	}

	public ISchedulingRule getSchedulingRule(IValidationContext helper) {
		return null;
	}

	public IStatus validateInJob(IValidationContext helper, IReporter reporter) throws ValidationException {
		IStatus status = Status.OK_STATUS;
		if (noJSDT){
			try {
				validate(helper, reporter);
			}
			catch (RuntimeException re) {
				Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
				status = new Status(IStatus.ERROR, bundle.getSymbolicName(), IStatus.ERROR, re.getMessage(), re);
				if (Platform.inDebugMode())
					Platform.getLog(bundle).log(status);
			}
		}
		return status;
	}

	protected IDocument _document;
	private String _source;

	public abstract void validateString(String script, int lineno, String uri, IReporter reporter) throws ValidationException;

	public void validateReader(Reader reader, String uri, IReporter reporter) throws ValidationException {
		if (noJSDT){
			StringBuffer script = new StringBuffer();
			char buf[] = new char[4096];
			int len = 0;
			try {
				while ((len = reader.read(buf, 0, 4096)) != -1) {
					script.append(buf, 0, len);
				}
				validateString(script.toString(), 0, uri, reporter);
			} catch (IOException ioe) {
				// TODO Auto-generated catch block
				ioe.printStackTrace();
			}
		}
	}

	/**
	 * Validate the whole document. This is a batch validation call (IValidator)
	 */
	public void validate(IValidationContext helper, IReporter reporter)
			throws ValidationException {
		if (noJSDT){
			if (reporter.isCancelled() == true) {
				throw new OperationCanceledException();
			}
	
			String[] deltaArray = helper.getURIs();
			if (deltaArray != null && deltaArray.length > 0) {
				validateDelta(helper, reporter);
			}
			else {
				validateFull(helper, reporter);
			}
		}
	}

	/**
	 * "As-you-type" validation, provided for ISourceValidation
	 * 
	 * Like IValidator#validate(IValidationContext helper, IReporter reporter)
	 * except passes the dirty region, so document validation can be better
	 * optimized.
	 * 
	 * @param dirtyRegion
	 * @param helper
	 * @param reporter
	 */
	public void validate(IRegion dirtyRegion, IValidationContext helper, IReporter reporter) {
//System.err.println("validate!"+this);
		if (noJSDT){
			if (reporter.isCancelled() == true) {
				throw new OperationCanceledException();
			}
	
			int offset = 0;
			String script;
	
			// Get the affected region as a string.
			if (_document instanceof IStructuredDocument) {
				// Get the entire SCRIPT
				IStructuredDocumentRegion region = ((IStructuredDocument)_document)
						.getRegionAtCharacterOffset(dirtyRegion.getOffset());
				script = region.getFullText();
				offset = region.getStartOffset();
			} else {
				// It's a .js file, pass the whole thing.
				script = _document.get();
			}
	
			String[] uris = helper.getURIs();
			String uri = (uris.length > 0) ? uris[0] : null;
	
			int lineno = 0;
			try {
				lineno = _document.getLineOfOffset(offset);
			} catch (BadLocationException ble) {
				ble.printStackTrace();
			}
	
			try {
				validateString(script, lineno, uri, reporter);
			} catch (ValidationException ve) {
				Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
				IStatus status = new Status(IStatus.ERROR, bundle.getSymbolicName(), IStatus.ERROR, ve.getMessage(), ve);
				if (Platform.inDebugMode())
					Platform.getLog(bundle).log(status);
			}
		}
	}
	
	/**
	 * As you type validation is getting "hooked up" to this IDocument.
	 * This is the instance of IDocument that the validator should
	 * operate on for each validate call.
	 * 
	 * @param document
	 */
	public void connect(IDocument document) {
		_document = document;
	}
	
	/**
	 * The same IDocument passed in from the connect() method.
	 * This indicates that as you type validation is "shutting down"
	 * for this IDocument.
	 * 
	 * @param document
	 */
	public void disconnect(IDocument document) {
		_document = null;
	}

	public void cleanup(IReporter reporter) {
		// nothing to do
	}

	/**
	 */
	private void validateDelta(IValidationContext helper, IReporter reporter)
			throws ValidationException {
		String[] deltaArray = helper.getURIs();
		for (int i = 0; i < deltaArray.length; i++) {
			String delta = deltaArray[i];
			if (delta == null)
				continue;
			IResource resource = getResource(delta);
			if (resource == null || !(resource instanceof IFile))
				continue;
			validateFile(helper, reporter, (IFile) resource);
		}
	}

	/**
	 */
	private void validateFile(IValidationContext helper, IReporter reporter, IFile file)
			throws ValidationException {
		if ((reporter != null) && (reporter.isCancelled() == true)) {
			throw new OperationCanceledException();
		}
		if (!shouldValidate(file)) {
			return;
		}
		
		reportValidatingFile(file, reporter);

		if (_onlyJavaScript) {
			// if it's not a structured document, assume it's all js.
			Reader reader = null;
			try {
				reader = new BufferedReader(new InputStreamReader(file.getContents(), file.getCharset()));
				StringBuffer script = new StringBuffer();
				char buf[] = new char[4096];
				int len = 0;
				while ((len = reader.read(buf, 0, 4096)) != -1) {
					script.append(buf, 0, len);
				}
				validateString(_source = script.toString(), 0, file.getFullPath().toPortableString(), reporter);
				_source = null;
			} catch (CoreException ce) {
				ce.printStackTrace();
				IStatus status = ce.getStatus();
				throw new ValidationException(new LocalizedMessage(IMessage.NORMAL_SEVERITY, status.getMessage())); //TODO i18n
//			} catch (RuntimeException re) {
//				re.printStackTrace();
//				throw re;
			} catch (IOException ioe) {
				ioe.printStackTrace();
				throw new ValidationException(new LocalizedMessage(IMessage.NORMAL_SEVERITY, ioe.getLocalizedMessage())); //TODO i18n
			} finally {
				if (reader != null)
					try {
						reader.close();
					} catch (IOException ioe) {
						ioe.printStackTrace();
						throw new ValidationException(new LocalizedMessage(IMessage.NORMAL_SEVERITY, ioe.getLocalizedMessage())); //TODO i18n
					}
			}
		} else {
			IDOMModel model = null;
			try {
				model = getModel(file.getProject(), file);
				if (model != null) validateDOM(reporter, file, model);
			} finally {
				if (model != null)releaseModel(model);
			}
		}
	}

	protected void reportValidatingFile(IFile file, IReporter reporter) {
		if (reporter != null) {
			String fileName = ""; //$NON-NLS-1$
			IPath filePath = file.getFullPath();
			if (filePath != null) {
				fileName = filePath.toString();
			}
			String args[] = new String[]{fileName};

			Message mess = new LocalizedMessage(IMessage.LOW_SEVERITY, NLS.bind(_validationMessageKey, args));
			mess.setParams(args);
			reporter.displaySubtask(this, mess);
		}
	}

	/**
	 */
	private void validateDOM(IReporter reporter, IFile file, IDOMModel model) throws ValidationException {
		if (file == null || model == null)
			return; // error

		IDOMDocument document = model.getDocument();
		if (document == null)
			return; // error

		if (!hasHTMLFeature(document))
			return; // ignore

		IStructuredDocument doc = model.getStructuredDocument();
		_document = doc;
		IDocumentPartitioner partitioner2 =
			((IDocumentExtension3)doc).getDocumentPartitioner(IStructuredPartitioning.DEFAULT_STRUCTURED_PARTITIONING);
		ITypedRegion[] regions = new ITypedRegion[0];
		int docLength = doc.getLength();
		regions = partitioner2.computePartitioning(0, docLength);
		for (int i=0; i<regions.length; i++) {
			String contentType = regions[i].getType();
			if (IHTMLPartitions.SCRIPT.equals(contentType)) {
				int offset = regions[i].getOffset();
				int length = regions[i].getLength();
				String uri = file.getFullPath().toPortableString();
				int lineno = doc.getLineOfOffset(offset);
				try {
					String script = doc.get(offset, length);
					validateString(script, lineno, uri, reporter);
				} catch (BadLocationException e) {
					// TODO Just keep processing regions for now
					e.printStackTrace();
				}
			}
		}

		//TODO: would be nice to identify and validate SCRIPT attributes like onclick as well
		// see Eclipse enhancement request #120963
		
//		NodeList scriptNodeList = document.getElementsByTagName("SCRIPT");
//		int scriptNodes = scriptNodeList.getLength();
//		for (int i = 0; i < scriptNodes; i++) {
//			Node scriptNode = scriptNodeList.item(i);
//			//TODO: could check for type (text/javascript) and/or version info
//			Node scriptContents = scriptNode.getFirstChild();
//			if (scriptContents != null && scriptContents.getNodeType() == Node.TEXT_NODE) {
//				String script = scriptContents.getNodeValue();
//				int lineno = 0; //TODO
//				String uri = file.getFullPath().toPortableString();
//				validateString(script, lineno, uri, reporter);
//			}
//		}
		_document = null;
	}

	/**
	 */
	private void validateFull(IValidationContext helper, IReporter reporter) throws ValidationException {
		IProject project = null;
		String[] fileDelta = helper.getURIs();
		if (helper instanceof IWorkbenchContext) {
			IWorkbenchContext wbHelper = (IWorkbenchContext) helper;
			project = wbHelper.getProject();
		}
		else {
			// won't work for project validation (b/c nothing in file delta)
			project = getResource(fileDelta[0]).getProject();
		}
		if (project == null)
			return;
		validateContainer(helper, reporter, project);
	}

	/**
	 */
	private void validateContainer(IValidationContext helper, IReporter reporter, IContainer container)
			throws ValidationException {
		try {
			IResource[] resourceArray = container.members(false);
			for (int i = 0; i < resourceArray.length; i++) {
				IResource resource = resourceArray[i];
				if (resource == null)
					continue;
				if (resource instanceof IFile) {
					validateFile(helper, reporter, (IFile) resource);
				}
				else if (resource instanceof IContainer) {
					validateContainer(helper, reporter, (IContainer) resource);
				}
			}
		}
		catch (CoreException ce) {
			ce.printStackTrace();
			IStatus status = ce.getStatus();
			throw new ValidationException(new LocalizedMessage(IMessage.NORMAL_SEVERITY, status.getMessage())); //TODO i18n
		}
	}

	/**
	 */
	protected IDOMModel getModel(IProject project, IFile file) {
		if (project == null || file == null)
			return null;
		if (!file.exists())
			return null;
//		if (!canHandle(file))
//			return null;

		IStructuredModel model = null;
		IModelManager manager = StructuredModelManager.getModelManager();

		try {
			try {
				model = manager.getModelForRead(file);
			}
			catch (UnsupportedEncodingException ex) {
				// retry ignoring META charset for invalid META charset
				// specification
				// recreate input stream, because it is already partially read
				model = manager.getModelForRead(file, new String(), null);
			}
		}
		catch (UnsupportedEncodingException uee) {
			uee.printStackTrace();
		}
		catch (IOException ioe) {
			ioe.printStackTrace();
		}
		catch (CoreException ce) {
			ce.printStackTrace();
		}

		if (model == null)
			return null;
		if (!(model instanceof IDOMModel)) {
			releaseModel(model);
			return null;
		}
		return (IDOMModel) model;
	}

	/**
	 */
	protected void releaseModel(IStructuredModel model) {
		if (model != null)
			model.releaseFromRead();
	}

	/*
	 * added to get rid or dependency on IWorkbenchHelper
	 * 
	 */
	private IResource getResource(String delta) {
		return ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(delta));
	}

	/**
	 */
	private boolean hasHTMLFeature(IDOMDocument document) {
		DocumentTypeAdapter adapter = (DocumentTypeAdapter) document.getAdapterFor(DocumentTypeAdapter.class);
		if (adapter == null)
			return false;
		return adapter.hasFeature("HTML");//$NON-NLS-1$
	}

	private static final String ORG_ECLIPSE_WST_JAVASCRIPT_CORE_JSSOURCE = "org.eclipse.wst.javascript.core.javascriptsource"; //$NON-NLS-1$
	private static final String ORG_ECLIPSE_JST_JSP_CORE_JSPSOURCE = "org.eclipse.jst.jsp.core.jspsource"; //$NON-NLS-1$
	private static final String ORG_ECLIPSE_WST_HTML_CORE_HTMLSOURCE = "org.eclipse.wst.html.core.htmlsource"; //$NON-NLS-1$
	// TODO: temp fix not to search AJAX runtimes.
	private static final String [] runtimeFolders = {"dojoAjax", "ricoAjax", "zimbraAjax"};
	private boolean shouldValidate(IFile file) {
		IResource resource = file;
		do {
			if (resource.isDerived() || resource.isTeamPrivateMember() || !resource.isAccessible() || (resource.getName().charAt(0) == '.' && resource.getType() == IResource.FOLDER)) {
				return false;
			}
			resource = resource.getParent();
		}
		while ((resource.getType() & IResource.PROJECT) == 0);

		IContentTypeManager contentTypeManager = Platform.getContentTypeManager();

		IContentDescription contentDescription;
		try {
			contentDescription = file.getContentDescription();
			IContentType jsContentType = contentTypeManager.getContentType(ORG_ECLIPSE_WST_JAVASCRIPT_CORE_JSSOURCE);
			IContentType htmlContentType = contentTypeManager.getContentType(ORG_ECLIPSE_WST_HTML_CORE_HTMLSOURCE);
			IContentType jspContentType = contentTypeManager.getContentType(ORG_ECLIPSE_JST_JSP_CORE_JSPSOURCE);
			if (contentDescription != null) {
				IContentType fileContentType = contentDescription.getContentType();
				
				boolean hasJavaScriptType = false;
				_onlyJavaScript = false;
				if (jsContentType != null) 
					if (fileContentType.isKindOf(jsContentType)) {
						hasJavaScriptType = true;
						_onlyJavaScript = true;
					}
				 
				if (htmlContentType != null) 
					if (fileContentType.isKindOf(htmlContentType)) {
						hasJavaScriptType = true;
						_onlyJavaScript = false;
					}
				
				if (jspContentType != null) 
					if (fileContentType.isKindOf(jspContentType)) {
						hasJavaScriptType = true;
						_onlyJavaScript = false;
					}
				 			
				if (hasJavaScriptType) {
						
					IPath path = file.getProjectRelativePath();
					// TODO: A temp change not to validate the runtime files.
					// 		 Need to implement a more gereral way to handle this. 
					//		 This plugin shouldn't have dependences on ATF
					String contentPath = FlexibleProjectUtils.getWebContentRelativePath(file);
					if (contentPath != null){
						if( !(FlexibleProjectUtils.getWebContentRelativePath(file).equals(file.getProjectRelativePath().toString()))) {
							path = path.removeFirstSegments(1); 
						}
						
						String folder = path.segment(0);	
						for (int i=0; i<runtimeFolders.length; i++){
							if (folder.equals(runtimeFolders[i])) return false;
						}
					}
					return true;
				}
			} 
		} catch (CoreException e) {
			// should be rare, but will ignore to avoid logging "encoding
			// exceptions" and the like here.
		}
		return false;		
	}

	protected int getLineOffset(int line) {
		if (_document != null) {
			try {
				return _document.getLineOffset(line);
			} catch (BadLocationException ble) {
				ble.printStackTrace();
			}
		} else if (_source != null) {
			// If we read the document from the IFile, we don't have a document.
			//TODO: Calculate from the source kept in a string by looking for the
			// nth occurrence of '\n'.  This is very costly.  Find a better way.
			int offset = 0;
			for (int i = 0; i < line; i++) {
				offset = _source.indexOf('\n', offset + 1);
				if (offset == -1)
					return IMessage.LINENO_UNSET;
			}

			if (_source.charAt(offset+1) == '\r')
				offset++;

			return offset+1;
		}

		return IMessage.LINENO_UNSET;
	}
	
	private static boolean checkForJSDT () {
		
		IContentTypeManager contentTypeManager = Platform.getContentTypeManager();
		IContentType[] contentTypes = contentTypeManager.getAllContentTypes();
		
		for (int i=0; i<contentTypes.length; i++){
			String id = contentTypes[i].getId();
			if (id.equals("org.eclipse.wst.jsdt.core.jsSource")) {
				return true;
			}
		}
		
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IProject[] projects = workspace.getRoot().getProjects();
		for (int j = 0; j < projects.length; j++) {
			IProject project = projects[j];
			//try {
				if (project.isOpen()) {
					try {
						if (project.hasNature("org.eclipse.wst.jsdt.core.jsNature")) {
							WorkbenchReporter.removeAllMessages(project, jsdtValidator, null);
						}
					} catch (CoreException e) {
						// Do nothing
					}
				}
		}
		
		return false;
	}
}