/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.javascript.internal.validation;

import java.io.IOException;
import java.io.Reader;

import org.eclipse.atf.javascript.validator.JavaScriptValidatorPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.wst.validation.internal.operations.LocalizedMessage;
import org.eclipse.wst.validation.internal.provisional.core.IMessage;
import org.eclipse.wst.validation.internal.provisional.core.IReporter;
import org.mozilla.javascript.CompilerEnvirons;
import org.mozilla.javascript.ErrorReporter;
import org.mozilla.javascript.EvaluatorException;
import org.mozilla.javascript.Parser;
import org.mozilla.javascript.ScriptOrFnNode;
import org.osgi.framework.Bundle;

public class JSSyntaxValidator extends JSAbstractValidator {

	public JSSyntaxValidator() {
		super(JSValidationMessages.MESSAGE_JSSYNTAX_VALIDATION_MESSAGE);
	}

	public void validateReader(Reader reader, String uri, IReporter reporter) {
		int lineno = 0;
//		System.err.println("JavascriptValidator: reader");
		long start = System.currentTimeMillis();
		parse(reader, uri, lineno, new RhinoErrorReporter(reporter));		
		long end = System.currentTimeMillis();
		if (Platform.inDebugMode()) {
			Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
			IStatus status = new Status(IStatus.INFO, bundle.getSymbolicName(), IStatus.INFO, 
					"Javascript Rhino parse time elapsed=" + (end - start), null);
			Platform.getLog(bundle).log(status);
		}
	}

	public void validateString(String script, int lineno, String uri, IReporter reporter) {
//		System.err.println("JavascriptValidator: lineno="+lineno+" script="+script);
		long start = System.currentTimeMillis();
		parse(script, uri, lineno, new RhinoErrorReporter(reporter));		
		long end = System.currentTimeMillis();
		if (Platform.inDebugMode()) {
			Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
			IStatus status = new Status(IStatus.INFO, bundle.getSymbolicName(), IStatus.INFO, 
					"Javascript Rhino parse time elapsed=" + (end - start), null);
			Platform.getLog(bundle).log(status);
		}
	}

	/**
	 * Use Rhino to parse the script, flagging errors and warnings along the way via the ErrorReporter interface
	 * 
	 * @param reader reader to be parsed
	 * @param uri uri of the script, for annotation only
	 * @param lineno beginning line number reference, for annotation only
	 * @param reporter callback for errors and warnings
	 */
	private void parse(Reader reader, String uri, int lineno,
			ErrorReporter reporter) {
		Parser p = createParser(reporter);
		ScriptOrFnNode node = null;
		try {
			node = p.parse(reader, uri, lineno);
		} catch (IOException ioe) {
			if (Platform.inDebugMode()) {
				Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
				IStatus status = new Status(IStatus.ERROR, bundle.getSymbolicName(), IStatus.ERROR, 
						"Javascript parse failed", ioe);
				Platform.getLog(bundle).log(status);
			}
		} catch (EvaluatorException e) {
			// Errors have been reported.  This exception is redundant for us.  Ignore.
		}

		if (node == null) {
			//TODO: under what conditions does this happen?  does this just mean there was at least one error?
			if (Platform.inDebugMode()) {
				Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
				IStatus status = new Status(IStatus.INFO, bundle.getSymbolicName(), IStatus.INFO, 
						"Javascript parse failed, returned null", null);
				Platform.getLog(bundle).log(status);
			}
			return;
		}
	}

	/**
	 * Use Rhino to parse the script, flagging errors and warnings along the way via the ErrorReporter interface
	 * 
	 * @param script script to be parsed
	 * @param uri uri of the script, for annotation only
	 * @param lineno beginning line number reference, for annotation only
	 * @param reporter callback for errors and warnings
	 */
	private void parse(String script, String uri, int lineno,
			ErrorReporter reporter) {
		Parser p = createParser(reporter);
		ScriptOrFnNode node = null;
		try {
			node = p.parse(script, uri, lineno);
		} catch (EvaluatorException e) {
			// Errors have been reported.  This exception is redundant for us.  Ignore.
		}

		if (node == null) {
			//TODO: under what conditions does this happen?
			if (Platform.inDebugMode()) {
				Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
				IStatus status = new Status(IStatus.INFO, bundle.getSymbolicName(), IStatus.INFO, 
						"Javascript parse failed, returned null", null);
				Platform.getLog(bundle).log(status);
			}
			return;
		}
	}

	private Parser createParser(ErrorReporter reporter) {
		//TODO: is the default locale sufficient for Eclipse? Or do we need to explicitly pass a Locale?
		CompilerEnvirons env = new CompilerEnvirons(); //TODO: can this be shared across parser instances?
		env.setReservedKeywordAsIdentifier(true);
		return new Parser(env, reporter);
	}

	class RhinoErrorReporter implements ErrorReporter {

		private IReporter fReporter;
		private int constKeywordLine = -1;

		public RhinoErrorReporter(IReporter reporter) {
			fReporter = reporter;
		}
		
		private final String reservedWarning = "illegal usage of future reserved keyword ";
		private final String separatorError = "missing ; before statement";
		public void warning(String message, String sourceName, int line,
				String lineSource, int lineOffset) {
//System.out.println("Rhino warning in line "+line+": "+message);

			//KLUDGE: Checks against localized strings, so will only work in English.
			if (message != null && message.startsWith(reservedWarning)) {
				String keyword = message.substring(reservedWarning.length());
				keyword = keyword.substring(0, keyword.indexOf(' ') - 1);
				if ("const".equals(keyword)) {
					// eat the subsequent ';' syntax error, since const is treated as a keyword by
					// the parser and not a modifier.
					constKeywordLine = line;
					// Const is valid in Mozilla and may be used in XUL applications, but we
					// still report usage as a warning, since it is not valid in IE or anywhere else.
					if (System.getProperty("JSSyntaxValidator.xulMode") != null)
						return;
				} else if ("debugger".equals(keyword)) {
					// debugger is generally harmless and at least as long as we're using it for
					// breakpoints, we don't want it to raise flags.
					return;
				}
			}

			IMessage messageObject = new LocalizedMessage(IMessage.NORMAL_SEVERITY, message);
			messageObject.setLineNo(line);
			messageObject.setOffset(getLineOffset(line));
			messageObject.setLength(lineOffset);
			messageObject.setTargetObject(sourceName);
			fReporter.addMessage(JSSyntaxValidator.this, messageObject);
		}

		public void error(String message, String sourceName, int line,
				String lineSource, int lineOffset) {
//System.out.println("Rhino error in line "+line+": "+message);

			if (constKeywordLine == line) {
				if (separatorError.equals(message)) {
					constKeywordLine = -1;
					return;
				}
			}

			IMessage messageObject = new LocalizedMessage(IMessage.HIGH_SEVERITY, message);
			messageObject.setLineNo(line);
			messageObject.setOffset(getLineOffset(line));
			messageObject.setLength(lineOffset);
			messageObject.setTargetObject(sourceName);
			fReporter.addMessage(JSSyntaxValidator.this, messageObject);
		}

		public EvaluatorException runtimeError(String message,
				String sourceName, int line, String lineSource, int lineOffset) {
//System.out.println("runtime error in line "+line+": "+message);
			return new EvaluatorException(message);
		}

	}
}