/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.javascript.internal.validation.jslint;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.util.ArrayList;

import org.eclipse.atf.javascript.jslint.JSLintPlugin;
import org.eclipse.atf.javascript.validator.JavaScriptValidatorPlugin;
import org.eclipse.atf.javascript.validator.preferences.JSValidationPreferences;
import org.eclipse.atf.ui.util.ErrorStatusDisplayUtilities;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.Status;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.RhinoException;
import org.mozilla.javascript.Script;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.osgi.framework.Bundle;

public class JSLint {

	private static JSLint instance = null;
	
	public static JSLint getInstance(){
		if( instance == null ){
			instance = new JSLint();
		}
		
		return instance;
	}
	
	private Scriptable _scope, _options;
	private Function _lintFunction;
	
	//Reference to the File that contains jslint
	protected File jsLintFile = null;
	
	//determines if JSLint is ready to be used
	protected boolean ready = false;
	
	private JSLint(){
		
		//check preference, if not there then fallback to the embedded jslint (if available)
		if( JavaScriptValidatorPlugin.getDefault().getPluginPreferences().contains(JSValidationPreferences.JSLINT_LOCATION) ){
			
			jsLintFile = new File( JavaScriptValidatorPlugin.getDefault().getPluginPreferences().getString( JSValidationPreferences.JSLINT_LOCATION ) );
			
		}
		else{
			jsLintFile = getEmbeddedJSLintFile();
		}
		
		//init might fail, in which case this object is not ready
		init();
		
		//add to PropertyChangeListener to detect changes to the location of JSLint
		JavaScriptValidatorPlugin.getDefault().getPluginPreferences().addPropertyChangeListener( new Preferences.IPropertyChangeListener(){

			public void propertyChange( Preferences.PropertyChangeEvent event) {
				
				if( event.getProperty() == JSValidationPreferences.JSLINT_LOCATION ){
					if( event.getNewValue() != null && !"".equals(event.getNewValue())){
						jsLintFile = new File( (String)event.getNewValue() );
						
					}
					else{
						jsLintFile = getEmbeddedJSLintFile();
					}
					
					init();
				}
			}
			
		});
	}
	
	static private final String JSLINT_FILENAME = "resources/jslint.js";
	protected File getEmbeddedJSLintFile(){		
		try {
			URL platformURL = FileLocator.find(JSLintPlugin.getDefault().getBundle(), new Path(JSLINT_FILENAME), null);
			
			if( platformURL != null )
				return new File(FileLocator.toFileURL(platformURL).getPath() );
			else
				return null;
			
		} catch (IOException e) {
			return null;
		}
	}
	
	/*
	 * At this point the File
	 */
	synchronized protected void init(){
		ready = false; //changes are being made so disable JSLint		
		
		Context cx = Context.enter();
		Reader reader = null;
		try {	
			
			//If there is no file available, do not init and show a message
			if( jsLintFile == null || !jsLintFile.exists() ){
				IStatus status = new Status(IStatus.ERROR, JavaScriptValidatorPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
						"JSLint file not found. Check ATF Preferences.", null);
				throw new CoreException( status );
			}
			
			try {
				reader = new BufferedReader(new InputStreamReader(new FileInputStream(jsLintFile), "US-ASCII"));
			} catch (IOException e) {
				IStatus status = new Status(IStatus.ERROR, JavaScriptValidatorPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
						"Error reading JSLint file "+ jsLintFile.getName() +".", e);
				throw new CoreException( status );
			} 
			
			// encoding should match whatever is used in jslint.js
			
			
			Script lintScript;
			try {
				lintScript = cx.compileReader(reader, "jslint", 0, null);
			} catch (IOException e) {
				IStatus status = new Status(IStatus.ERROR, JavaScriptValidatorPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
						"Error compiling JSLint file "+ jsLintFile.getName() +".", e);
				throw new CoreException( status );
			}
			
			_scope = cx.initStandardObjects(null);
			lintScript.exec(cx, _scope);
			//Object jslint = _scope.get("jslint", _scope);
			Object jslint = cx.evaluateString(_scope, "var jslint,JSLINT;(jslint || JSLINT)", "_internal_jslint_lookup_", 1, null);
			if (!(jslint instanceof Function)) {
				if (Platform.inDebugMode()) {
					Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
					IStatus status = new Status(IStatus.ERROR, bundle.getSymbolicName(), IStatus.ERROR, 
							"lint is undefined or not a function.", null);
					Platform.getLog(bundle).log(status);
				}
			    //throw new ExceptionInInitializerError("Unable to initialize jslint");
				IStatus status = new Status(IStatus.ERROR, JavaScriptValidatorPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
						"Could not find the function \"jslint\" inside "+ jsLintFile.getName() +".", null);
				
				if (Platform.inDebugMode()) {
					Bundle bundle = JavaScriptValidatorPlugin.getDefault().getBundle();
					Platform.getLog(bundle).log(status);
				}
				
				throw new CoreException( status );
			}
			_lintFunction = (Function)jslint;

			_options = new ScriptableObject(_scope, null) {
				public String getClassName() {return "options";};
			};
			_options.put("debug", _options, Boolean.TRUE); // allow the "debugger" keyword
			_options.put("evil", _options, Boolean.TRUE); // allow use of eval()
			
			//make JSLint ready
			ready = true;
		}
		catch( CoreException ce ){
			//open a dialog			
			if( jsLintFile != null ) {
				if ( jsLintFile.exists() ){
					ErrorStatusDisplayUtilities.showErrorDialog("ATF Installation Error Detected", 
							"JSLint syntax validation will be disabled.", 
							ce.getStatus() );
				}
			}			
		}
		finally {
			Context.exit();
			if (reader != null)
				try {
					reader.close();
				} catch (IOException ioe) {
					ioe.printStackTrace();
				}
		}
	}
	
	public boolean isReady(){
		return ready;
	}
	
	private static final JSLintError[] noErrors = new JSLintError[]{};
	/*
	 * This is a wrapper for the JSLINT funtion in JavaScript
	 * 
	 * The Array of JSLintError can be empty, meaning that there was nothing to report.
	 * 
	 * To ensure that this call is successfull, the isReady method should be called
	 *
	 */
	synchronized public JSLintError[] lint( String source ) throws RhinoException{
		Assert.isTrue(ready);
		Context cx = Context.enter();
		try {
		    Object functionArgs[] = { source, _options };
		    Object result = _lintFunction.call(cx, _scope, _scope, functionArgs);
		    boolean lintFree = Context.toBoolean(result);
		    
		    //check if there are lint errros
		    if (!lintFree) {
		    	Scriptable lint = (Scriptable)_lintFunction.get("errors", _lintFunction);
		    	Object lengthObject = lint.get("length", lint);
		    	if (!(lengthObject instanceof Double)) {
		    		return noErrors;
		    	}
		    	int length = ((Double)lengthObject).intValue();
		    	
		    	ArrayList errorList = new ArrayList( length );
		    	for (int i = 0; i < length; i++) {

		    		Scriptable error = (Scriptable)lint.get(i, lint);
			    	
			    	if( error != null )
			    		errorList.add( new JSLintError(error) );
			    	else
			    		errorList.add( null );
		    	}
		    	
		    	JSLintError[] errorsArray = new JSLintError[ errorList.size() ];
		    	return (JSLintError[])errorList.toArray( errorsArray );
		    } 
		    else {
		    	return noErrors;
		    }
		} finally {
			Context.exit();
		}		
	}
	
	
}
