/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.javascript.validator.preferences;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;

import org.eclipse.atf.javascript.validator.JavaScriptValidatorPlugin;
import org.eclipse.atf.ui.util.StatusInfo;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.EcmaError;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.Script;
import org.mozilla.javascript.Scriptable;

/*
 * JSLint can be either embedded in a plugin or set by the user to an location in the file system.
 * This preference page is the means for the user to set the file location for JSLint.
 */
public class JSValidationPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {

	protected static final String PAGE_TITLE = "JavaScript Validation Settings";
	protected static final String PAGE_DESC = "Various settings used for JavaScript validation.";
	
	//holds the current location of JSLint set by the user
	protected String currentJSLintLocation = null;	
	
	
	//UI Components
	protected Text jsLintLocText = null;
	
	//validation status
	protected IStatus [] validationStatus = new IStatus[1];
	protected int mostCurrentBadStatus = -1; //this is so that the error message is for the last action that the user took
	protected static final int JSLINT_LOC_STATUS = 0;
	
	
	public JSValidationPreferencePage(){
		super();
		
		setTitle( PAGE_TITLE );
		setDescription( PAGE_DESC );
	}
	
	protected Control createContents(Composite parent) {
		initializeDialogUnits(parent);
		
		noDefaultAndApplyButton();
		
		//client area UI
		Composite displayArea = new Composite( parent, SWT.NONE );
		GridData layoutData = new GridData();
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.verticalAlignment = GridData.FILL;
		layoutData.grabExcessVerticalSpace = true;
		layoutData.grabExcessHorizontalSpace = true;
		displayArea.setLayoutData(layoutData);
		
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.marginHeight= 0;
		layout.marginWidth= 0;
		
		displayArea.setLayout( layout );
		
		//JSLint Location
		Label jsLintLocLabel = new Label( displayArea, SWT.LEFT | SWT.WRAP );
		jsLintLocLabel.setFont( displayArea.getFont() );
		jsLintLocLabel.setText( "JSLint Location:" );
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		jsLintLocLabel.setLayoutData(layoutData);
		
		jsLintLocText = new Text( displayArea, SWT.SINGLE | SWT.BORDER );
		jsLintLocText.setFont( displayArea.getFont() );

		if( currentJSLintLocation != null ){
			jsLintLocText.setText( currentJSLintLocation );
		}
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.widthHint = convertWidthInCharsToPixels(50);
		jsLintLocText.setLayoutData(layoutData);
		
		jsLintLocText.addModifyListener( new ModifyListener(){

			public void modifyText(ModifyEvent e) {
				validationStatus[JSLINT_LOC_STATUS] = validateJSLintLocation();
				
				if( !validationStatus[JSLINT_LOC_STATUS].isOK() ){
					mostCurrentBadStatus = JSLINT_LOC_STATUS;
				}
				
				updatePageStatus();					
			}
			
		});
		
		Button jsLintLocBrowseButton = new Button( displayArea, SWT.PUSH );
		jsLintLocBrowseButton.setFont( displayArea.getFont() );
		jsLintLocBrowseButton.setText( "Browse..." );
		jsLintLocBrowseButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event evt) {
				handleBrowseButton();
			}
		});
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.widthHint = 60;
		jsLintLocBrowseButton.setLayoutData(layoutData);
		
		applyDialogFont(parent);
		
		return parent;
	}

	public void init(IWorkbench workbench) {
		//get the current value of the jsLintLocation
		if( JavaScriptValidatorPlugin.getDefault().getPluginPreferences().contains(JSValidationPreferences.JSLINT_LOCATION) ){
			
			currentJSLintLocation = JavaScriptValidatorPlugin.getDefault().getPluginPreferences().getString( JSValidationPreferences.JSLINT_LOCATION );
			
		}			
	}
	
	public boolean performOk() {
		BusyIndicator.showWhile(null, new Runnable() {
			public void run() {
				
				if( jsLintLocText.getText() == null || "".equals(jsLintLocText.getText().trim()) ){
					JavaScriptValidatorPlugin.getDefault().getPluginPreferences().setToDefault( JSValidationPreferences.JSLINT_LOCATION );	
				}
				else{
					JavaScriptValidatorPlugin.getDefault().getPluginPreferences().setValue( JSValidationPreferences.JSLINT_LOCATION, jsLintLocText.getText() );
				}
			}
		});
		
		return super.performOk();
	}	

	protected IStatus validateJSLintLocation(){
		final StatusInfo status = new StatusInfo();
		
		//supports if user is just clearing the text box
		if( jsLintLocText.getText() == null || "".equals(jsLintLocText.getText().trim()) ){
			return status;
		}
		
		/*
		 * This code will load the JavaScript file using Rhino and validate that it contains
		 * the jslint function. It is inside a Runnable to show the user a busy indicator.
		 */
		BusyIndicator.showWhile(null, new Runnable() {
			public void run() {
				
				Context cx = Context.enter();
				Reader reader = null;
				try {
					
					reader = new BufferedReader(new InputStreamReader(new FileInputStream(jsLintLocText.getText()), "US-ASCII")); // encoding should match whatever is used in jslint.js
					Script lintScript = cx.compileReader(reader, "jslint", 0, null);
					Scriptable _scope = cx.initStandardObjects(null);
					lintScript.exec(cx, _scope);
					Object jslint = cx.evaluateString(_scope, "var jslint,JSLINT;(jslint || JSLINT)", "_internal_jslint_lookup_", 1, null);
					
					//if the function does not exist, it is not a valid jslint.js file
					if (!(jslint instanceof Function)) {
						status.setError( "The file specified does not contain JSLint." );
					}
				}
				catch( FileNotFoundException fnfe ){
					status.setError( "The file specified does not exist." );
				}
				catch( EcmaError ee ){
					status.setError( "The file specified contains errors." );
				}
				catch( Exception e ){
					e.printStackTrace();
					status.setError( e.getMessage() );
				}
				finally{
					if( reader != null )
						try {
							reader.close();
						} catch (IOException e) {}
				}
			}
		});
		
		
		return status;
	}
	
	protected void handleBrowseButton(){
		FileDialog dialog= new FileDialog(getShell());
		dialog.setFilterPath( jsLintLocText.getText());
		dialog.setText( "Select the JSLint JavaScript file:" );
		dialog.setFilterExtensions( new String[]{"*.js"} );
		String newPath= dialog.open();
		if (newPath != null) {
			jsLintLocText.setText(newPath);
		}
	}
	
	protected void updatePageStatus(){
		
		if( mostCurrentBadStatus >= 0 && !validationStatus[mostCurrentBadStatus].isOK()){
			setErrorMessage( validationStatus[mostCurrentBadStatus].getMessage() );
			setValid( false );
		}
		else{
			//find the first bad status from the list
			boolean foundBad = false;
			for (int i = 0; i < validationStatus.length; i++) {
				if( !validationStatus[i].isOK() ){
					foundBad = true;
					setErrorMessage( validationStatus[i].getMessage() );
					break;
				}
			}
			
			if( foundBad ){
				setValid(false);
			}
			else{
				//clear all errors
				setErrorMessage(null);
				setValid(true);
			}
		}
	}
}
