/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.personality.zimbra.core.builders;

import java.util.Map;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;


public abstract class ResourceDeltaBuilder extends IncrementalProjectBuilder implements IResourceDeltaVisitor {

	protected IProgressMonitor pMonitor = null;
	
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		IResourceDelta delta = getDelta(getProject());
		
		try {
			pMonitor = monitor;
			if(delta != null)
				delta.accept(this);
			
		} catch (CoreException ex) {
			CorePlugin.log(ex);
		}
		return null;
	}

	public boolean visit(IResourceDelta delta) throws CoreException {
		//Pull out resource
		try {
			IResource resource = delta.getResource();
			int kind = delta.getKind();
			switch (kind) {
				
				case IResourceDelta.ADDED :
					handleResourceAdd(resource);
					break;
				case IResourceDelta.ADDED_PHANTOM :
					handleResourcePhantomAdd(resource);
					break;
				case IResourceDelta.REMOVED :
					handleResourceRemove(resource);
					break;
				case IResourceDelta.CHANGED :
					handleResourceChanged(resource);
					break;
				case IResourceDelta.MOVED_FROM :
					handleResourceMovedFrom(resource);
					break;
				case IResourceDelta.MOVED_TO :
					handleResourceMovedTo(resource);
					break;
				case IResourceDelta.OPEN : 
					handleResourceOpen(resource);
					break;
			
			}
			
		} catch (Exception ex) {
			throw new CoreException(new Status(
					IStatus.ERROR,
					CorePlugin.PLUGIN_ID,
					IStatus.ERROR,
					"Error processing resource delta!",
					ex));
		}
		return true;
	}
	
	protected abstract void handleResourceAdd(IResource resource) throws Exception;

	protected abstract void handleResourcePhantomAdd(IResource resource) throws Exception;
	
	protected abstract void handleResourceChanged(IResource resource) throws Exception;
	
	protected abstract void handleResourceMovedFrom(IResource resource) throws Exception;
	
	protected abstract void handleResourceMovedTo(IResource resource) throws Exception;
	
	protected abstract void handleResourceOpen(IResource resource) throws Exception;
	
	protected abstract void handleResourceRemove(IResource resource) throws Exception;
}
