/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.personality.zimbra.core.builders;

import org.eclipse.atf.personality.zimbra.core.ZimbraCorePlugin;
import org.eclipse.atf.util.ErrorStatusUtilities;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

public class ZimbraImageBuilder extends ResourceDeltaBuilder {

	public static final String BUILDER_ID = "org.eclipse.atf.personality.zimbra.core.ImageBuilder";
	
			
	protected void handleResourceAdd(IResource resource) throws Exception {
		String extension = resource.getFileExtension();
		if (resource.getType() == IResource.FILE &&
				(extension.equals("gif") || extension.equals("jpg") || extension.equals("png"))) {
			IZimbraImageProcessor imageProcessor = getRegisteredImageProcessor(resource.getProject());
			if(imageProcessor != null) {
				try {
					imageProcessor.processImageAdd(resource);
				} catch(Exception e) {
					showErrorDialog(e);
					throw new CoreException(new Status(
							IStatus.ERROR,
							ZimbraCorePlugin.PLUGIN_ID,
							IStatus.ERROR,
							e.getMessage(),
							e));
				}
			}
		}
	}
	
	protected void handleResourceRemove(IResource resource) throws Exception {
		String extension = resource.getFileExtension();
		if (resource.getType() == IResource.FILE &&
				(extension.equals("gif") || extension.equals("jpg") || extension.equals("png"))) {
			IZimbraImageProcessor imageProcessor = getRegisteredImageProcessor(resource.getProject());
			if(imageProcessor != null) {
				try {
					imageProcessor.processImageRemove(resource);
				} catch(Exception e) {
					showErrorDialog(e);
					throw new CoreException(new Status(
							IStatus.ERROR,
							ZimbraCorePlugin.PLUGIN_ID,
							IStatus.ERROR,
							"Error processing image after remove!",
							e));
				}
			}
		}
	}
	
	protected void handleResourcePhantomAdd(IResource resource) throws Exception {}
	
	protected void handleResourceChanged(IResource resource) throws Exception {}
	
	protected void handleResourceMovedFrom(IResource resource) throws Exception {}
	
	protected void handleResourceMovedTo(IResource resource) throws Exception {}
	
	protected void handleResourceOpen(IResource resource) throws Exception {}
	
	private IZimbraImageProcessor getRegisteredImageProcessor(IProject project) throws CoreException {
		
		IZimbraImageProcessor lastProcessor = null;
		ImageBuilderExtensionReader reader = new ImageBuilderExtensionReader();
		
		String[] natureIds = project.getDescription().getNatureIds();
		for(int i=0; i<natureIds.length; i++) {
			IZimbraImageProcessor processor = reader.getImageProcessor(natureIds[i]);
			if(processor != null)
				lastProcessor = processor;
		}
		return lastProcessor;
	} 
	
	private void showErrorDialog(Exception e) {
		
		Shell shell = null;
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if(window == null) {
			IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
			if(windows.length != 0) {
				window = windows[0];
			}
		}
		if(window != null)
			shell = window.getShell();
		if(shell != null) {
			final Exception finalEx = e;
			final Class clz = this.getClass();
			shell.getDisplay().asyncExec(new Runnable() {
				public void run() {
					MultiStatus multiStatus = ErrorStatusUtilities.createMultiStatusFromException(finalEx, clz, ZimbraCorePlugin.PLUGIN_ID, "Error processing images\n" + finalEx.getMessage());
					ErrorDialog.openError(null, "Zimbra image processing failed", "Zimbra image processing failed", multiStatus);	
				}
			});
		}
		
	}
}
