/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.personality.zimbra.core.builders;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Map;

import org.eclipse.atf.personality.zimbra.core.ZimbraCorePlugin;
import org.eclipse.atf.personality.zimbra.runtime.IZimbraImageMergeAdapter;
import org.eclipse.atf.personality.zimbra.runtime.ZimbraImageMergeAdapter;
import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

public class ZimbraImageProcessor implements IZimbraImageProcessor {

	protected static final String AJAX_ROOT_DIR = "zimbraAjax";
	protected static final String PROCESSED_IMAGES_ROOT = AJAX_ROOT_DIR + "/img";
	protected static final String WEBINF_DIR = "WEB-INF";
	protected static final String METAINF_DIR = "META-INF";
	protected static final String WEBMODULE_DIR = "webmodule";
	protected static final String HI_RES_DIR = "hiRes";
	protected static final String LO_RES_DIR = "loRes";
	protected static final String CSS_FILENAME = "imgs.css";
	protected static final String GIF_EXTENSION = "gif";
	protected static final String JPG_EXTENSION = "jpg";
	protected static final String PNG_EXTENSION = "png";
	
	public void processImageAdd(IResource image) throws Exception {
		
		IProject project = image.getProject();
		IPath projectPath = image.getProjectRelativePath();
		if(!isValidImage(image))
			return;
				
		String webModuleName = getModuleName(image);	
		
		IFolder targetRelativeFolder = project.getFolder(getProcessedImagesRoot(image));
		
		//Make sure we're not reacting a resource change in the webmodule's target image folder
		if(targetRelativeFolder.getProjectRelativePath().isPrefixOf(projectPath))
			return;
		
		targetRelativeFolder = targetRelativeFolder.getFolder(WEBMODULE_DIR);
		if(!targetRelativeFolder.exists())
			targetRelativeFolder.create(true, true, null);
				
		//hi res folder
		IFolder hiResTargetFolder = targetRelativeFolder.getFolder(HI_RES_DIR);
		if(!hiResTargetFolder.exists()) {
			hiResTargetFolder.create(true, true, null);
		}
		else {
			//clear target directory
			File hiResDir = new File(hiResTargetFolder.getLocation().toString());
			deleteDirectoryContents(hiResDir);
		}
		
		//low res folder
		IFolder loResTargetFolder = targetRelativeFolder.getFolder(LO_RES_DIR);
		if(!loResTargetFolder.exists()) {
			loResTargetFolder.create(true, true, null);
		} else {
			//clear target directory
			File loResDir = new File(loResTargetFolder.getLocation().toString());
			deleteDirectoryContents(loResDir);
		}
		
		IFolder[] imageFolders = null;
		Map hiResCssMap = null;
		String loResCssPath = "/"+webModuleName+"/"+PROCESSED_IMAGES_ROOT+ "/" + WEBMODULE_DIR + "/" + LO_RES_DIR + "/";
		
		try {
			imageFolders = getImageFolders(project, webModuleName);
			hiResCssMap = getHiResCssValues(imageFolders);
		}
		catch(CoreException e) {
			ZimbraCorePlugin.log(e);
		}
		String[] locations = new String[imageFolders.length];
		for(int i=0; i<imageFolders.length; i++) {
			locations[i] = imageFolders[i].getLocation().toString();
		}
		
		IZimbraImageMergeAdapter imageMergeAdapter = getImageMergeAdapter();
		imageMergeAdapter.loadCssInfo(locations, hiResCssMap, loResCssPath, CSS_FILENAME);
		imageMergeAdapter.merge(hiResTargetFolder, true, false);
		imageMergeAdapter.merge(loResTargetFolder, false, false);
		
	}
	
	public void processImageRemove(IResource image) throws Exception {
		
		IProject project = image.getProject();
		IPath projectPath = image.getProjectRelativePath();
		if(!isValidImage(image))
			return;
		
		String webModuleName = getModuleName(image);	
		
		IFolder targetRelativeFolder = project.getFolder(getProcessedImagesRoot(image));
		
		//Make sure we're not reacting a resource change in the webmodule's target image folder
		if(targetRelativeFolder.getProjectRelativePath().isPrefixOf(projectPath))
			return;
		
		targetRelativeFolder = targetRelativeFolder.getFolder(WEBMODULE_DIR);
		if(!targetRelativeFolder.exists())
			return;
		
		IFolder[] imageFolders = null;
		Map hiResCssMap = null;
		String loResCssPath = "/"+webModuleName+"/"+PROCESSED_IMAGES_ROOT+ "/" + WEBMODULE_DIR + "/" + LO_RES_DIR + "/";
		try {
			imageFolders = getImageFolders(project, webModuleName);
			hiResCssMap = getHiResCssValues(imageFolders);
		}
		catch(CoreException e) {
			ZimbraCorePlugin.log(e);
		}
		String[] locations = new String[imageFolders.length];
		for(int i=0; i<imageFolders.length; i++) {
			locations[i] = imageFolders[i].getLocation().toString();
		}
		
		IZimbraImageMergeAdapter imageMergeAdapter = getImageMergeAdapter();
		imageMergeAdapter.loadCssInfo(locations, hiResCssMap, loResCssPath, CSS_FILENAME);
		
		//hi res folder
		IFolder hiResTargetFolder = targetRelativeFolder.getFolder(HI_RES_DIR);
		if(!hiResTargetFolder.exists()) {
			return;
		}
		else {
			//clear target directory
			File hiResDir = new File(hiResTargetFolder.getLocation().toString());
			deleteDirectoryContents(hiResDir);
						
			imageMergeAdapter.merge(hiResTargetFolder, true, false);
			
		}
		
		//low res folder
		IFolder loResTargetFolder = targetRelativeFolder.getFolder(LO_RES_DIR);
		if(!loResTargetFolder.exists()) {
			return;
		} 
		else {
			//clear target directory
			File loResDir = new File(loResTargetFolder.getLocation().toString());
			deleteDirectoryContents(loResDir);
						
			imageMergeAdapter.merge(hiResTargetFolder, false, false);
		}
		
		targetRelativeFolder.refreshLocal(IResource.DEPTH_INFINITE, null);
		
	}
	
	protected IZimbraImageMergeAdapter getImageMergeAdapter() {
		return new ZimbraImageMergeAdapter();		
	}
	
	protected String getProjectRelativeModuleDir(IResource image) {
		return "";
	}
	
	protected IPath getProcessedImagesRoot(IResource image) {
		try {
			IPath webContentPath = FlexibleProjectUtils.getWebContentFolder(image.getProject()).getProjectRelativePath();
			IPath root = new Path(getProjectRelativeModuleDir(image) + "/" + webContentPath.removeTrailingSeparator().toString() + "/" + PROCESSED_IMAGES_ROOT);
			return root.makeRelative();
		} catch(Exception e) {
			ZimbraCorePlugin.log(e);
			return null;
		}
	}
	
	protected boolean isValidImage(IResource image) {
		IPath projectPath = image.getProjectRelativePath();
		try {
			IPath webContentPath = FlexibleProjectUtils.getWebContentFolder(image.getProject()).getProjectRelativePath();
			if(projectPath.segment(0) == null || !(projectPath.matchingFirstSegments(webContentPath) == webContentPath.segmentCount()))
				return false;
			return true;
		} catch(Exception e) {
			ZimbraCorePlugin.log(e);
			return false;
		}
	}
	
	protected String getModuleName(IResource image) {
		return image.getProject().getName();
	}
		
	
	
	private Map getHiResCssValues(IFolder[] imageFolders) {
		
		Hashtable hash = new Hashtable();
		for(int i=0; i<imageFolders.length; i++) {
			String cssValue = "";
			String moduleName = getModuleName(imageFolders[i]);
			IPath webContentRelPath = new Path(FlexibleProjectUtils.getWebContentRelativePath(imageFolders[i]));
			if(webContentRelPath.segmentCount() != 0)
				cssValue = "/"+moduleName+"/"+webContentRelPath+"/";
			else
				cssValue = "/"+moduleName+"/";
			hash.put(imageFolders[i].getLocation().toString(), cssValue);
		}
		return hash;
	}
		
	private IFolder[] getImageFolders(IProject project, String webModuleName) throws Exception {
		
		IContainer webContentFolder = FlexibleProjectUtils.getWebContentFolder(project);
		if(!webContentFolder.exists())
			return null;
		
		ArrayList folders = new ArrayList();
		findImageFolders(folders, webContentFolder, webModuleName);
		IFolder[] imageFolders = new IFolder[folders.size()];
		imageFolders = (IFolder[])folders.toArray(imageFolders);
		return imageFolders;
	}
	
	private void findImageFolders(ArrayList folders, IContainer folder, String webModuleName) throws Exception {
		
		IResource[] members = folder.members();
		IPath webContentPath = FlexibleProjectUtils.getWebContentFolder(folder.getProject()).getProjectRelativePath();
		for(int i=0; i<members.length; i++) {
			IResource resource = members[i];
			if(resource instanceof IFolder) {
				IFolder f = (IFolder)resource;
				IPath fPath = f.getProjectRelativePath();
				String lastSegment = fPath.lastSegment();
				fPath = fPath.removeLastSegments(1);
				if(fPath.matchingFirstSegments(webContentPath) == webContentPath.segmentCount() &&
					(lastSegment.equals(WEBINF_DIR) || lastSegment.equals(AJAX_ROOT_DIR) || lastSegment.equals(METAINF_DIR)))
					continue;
				findImageFolders(folders, (IFolder)resource, webModuleName);
			}
			else if(resource instanceof IFile && 
					(((IFile)resource).getFileExtension().equals(GIF_EXTENSION) ||
						((IFile)resource).getFileExtension().equals(JPG_EXTENSION) ||
						((IFile)resource).getFileExtension().equals(PNG_EXTENSION))){
					//find the folder
					IContainer container = resource.getParent();
					if(container instanceof IFolder) {
						IFolder f = (IFolder)container;
						if(!folders.contains(f))
							folders.add(f);
						//break;
					}
					
			}
		}
		
	}
	
			
	/**
	 * Recursive method which deletes the specified directory and all of 
	 * its contents.
	 * @param directory
	 * @throws IOException
	 */
	private void deleteDirectoryContents(File directory) throws IOException {
		
		if(!directory.isDirectory())
			return;
		File[] children = directory.listFiles();
		for(int i=0; i<children.length; i++) {
			File file = children[i];
			if(file.isDirectory())
				deleteDirectoryContents(file);
			else
				file.delete();
		}
				
	}

}
