/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.personality.zimbra.core.runtime.handler;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.atf.natures.NatureUtil;
import org.eclipse.atf.personality.zimbra.core.ZimbraCorePlugin;
import org.eclipse.atf.personality.zimbra.core.natures.ZimbraNature;
import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.atf.runtime.installer.WARRuntimeInstaller;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.templates.Template;
import org.eclipse.jface.text.templates.TemplateBuffer;
import org.eclipse.jface.text.templates.TemplateContext;
import org.eclipse.jface.text.templates.TemplateContextType;
import org.eclipse.jface.text.templates.TemplateException;
import org.eclipse.jface.text.templates.TemplateTranslator;
import org.eclipse.wst.common.componentcore.internal.util.IModuleConstants;
import org.eclipse.wst.common.project.facet.core.IFacetedProject;
import org.eclipse.wst.common.project.facet.core.IProjectFacet;
import org.eclipse.wst.common.project.facet.core.ProjectFacetsManager;
import org.eclipse.wst.common.project.facet.core.internal.FacetedProjectNature;

public class EmbeddedZimbraRuntimeInstaller extends WARRuntimeInstaller {

	/*
	 * Names of files that are to be preprocessed must start with this prefix
	 */
	protected static final String PREPROCESS_MARKER = "_preprocess_";
	
	protected TemplateTranslator translator= new TemplateTranslator();
	protected TemplateContext templateContext = new TemplateContext( new TemplateContextType("internal") ){

		public boolean canEvaluate(Template template) {
			return true;
		}

		public TemplateBuffer evaluate(Template template) throws BadLocationException, TemplateException {
			if (!canEvaluate(template))
				return null;

			TemplateTranslator translator= new TemplateTranslator();
			TemplateBuffer buffer= translator.translate(template);

			getContextType().resolve(buffer, this);

			return buffer;
		}
		
	};
	
	public EmbeddedZimbraRuntimeInstaller(){
		
		templateContext.setVariable( "ajaxRoot", "zimbraAjax" );
	}
	
	
	public void install(IRuntimeInstance runtimeInstance, IProject project, IProgressMonitor monitor) throws CoreException {
		
		//make sure that this project is a J2EE project
		if( project.hasNature( FacetedProjectNature.NATURE_ID ) ){
			IFacetedProject fProject = ProjectFacetsManager.create( project );
			IProjectFacet facet = ProjectFacetsManager.getProjectFacet( IModuleConstants.JST_WEB_MODULE ); //"jst.web"
					
			//throw CoreE since Zimbra can only be installed on a J2EE project
			if( !fProject.hasProjectFacet(facet) ) {
				throw new CoreException( new Status(
						IStatus.ERROR,
						ZimbraCorePlugin.PLUGIN_ID,
						IStatus.ERROR,
						"Zimbra Tookit cannot be installed on this project.", null ) );
			}
		}
		
		
		//add the Zimbra Nature
		NatureUtil.addNature( project, ZimbraNature.NATURE_ID, monitor );
		
		//calculate the context root that will be needed for preprocessing
		String contextRoot = FlexibleProjectUtils.getServerContextRoot( project );
		if( contextRoot == null )
			contextRoot = project.getName();
		
		templateContext.setVariable( "contextRoot", contextRoot );
		
		super.install(runtimeInstance, project, monitor);
	}

	
	public void uninstall(IRuntimeInstance runtimeInstance, IProject project,
			IProgressMonitor monitor) throws CoreException {
		
		//add the Zimbra Nature
		NatureUtil.removeNature( project, ZimbraNature.NATURE_ID, monitor );
		
		super.uninstall(runtimeInstance, project, monitor);
	}


	/*
	 * Overrides the basic copy of a file to add a preprocessing phase.
	 * 
	 * Names of files prefixed with the PREPROCESS_MARKER will run through a variable 
	 * substitution phase before copy.
	 * 
	 * @see org.eclipse.atf.runtime.installer.WARRuntimeInstaller#copyFile(java.io.File, org.eclipse.core.resources.IContainer, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected void copyFile(File srcFile, IContainer targetContainer, IProgressMonitor monitor) throws FileNotFoundException, CoreException, IOException {
		monitor.subTask( "Copying file " + srcFile.getName() + "..." );
		
		IFile copyFile;
		
		InputStream sourceStream;
		
		if( srcFile.getName().startsWith(PREPROCESS_MARKER) ){
			//need to preprocess the file
			SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
			sub.setTaskName( "Preprocessing file..." );
			
			//1. read content into string
			StringBuffer buffer = new StringBuffer();
			try {
				FileReader reader = new FileReader( srcFile );
				

				int ch;
				while ((ch = reader.read()) != -1) {

					buffer.append((char) ch);
				}

			} catch (IOException ioe) {
				ioe.printStackTrace();
			}
			
			//2. Run through TemplateTranslator to get TemplateBuffer
			try {
				
				final String pattern = buffer.toString();
				Template t = new Template(){

					public String getPattern() {
						return pattern;
					}
					
				};
				
				TemplateBuffer evaluatedBuffer = templateContext.evaluate( t );
				sourceStream = new ByteArrayInputStream( evaluatedBuffer.getString().getBytes() );
				
			} 
			catch ( Exception  e) {
				throw new CoreException( new Status(
						IStatus.ERROR,
						ZimbraCorePlugin.PLUGIN_ID,
						IStatus.OK,
						"Error preprocessing file <"+srcFile.getAbsolutePath()+">", e) );
			}
			
			
			//created reference to copy of file without the prefix
			copyFile = targetContainer.getProject().getFile( targetContainer.getProjectRelativePath().append( srcFile.getName().substring(PREPROCESS_MARKER.length()) ) );
			
			sub.done();
		}
		else{
			sourceStream = new FileInputStream(srcFile);
			copyFile = targetContainer.getProject().getFile( targetContainer.getProjectRelativePath().append(srcFile.getName()) );
		}
		
		
		SubProgressMonitor sub = new SubProgressMonitor( monitor, 1 );
		sub.setTaskName( "Creating file in project..." );
		boolean isWebDescriptor = copyFile.getName().toLowerCase().equals("web.xml");
		if (!(copyFile.exists()) && ! isWebDescriptor     ) {
			copyFile.create(sourceStream, true, sub);
		} else if(!isWebDescriptor){
			copyFile.setContents(sourceStream, true, false, sub);		
		}
		sourceStream.close();
	}	
}
