/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.atf.server.http.core;

import java.net.URL;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.osgi.util.NLS;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IRuntime;
import org.eclipse.wst.server.core.IRuntimeType;
import org.eclipse.wst.server.core.IRuntimeWorkingCopy;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerType;
import org.eclipse.wst.server.core.IServerWorkingCopy;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.core.ServerPort;
import org.eclipse.wst.server.core.model.IURLProvider;
import org.eclipse.wst.server.core.model.ServerDelegate;

/**
 * Generic HTTP server.
 */
public class HttpServer extends ServerDelegate implements IURLProvider {

    public static final String PROPERTY_SECURE = "secure";
    public static final String PROPERTY_DEBUG = "debug";
    public static final String PROPERTY_TEST_ENVIRONMENT = "testEnvironment";

    public static final String BASE_URL = "base_url";
    public static final String URL_PREFIX = "url_prefix";
    //public static final String DOCUMENT_ROOT = "document_root";
    //public static final String PUBLISH = "publish";
    public static final String PORT = "port";
    public static final String PUB_DIR = "publish_directory";
    
    public static final String ID = "org.eclipse.atf.server.http";

    protected transient HttpServerConfiguration serverConfiguration;

    public static int count = 1;

    /**
     * HttpServer.
     */
    public HttpServer() {
        super();
    }

    protected void initialize() {
        super.initialize();
    }

    public String getContextRoot(IProject project) {
        
    	return project.getName();
    }

    public HttpRuntime getHttpRuntime() {
        if (getServer().getRuntime() == null)
            return null;

        return (HttpRuntime) getServer().getRuntime().loadAdapter(HttpRuntime.class, null);
    }

    public HttpServerConfiguration getHttpServerConfiguration() {
        if (this.serverConfiguration == null)
            this.serverConfiguration = new HttpServerConfiguration(this);

        return serverConfiguration;
    }

    public void importRuntimeConfiguration(IRuntime arg0, IProgressMonitor arg1) throws CoreException {
    	if( getHttpRuntime() != null ) {
	        if( !getHttpRuntime().publishToDirectory() ) {
				setAttribute("auto-publish-setting", 1);
			} else {
				setAttribute("auto-publish-setting", 2);
				setAttribute("auto-publish-time", 1);
			}
			setURLPrefix(getHttpRuntime().getPrefixPath());
			setPort(getHttpRuntime().getPort());
			setPublishDirectory(getHttpRuntime().getPublishLocation());
    	}
	}

    public void saveConfiguration(IProgressMonitor monitor) throws CoreException {
        HttpServerConfiguration config = getHttpServerConfiguration();
        if (config == null)
            throw new CoreException(new Status(IStatus.ERROR, HttpCorePlugin.PLUGIN_ID, 0, NLS.bind(Messages.errorCouldNotSaveConfiguration, "null"), null));
        config.save(getServer().getServerConfiguration(), monitor);
    }

    public void configurationChanged() {
        serverConfiguration = null;
    }

    public String getBaseURL() {
        return getAttribute(HttpServer.BASE_URL, "");
    }
    
    public boolean dontPublish() {
    	return getAttribute("auto-publish-setting", "2").equals("1");
    }

    public void setBaseURL(String url) {
        setAttribute(HttpServer.BASE_URL, url);
    }

    //public void setDocumentRoot(String docRoot) {
    //    setAttribute(HttpServer.DOCUMENT_ROOT, docRoot);
    //}

    //public String getDocumentRoot() {
    //    return getAttribute(HttpServer.DOCUMENT_ROOT, "");
    //}

    //public boolean canPublish() {
    //    return getAttribute(HttpServer.PUBLISH, true);
    //}

    //public void setPublish(boolean publish) {
    //    setAttribute(HttpServer.PUBLISH, publish);
    //}

    /**
     * Return the root URL of this module.
     * @param module org.eclipse.wst.server.core.model.IModule
     * @return java.net.URL
     */
    public URL getModuleRootURL(IModule module) {

        try {
            String port = getPortString();

            IServer server = this.getServer();
            String base = getBaseURL();
            if (base.equals(""))
                base = "http://" + server.getHost();

            URL url = null;

            if (port.equals("80"))
                url = new URL(base + "/");
            else
                url = new URL(base + ":" + port + "/");

            return url;

        } catch (Exception e) {
            Trace.trace(Trace.SEVERE, "Could not get root URL", e);
            return null;
        }

    }

    /**
     * Returns true if the process is set to run in debug mode.
     *
     * @return boolean
     */
    public boolean isDebug() {
        return getAttribute(PROPERTY_DEBUG, false);
    }

    /**
     * Returns true if this is a test (run code out of the workbench) server.
     *
     * @return boolean
     */
    public boolean isTestEnvironment() {
        return false;
    }

    /**
     * Returns true if the process is set to run in secure mode.
     *
     * @return boolean
     */
    public boolean isSecure() {
        return getAttribute(PROPERTY_SECURE, false);
    }

    protected static String renderCommandLine(String[] commandLine, String separator) {
        if (commandLine == null || commandLine.length < 1)
            return "";
        StringBuffer buf = new StringBuffer(commandLine[0]);
        for (int i = 1; i < commandLine.length; i++) {
            buf.append(separator);
            buf.append(commandLine[i]);
        }
        return buf.toString();
    }

    /**
     * Return a string representation of this object.
     * @return java.lang.String
     */
    public String toString() {
        return "HttpServer";
    }

    /*
     * Returns the child module(s) of this module.
     */
    public IModule[] getChildModules(IModule[] module) {
        return new IModule[0];
    }

    /*
     * Returns the root module(s) of this module.
     */
    public IModule[] getRootModules(IModule module) throws CoreException {
        return new IModule[] { module };
    }

    /**
     * Returns true if the given project is supported by this
     * server, and false otherwise.
     *
     * @param add modules
     * @param remove modules
     * @return the status
     */
    public IStatus canModifyModules(IModule[] add, IModule[] remove) {
        return new Status(IStatus.OK, HttpCorePlugin.PLUGIN_ID, 0, "%canModifyModules", null);
    }

    public ServerPort[] getServerPorts() {
        int port = getPort();
        ServerPort[] ports = { new ServerPort(null, null, port, "http") };

        return ports;
    }

    public int getPort() {
    	if( getHttpRuntime() != null ) {
	        String p = getAttribute(HttpServer.PORT, getHttpRuntime().getPort() );
	        return Integer.parseInt(p);
    	} else {
    		String p = getAttribute(HttpServer.PORT, "80" );
	        return Integer.parseInt(p);
    	}
    }

    public String getPortString() {
        return getAttribute(HttpServer.PORT, getHttpRuntime().getPort());
    }

    public void setPort(String port) {
        try {
            if (port.equals("")) {
                setAttribute(HttpServer.PORT, getHttpRuntime().getPort());
            } else {
                setAttribute(HttpServer.PORT, port);
            }
        } catch (Throwable e) {
            ;
        }
    }
    
    public void setURLPrefix(String prefix) {
    	try {
    		if(prefix != null)
    			setAttribute(HttpServer.URL_PREFIX, prefix);
    	} catch (Throwable e) {
    		;
    	}
    }
    
    public String getURLPrefix() {
    	if( getHttpRuntime() != null ) {
    		return getAttribute(HttpServer.URL_PREFIX, getHttpRuntime().getPrefixPath());
    	} else {
    		return getAttribute(HttpServer.URL_PREFIX, "");
    	}
    }

    public void setDefaults(IProgressMonitor monitor) {
        setTestEnvironment(true);
    }

    /**
     * Sets this process to debug mode. This feature only works
     *
     * @param b boolean
     */
    public void setDebug(boolean b) {
        setAttribute(PROPERTY_DEBUG, b);
    }

    /**
     * Sets this process to secure mode.
     * @param b boolean
     */
    public void setSecure(boolean b) {
        setAttribute(PROPERTY_SECURE, b);
    }

    /**
     * Sets this server to test environment mode.
     * 
     * @param b boolean
     */
    public void setTestEnvironment(boolean b) {
        setAttribute("testEnvironment", b);
    }

    /**
     * @see ServerDelegate#modifyModules(IModule[], IModule[], IProgressMonitor)
     */
    public void modifyModules(IModule[] add, IModule[] remove, IProgressMonitor monitor) throws CoreException {

    }

    public static IServer createHttpServer(String serverName, String baseURL, String ID) {

        String host = baseURL;
        if (baseURL.startsWith("http://"))
            host = host.substring(7);
        int index = host.indexOf("/");
        if (index != -1)
            host = host.substring(0, index - 1);
        index = host.indexOf(":");
        if (index != -1)
            host = host.substring(0, index - 1);

        IServer defaultServer = null;
        try {
            NullProgressMonitor monitor = new NullProgressMonitor();
            IRuntimeType runtimeType = ServerCore.findRuntimeType("org.eclipse.atf.server.http.runtime");
            IRuntimeWorkingCopy runtimeCopy = runtimeType.createRuntime("org.eclipse.atf.server.http.runtime", monitor);
            runtimeCopy.save(true, monitor);
            IRuntime runtime = runtimeCopy.getOriginal();
            IServerType serverType = ServerCore.findServerType("org.eclipse.atf.server.http");
            IServerWorkingCopy workingCopy = serverType.createServer(ID, null, runtime, monitor);
            workingCopy.setName(serverName);
            workingCopy.setHost(host);
            workingCopy.save(true, monitor);

            HttpServer as = (HttpServer) workingCopy.getAdapter(HttpServer.class);
            if (as == null)
                as = (HttpServer) workingCopy.loadAdapter(HttpServer.class, null);

            as.setBaseURL(baseURL);
            as.saveConfiguration(null);
            workingCopy.save(true, monitor);
            defaultServer = workingCopy.getOriginal();
        } catch (Exception e) {
            System.out.println(e);
        }

        return defaultServer;
    }

    public static IServer checkForHttpServer(String ID) {

        IServer[] servers = ServerCore.getServers();
        for (int i = 0; i < servers.length; i++) {
            if (servers[i].getId().equals(ID)) {
                return servers[i];
            }
        }
        return null;

    }

	public void setPublishDirectory(String pubDir) {
		setAttribute(PUB_DIR, pubDir);
	}
	
	public String getPublishDirectory() {
		if( getHttpRuntime() != null ) {
			return getAttribute(PUB_DIR, getHttpRuntime().getPublishLocation());
		} else {
			return getAttribute(PUB_DIR, "");
		}
	}

    /*public static void updateBaseURL(String id, String baseURL) {
        updateBaseURL(checkForHttpServer(id), baseURL);
    }*/

    /*public static void updateBaseURL(IServer server, String baseURL) {
        if (server == null)
            return;

        IServerWorkingCopy workingCopy = server.createWorkingCopy();

        HttpServer as = (HttpServer) workingCopy.getAdapter(HttpServer.class);
        if (as == null)
            as = (HttpServer) workingCopy.loadAdapter(HttpServer.class, null);

        String currentURL = as.getBaseURL();

        if (currentURL.equals(baseURL))
            return;
        as.setBaseURL(baseURL);
        try {
            as.saveConfiguration(null);
            workingCopy.save(true, null);
        } catch (CoreException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }*/
}
