/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.atf.server.http.core;

import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IDebugEventSetListener;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.model.IModuleResourceDelta;
import org.eclipse.wst.server.core.model.ServerBehaviourDelegate;

/**
 * Generic Http server.
 */
public class HttpServerBehaviour extends ServerBehaviourDelegate {

    // the thread used to ping the server to check for startup
    protected transient PingThread ping = null;
    protected transient IProcess process;
    protected transient IDebugEventSetListener processListener;
   
    /**
     * HttpServer.
     */
    public HttpServerBehaviour() {
        super();
    }

    public void initialize(IProgressMonitor monitor) {
        // do nothing
    }

    public HttpRuntime getHttpRuntime() {
        if (getServer().getRuntime() == null)
            return null;

        return (HttpRuntime) getServer().getRuntime().loadAdapter(HttpRuntime.class, null);
    }

    public HttpServer getHttpServer() {
        return (HttpServer) getServer().getAdapter(HttpServer.class);
    }

    /**
     * Returns the runtime base path for relative paths in the server
     * configuration.
     * 
     * @return the base path
     */
    public IPath getRuntimeBaseDirectory() {
        return getServer().getRuntime().getLocation();
       
    }

    /**
     * Return the program's runtime arguments to start or stop.
     *
     * @param starting true if starting
     * @return an array of runtime program arguments
     */
    protected String[] getRuntimeProgramArguments(boolean starting) {
        return null;
    }

    public void setProcess(final IProcess newProcess) {
        if (process != null)
            return;

        process = newProcess;
        if (processListener != null)
            DebugPlugin.getDefault().removeDebugEventListener(processListener);
        if (newProcess == null)
            return;

        processListener = new IDebugEventSetListener() {
            public void handleDebugEvents(DebugEvent[] events) {
                if (events != null) {
                    int size = events.length;
                    for (int i = 0; i < size; i++) {
                        Object obj = events[i].getSource();

                        if (!(obj instanceof IDebugTarget))
                            continue;

                        IDebugTarget target = (IDebugTarget) obj;
                        IProcess targetProcess = target.getProcess();

                        if (process != null && process.equals(targetProcess) && events[i].getKind() == DebugEvent.TERMINATE) {
                            DebugPlugin.getDefault().removeDebugEventListener(this);
                            //stopImpl();
                        }
                    }
                }
            }
        };
        DebugPlugin.getDefault().addDebugEventListener(processListener);
    }

    protected void setServerStarted() {
        setServerState(IServer.STATE_STARTED);
    }

    protected void stopImpl() {
        if (process != null) {
            process = null;
            DebugPlugin.getDefault().removeDebugEventListener(processListener);
            processListener = null;
        }

        setServerState(IServer.STATE_STOPPED);
    }

    protected void publishServer(int kind, IProgressMonitor monitor) throws CoreException {
        monitor = ProgressUtil.getMonitorFor(monitor);

        monitor.done();

        setServerPublishState(IServer.PUBLISH_STATE_NONE);
    }

    /**
     * Publish the project files into the server.
     * 
     * @param module
     * @param monitor
     * @return True, only if the publish process was successful
     * @throws CoreException
     */
    public boolean publish(IModule module, IProgressMonitor monitor) throws CoreException {
        if( module.getProject() != null ) {
	        IPath from = FlexibleProjectUtils.getWebContentFolder(module.getProject()).getLocation();
	        
	        String contextRoot = FlexibleProjectUtils.getServerContextRoot(module.getProject());
	
	        IPath to = new Path(getHttpServer().getPublishDirectory());
	        if (contextRoot != null && !contextRoot.equals(""))
	            to = to.append(contextRoot);
	        
	        String source = from.toOSString();
	        String dest = to.toOSString();
	        return FileUtil.smartCopyDirectory(source, dest, monitor);
        }
        return false;
    }

    /*
     * Publishes the given module to the server.
     */
    protected void publishModule(int kind, int deltaKind, IModule[] moduleTree, IProgressMonitor monitor) throws CoreException {
        if (moduleTree != null)
            publish(moduleTree[0], monitor);

        setModulePublishState(moduleTree, IServer.PUBLISH_STATE_NONE);
    }

    public void restart(String launchMode) throws CoreException {
        setServerState(IServer.STATE_STOPPED);
        setServerState(IServer.STATE_STARTED);
        return;
    }

    /**
     * Cleanly shuts down and terminates the server.
     * 
     * @param force <code>true</code> to kill the server
     */
    public void stop(boolean force) {
        setServerState(IServer.STATE_STOPPED);
    }

    /**
     * Terminates the server.
     */
    protected void terminate() {
        if (getServer().getServerState() == IServer.STATE_STOPPED)
            return;

        try {
            setServerState(IServer.STATE_STOPPING);
            Trace.trace(Trace.FINER, "Killing the HTTP process");
            if (process != null && !process.isTerminated()) {
                process.terminate();
                stopImpl();
            }
        } catch (Exception e) {
            Trace.trace(Trace.SEVERE, "Error killing the process", e);
        }
    }

    public IPath getTempDirectory() {
        return super.getTempDirectory();
    }

    /**
     * Return a string representation of this object.
     * @return java.lang.String
     */
    public String toString() {
        return "HttpServer";
    }

    protected IModuleResourceDelta[] getPublishedResourceDelta(IModule[] module) {
        return super.getPublishedResourceDelta(module);
    }
}
