/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.atf.server.http.core;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.wst.server.core.ServerPort;

public class HttpServerConfiguration {

    public static final String MODIFY_PORT_PROPERTY = "modifyPort";
    public static final String MODIFY_URL_PREFIX_PROPERTY = "modifyURLPrefix";
    public static final String MODIFY_PUB_DIR = "modifyPublishDirectory";

    protected String port = null;
    protected String documentRoot = null;
    protected String location = null;
    protected ArrayList propertyListeners = null;
    protected HttpServer server = null;
    protected boolean isServerDirty;

    public HttpServerConfiguration(HttpServer apacheServer) {
        this.server = apacheServer;
    }
    
    public List getServerPorts() {
		List ports = new ArrayList();
	
		// first add server port
		try {
			int port = server.getPort();
			ports.add(new ServerPort("server", Messages.portServer, port, "HTTP"));
		} catch (Exception e) {
			// ignore
		}
		return ports;
	}
    
    public String getPublishDirectory() {
    	return server.getPublishDirectory();
    }
    
    /**
	 * Modify the port with the given id.
	 *
	 * @param id java.lang.String
	 * @param port int
	 */
	public void modifyServerPort(String id, int port) {
		try {
			if ("server".equals(id)) {
				server.setPort(port + "");
				isServerDirty = true;
				firePropertyChangeEvent(MODIFY_PORT_PROPERTY, id, new Integer(port));
				return;
			}
		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error modifying server port " + id, e);
		}
	}
	
	public void modifyPublishDirectory(String pubDir) {
		try {
			String oldValue = server.getPublishDirectory();
			if( !oldValue.equals(pubDir) ) {
				server.setPublishDirectory(pubDir);
				isServerDirty = true;
				firePropertyChangeEvent(MODIFY_PUB_DIR, oldValue, pubDir);
			}
		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error modify publish directory: "+pubDir, e);
		}
	}
	
	public void modifyURLPrefix(String prefix) {
		try {
			String oldValue = server.getURLPrefix();
			if(oldValue == prefix)
				return;
			server.setURLPrefix(prefix);
			isServerDirty = true;
			firePropertyChangeEvent(MODIFY_URL_PREFIX_PROPERTY, oldValue, prefix);
			return;
		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error modifying URL prefix with value; " + prefix, e);
		}
	}
	
	public String getServerURLPrefix() {
		return server.getURLPrefix();
	}
	
    public void load(IPath path, IProgressMonitor monitor) throws CoreException {
    }

    public void load(IFolder folder, IProgressMonitor monitor) throws CoreException {
    }

    public void save(IFolder folder, IProgressMonitor monitor) throws CoreException {
    }

    protected void firePropertyChangeEvent(String propertyName, Object oldValue, Object newValue) {
		if (propertyListeners == null)
			return;
	
		PropertyChangeEvent event = new PropertyChangeEvent(this, propertyName, oldValue, newValue);
		try {
			Iterator iterator = propertyListeners.iterator();
			while (iterator.hasNext()) {
				try {
					PropertyChangeListener listener = (PropertyChangeListener) iterator.next();
					listener.propertyChange(event);
				} catch (Exception e) {
					Trace.trace(Trace.SEVERE, "Error firing property change event", e);
				}
			}
		} catch (Exception e) {
			Trace.trace(Trace.SEVERE, "Error in property event", e);
		}
	}
    
    /**
     * Adds a property change listener to this server.
     *
     * @param listener java.beans.PropertyChangeListener
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        if (propertyListeners == null)
            propertyListeners = new ArrayList();
        propertyListeners.add(listener);
    }

    /**
     * Removes a property change listener from this server.
     *
     * @param listener java.beans.PropertyChangeListener
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        if (propertyListeners != null)
            propertyListeners.remove(listener);
    }
}
