/*******************************************************************************
Copyright (c) 2006, 2007 Zend Corporation, IBM Corporation and others. * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.atf.server.http.ui.wizard;

import org.eclipse.atf.server.http.core.IHttpRuntimeWorkingCopy;
import org.eclipse.atf.server.http.ui.ContextIds;
import org.eclipse.atf.server.http.ui.HttpUIPlugin;
import org.eclipse.atf.server.http.ui.Messages;
import org.eclipse.atf.server.http.ui.Trace;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.wst.server.core.IRuntime;
import org.eclipse.wst.server.core.IRuntimeWorkingCopy;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.ui.wizard.IWizardHandle;

/**
 * Wizard page to set the server install directory.
 */
public class HttpRuntimeComposite extends Composite {

	protected IRuntimeWorkingCopy runtimeWc;
	protected IHttpRuntimeWorkingCopy runtime;
	
	protected Text name;
	protected Text prefix;
	protected Text port;
	protected Combo combo;
	protected Button publishCheckBox;
	protected Text publishDir;
	protected Button browseButton;
	protected Label locationLabel;
	protected Label portLabel;
	protected Label prefixLabel;
	private ValuesCache originalValuesCache = new ValuesCache();
	private ValuesCache modifiedValuesCache;

	protected IWizardHandle wizard;
	
	/**
	 * ServerCompositeFragment
	 * 
	 * @param parent the parent composite
	 * @param wizard the wizard handle
	 */
	public HttpRuntimeComposite(Composite parent, IWizardHandle wizard) {
		super(parent, SWT.NONE);
		this.wizard = wizard;
		wizard.setTitle(Messages.wizardTitle);
		wizard.setDescription(Messages.wizardDescription);
		wizard.setImageDescriptor(HttpUIPlugin.getImageDescriptor(HttpUIPlugin.IMG_WIZ_SERVER));
		
		createControl();
	}
	
	
	protected void setRuntime(IRuntimeWorkingCopy newRuntime) {
		if (newRuntime == null) {
			runtimeWc = null;
			runtime = null;
		} else {
			runtimeWc = newRuntime;
			runtime = (IHttpRuntimeWorkingCopy) newRuntime.loadAdapter(IHttpRuntimeWorkingCopy.class, null);
		}
		
		init();
		validate();
	}
	
	/**
	 * Provide a wizard page to change the Apache installation directory.
	 */
	protected void createControl() {
		GridLayout layout = new GridLayout(1, true);
		setLayout(layout);
		setLayoutData(new GridData(GridData.FILL_BOTH));

		Composite nameGroup = new Composite(this, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 2;
		nameGroup.setLayout(layout);
		nameGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		PlatformUI.getWorkbench().getHelpSystem().setHelp(this, ContextIds.RUNTIME_COMPOSITE);

		Label label = new Label(nameGroup, SWT.NONE);
		label.setText(Messages.runtimeName);
		GridData data = new GridData();
		label.setLayoutData(data);

		name = new Text(nameGroup, SWT.BORDER);
		data = new GridData(GridData.FILL_HORIZONTAL);
		name.setLayoutData(data);
		name.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (runtimeWc != null)
					modifiedValuesCache.name = name.getText();
				validate();
			}
		});

		createServerInfoGroup(this);
		createPublishInfoGroup(this);
		init();
		validate();

		Dialog.applyDialogFont(this);

		name.forceFocus();
	}

	protected void init() {
		if (name == null || runtimeWc == null || runtime == null)
			return;

		originalValuesCache.publishDir = runtime.getPublishLocation();
		originalValuesCache.name = runtimeWc.getName();
		originalValuesCache.port = runtime.getPort();
		originalValuesCache.prefix = runtime.getPrefixPath();
		originalValuesCache.canPublish = runtime.publishToDirectory();
		// Clone the cache
		modifiedValuesCache = new ValuesCache(originalValuesCache);

		if (originalValuesCache.name != null) {
			name.setText(originalValuesCache.name);
		} else {
			name.setText("");
		}
		
		if(originalValuesCache.publishDir != null) {
			publishDir.setText(originalValuesCache.publishDir);
		} else {
			publishDir.setText("");
		}
		
		if(originalValuesCache.port != null ) {
			port.setText(originalValuesCache.port);
		} else {
			port.setText("80"); //default port
		}
			
		if(originalValuesCache.prefix != null ) {
			prefix.setText(originalValuesCache.prefix);
		} else {
			prefix.setText("");
		}
		
		publishCheckBox.setSelection(originalValuesCache.canPublish);

		publishDir.setEnabled(originalValuesCache.canPublish);
		browseButton.setEnabled(originalValuesCache.canPublish);
		locationLabel.setEnabled(originalValuesCache.canPublish);
		
	}

	protected void validate() {
		if (runtime == null) {
			wizard.setMessage("", IMessageProvider.ERROR);
			return;
		}

		wizard.setMessage("", IMessageProvider.NONE);

		if (runtimeWc != null) {
			String name = modifiedValuesCache.name;
			if (name == null || name.trim().equals("")) {
				wizard.setMessage(Messages.wizardMissingRuntimeName, IMessageProvider.ERROR);
			} else {
				boolean ok = checkRuntimeName(name);
				if (!ok) {
					wizard.setMessage(Messages.wizardDuplicateName, IMessageProvider.ERROR);
				}
			}
		}
		
	}
	
	private void createServerInfoGroup(Composite parent) {
		Font font = parent.getFont();
		// Server information group
		Group serverInfoGroup = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		serverInfoGroup.setLayout(layout);
		serverInfoGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		serverInfoGroup.setFont(font);
		serverInfoGroup.setText("Server Information");
		
		// port label
		portLabel = new Label(serverInfoGroup, SWT.NONE);
		portLabel.setFont(font);
		portLabel.setText("HTTP Port:");

		// port entry field
		port = new Text(serverInfoGroup, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = 305;
		port.setLayoutData(data);
		port.setFont(font);
		port.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (runtime != null)
					modifiedValuesCache.port = port.getText();
				validate();
			}
		});
		
		// prefix label
		prefixLabel = new Label(serverInfoGroup, SWT.NONE);
		prefixLabel.setFont(font);
		prefixLabel.setText("URL Prefix Path:");

		// prefix entry field
		prefix = new Text(serverInfoGroup, SWT.BORDER);
		data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = 305;
		prefix.setLayoutData(data);
		prefix.setFont(font);
		prefix.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (runtime != null)
					modifiedValuesCache.prefix = prefix.getText();
				validate();
			}
		});
		
		
	}
	
	private final void createPublishInfoGroup(Composite parent) {

		Font font = parent.getFont();
		// publish information group
		Group publishInfoGroup = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		publishInfoGroup.setLayout(layout);
		publishInfoGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		publishInfoGroup.setFont(font);
		publishInfoGroup.setText("Publish Information");

		publishCheckBox = new Button(publishInfoGroup, SWT.CHECK | SWT.RIGHT);
		publishCheckBox.setText("Publish Projects to this Server");
		publishCheckBox.setFont(font);
		
		publishCheckBox.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent se) {
				Button b = (Button) se.getSource();
				boolean selected = b.getSelection();

				publishDir.setEnabled(selected);
				browseButton.setEnabled(selected);
				locationLabel.setEnabled(selected);
				publishDir.setText("");
				if( runtime != null ) 
					modifiedValuesCache.canPublish = selected;
				validate();
			}
		});

		GridData buttonData = new GridData();
		buttonData.horizontalSpan = 3;
		publishCheckBox.setLayoutData(buttonData);
		
		createPublishLocationGroup(publishInfoGroup);
	}
	
	

	private void createPublishLocationGroup(Composite publishInfoGroup) {
		Font font = publishInfoGroup.getFont();
		// location label
		locationLabel = new Label(publishInfoGroup, SWT.NONE);
		locationLabel.setFont(font);
		locationLabel.setText("Directory:");

		// project location entry field
		publishDir = new Text(publishInfoGroup, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = 305;
		publishDir.setLayoutData(data);
		publishDir.setFont(font);
		publishDir.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (runtime != null)
					modifiedValuesCache.publishDir = publishDir.getText();
				validate();
			}
		});

		// browse button
		browseButton = new Button(publishInfoGroup, SWT.PUSH);
		browseButton.setFont(font);
		browseButton.setText(Messages.browse);
		browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				DirectoryDialog dialog = new DirectoryDialog(HttpRuntimeComposite.this.getShell());
				dialog.setMessage(Messages.selectInstallDir);
				dialog.setFilterPath(publishDir.getText());
				String selectedDirectory = dialog.open();
				if (selectedDirectory != null)
					publishDir.setText(selectedDirectory);
			}
		});
	}

	private boolean checkRuntimeName(String name) {
		name = name.trim();
		if (name.equals(originalValuesCache.name)) {
			return true;
		}
		IRuntime[] allRuntimes = ServerCore.getRuntimes();
		
		if (allRuntimes != null) {
			int size = allRuntimes.length;
			for (int i = 0; i < size; i++) {
				IRuntime runtime2 = allRuntimes[i];
				if (name.equals(runtime2.getName()))
					return false;
			}
		}
		return true;
	}

	/**
	 * Saves the IServerWorkingCopy.
	 */
	public boolean performOk() {
		try {
			// Save any modification logged into the modified value cache object.
			if (runtime != null) {
				runtime.setPublishLocation(modifiedValuesCache.publishDir);
				runtime.setPort(modifiedValuesCache.port);
				runtime.setPrefixPath(modifiedValuesCache.prefix);
				runtime.setPublishToDirectory(modifiedValuesCache.canPublish);
			}
			runtimeWc.setName(modifiedValuesCache.name);
			runtimeWc.setLocation(new Path(modifiedValuesCache.publishDir));
			runtimeWc.save(true, null);
		} catch (CoreException e) {
			Trace.trace(Trace.SEVERE, "Error while saving the new server settings", e);
			return false;
		}
		return true;
	}

	// A class used as a local original IServerWorkingCopy values cache.
	private class ValuesCache {
		String name;
		String publishDir;
		String port;
		String prefix;
		boolean canPublish;
				
		public ValuesCache() {
		}

		public ValuesCache(ValuesCache cache) {
			this.name = cache.name;
			this.publishDir = cache.publishDir;
			this.port = cache.port;
			this.prefix = cache.prefix;
			this.canPublish = cache.canPublish;
		}
	}

}
