/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.ui.wizard;

import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;

import org.eclipse.atf.templates.model.wizard.PageModel;
import org.eclipse.atf.templates.model.wizard.WizardModel;
import org.eclipse.atf.templates.model.wizard.WizardModelRegistry;
import org.eclipse.atf.templates.operation.IOperation;
import org.eclipse.atf.templates.variable.IVariableRegistry;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jdt.internal.core.JavaProject;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.WorkspaceModifyOperation;



/**
 * This implements a wizard that results in the creation of 1 or more files in 
 * an existing project. The wizard is driven by a set of variables that
 * translate into USER input. The USER assign values are then used to interpret
 * File templates. 
 */
abstract public class ModelDrivenWizard extends Wizard implements INewWizard{

	protected WizardModel wizardModel = null;
	
	protected IStructuredSelection mSelection = null;
	
	public boolean performFinish() {
			
		WorkspaceModifyOperation finishOperation = new WorkspaceModifyOperation(){

			protected void execute(IProgressMonitor monitor) throws CoreException, InvocationTargetException, InterruptedException {
				
				monitor.beginTask( "Finish Operations", wizardModel.operationCount() );
				for (Iterator iter = wizardModel.operations(); iter.hasNext();) {
					IOperation operation = (IOperation) iter.next();
					
					monitor.worked( 1 );
					
					//if an operation fails, stop
					IStatus status = operation.execute( wizardModel );
					if( !status.isOK() ){
						monitor.done();
						throw new CoreException( status );
					}
									
				}
				
				monitor.done();
			}			
			
		};
		
		try {
			getContainer().run( false, true, finishOperation );
			return true;
		} catch (InvocationTargetException e) {
			e.printStackTrace();
			IStatus status = null;
			if( e.getTargetException() instanceof CoreException ){
				status = ((CoreException)e.getTargetException()).getStatus();
			}
			ErrorDialog.openError(getShell(), "Wizard Error!", "A task reported an error", status );
			return false;
		} catch (InterruptedException e) {
			e.printStackTrace();
			ErrorDialog.openError(getShell(), "Wizard Error!", "The finish task was interrupted.", null );
			return false;
		}
		finally{
			wizardModel.reset();
		}
	}
	
	public boolean performCancel() {
		wizardModel.reset();
		return super.performCancel();
	}
	

	public void init(IWorkbench workbench, IStructuredSelection selection) {
		WizardModelRegistry registry = WizardModelRegistry.getInstance();
		
		wizardModel = registry.getWizardModel( getWizardId() );
		
		setWindowTitle( wizardModel.getTitle() );
		
		mSelection = selection;
	}
	
	public void addPages() {
		for (Iterator iter = wizardModel.pages(); iter.hasNext();) {
			PageModel pageModel = (PageModel) iter.next();
			
			addPage( createWizardPage(pageModel, wizardModel));
			
		}
	}
	
	protected WizardPage createWizardPage( PageModel pageModel, IVariableRegistry varReg ){
		
		return new ModelDrivenWizardPage(pageModel, wizardModel);
	}

	public IProject getInitiallySelectedProject() {
		
		if(mSelection != null && mSelection.getFirstElement() instanceof IResource) {
			return ((IResource)mSelection.getFirstElement()).getProject();
		} else if(mSelection != null && mSelection.getFirstElement() instanceof JavaProject) {
			return ((JavaProject)mSelection.getFirstElement()).getProject();
		}
		return null;
	}
	
	abstract public String getWizardId();
	
	
}
