/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.templates.ui.wizard;

import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.atf.templates.model.IValidator;
import org.eclipse.atf.templates.model.PagePart;
import org.eclipse.atf.templates.model.VariableRef;
import org.eclipse.atf.templates.model.VariableRefGroup;
import org.eclipse.atf.templates.model.wizard.PageModel;
import org.eclipse.atf.templates.ui.variablerenderer.IVariableRenderer;
import org.eclipse.atf.templates.ui.variablerenderer.IVariableRendererRegistry;
import org.eclipse.atf.templates.ui.variablerenderer.factory.IVariableRendererFactory;
import org.eclipse.atf.templates.ui.variablerenderer.factory.VariableRendererFactoryRegistry;
import org.eclipse.atf.templates.variable.IVariable;
import org.eclipse.atf.templates.variable.IVariableRegistry;
import org.eclipse.atf.templates.variable.VariableListener;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;


public class ModelDrivenWizardPage extends WizardPage implements IVariableRendererRegistry, VariableListener {

	protected PageModel pageModel;
	protected IVariableRegistry varReg;
	
	protected HashMap varRendererReg = new HashMap();
	
	protected ModelDrivenWizardPage(PageModel pageModel, IVariableRegistry varReg ) {
		super(pageModel.getName());
		
		this.pageModel = pageModel;
		this.varReg = varReg;
		
		setTitle( pageModel.getTitle() );
		setDescription( pageModel.getDescription() );
	}

	public void createControl(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setFont(parent.getFont());
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		//composite.setLayout( new RowLayout(SWT.VERTICAL) );
		
		Composite composite2 = null;
		
		for (Iterator iter = pageModel.parts(); iter.hasNext();) {
			PagePart part = (PagePart) iter.next();
			
			//stupid check, needs to fix design
			if( part instanceof VariableRef ) {
				if(composite2 == null) {
					composite2 = new Composite(composite, SWT.NONE);
					composite2.setFont(composite.getFont());
					GridLayout layout = new GridLayout();
					layout.numColumns = 2;
					composite2.setLayout(layout);
					composite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
				}
				createControl( (VariableRef)part, composite2 );
			}
			else if ( part instanceof VariableRefGroup ){
				createControl( (VariableRefGroup)part, composite );
			}
		}
		
		setControl( composite );
		
	}
	
	protected void createControl( VariableRef ref, Composite parent ){
				
		IVariable variable = varReg.lookupVariable(ref.getName());
		
		IVariableRendererFactory rendererFactory = VariableRendererFactoryRegistry.getInstance().getVariableRendererFactory(variable.getType());
		if(rendererFactory == null)
			return;
		IVariableRenderer renderer = rendererFactory.createRenderer();
		renderer.createVariableParts(parent, variable, this);
		
		//add as listener
		variable.addListener( this );

		//register the renderer
		registerVariableRenderer( variable, renderer );
		
	}
	
	protected void createControl( VariableRefGroup refGroup, Composite parent ){

		Group group = new Group( parent, SWT.SHADOW_ETCHED_IN );
		group.setText( refGroup.getDisplayName() );
		
		//group.setLayout( new RowLayout(SWT.VERTICAL));
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		group.setLayout(layout);
		group.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		for (Iterator iter = refGroup.refs(); iter.hasNext();) {
			createControl( (VariableRef)iter.next(), group );
			
		}
		
	}

	/*
	 * IVariableRendererRegistry interface
	 */
	public void registerVariableRenderer(IVariable variable, IVariableRenderer renderer) {
		varRendererReg.put( variable, renderer );
		
	}

	public IVariableRenderer lookupVariable(IVariable variable) {
		return (IVariableRenderer)varRendererReg.get( variable );
	}

	public Iterator renderers() {
		return varRendererReg.values().iterator();
	}

	public Iterator variables(){
		return varRendererReg.keySet().iterator();
	}

	public void variableChanged(IVariable source) {
		/*
		 * check the validity of the page if visible. Everytime a variable
		 * changes.
		 */
		if( getControl().isVisible() )
			validatePage();		
	}
	
	public void setVisible(boolean visible) {
	
		super.setVisible(visible);
		
		//reset the validation on the page
		if( visible )
			validatePage();
	}
	
		
	public void dispose() {
		super.dispose();
		
		//clean listeners
		for (Iterator iter = pageModel.pageVariables( varReg ); iter.hasNext();) {
			IVariable variable = (IVariable) iter.next();
			variable.removeListener( this );			
		}
		
	}

	/*
	 * This method gets called when the Page becomes visible and while visible, 
	 * every time a Variable on the page changes value. The validation is
	 * delegated to an implementation of an IValidator. The IValidator
	 * implementation is declared with markup in the plugin.xml.
	 */
	protected void validatePage(){		
		
		//First check that all required Variables contain valid values
		for (Iterator iter = pageModel.pageVariables( varReg ); iter.hasNext();) {
			IVariable variable = (IVariable) iter.next();
			
			//right now it fails when on the first variable that is required and
			//does not have a valid value
			if( variable.isRequired() && !variable.isValidValue() ){
				setPageComplete( false );
				setErrorMessage( "Required field '"+variable.getDisplayName()+"' does not have a valid value!" );
				return;
			}
		}		
		
		//If a custom validator is set, run it next
		IValidator validator = pageModel.getValidator();
		if( validator != null && !validator.validate(varReg) ){
			setPageComplete( false );
			setErrorMessage( validator.getErrorMessage() );
			return;
		}
		
		//if we get to this point it means that all validation passed
		//reset
		setPageComplete( true );
		setErrorMessage(null);
		
	}
}
