/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.operation;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.atf.templates.TemplatesPlugin;
import org.eclipse.atf.templates.util.JETExecutionHelper;
import org.eclipse.atf.templates.util.VariableSubtitutionHelper;
import org.eclipse.atf.templates.variable.IVariableRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;


/*
 * This operation just wraps Operations and executes them based on the result
 * of a conditional statement. The conditional statement is a combination of
 * JET expressions and ${} variables. Something to be aware of is that when
 * declaring the operation ''' characters need to be added around the variables
 * to comparing using the String methods. When evaluating the conditional, the
 * variables are substituded and ''' characters are changed to '"'.
 * 
 * If an operation return false, the remaining operations are skipped.
 * 
 * For example:
 * 
 * 'some string'.equals('${var}')
 */
public class ConditionalOperation implements IOperation {

	protected String condition; /*SIMPLE JET/VAR STATEMENT*/
	protected ArrayList operations = new ArrayList();
	
	public IStatus execute( IVariableRegistry varReg ) {
		
		VariableSubtitutionHelper substitutionHelper = VariableSubtitutionHelper.getInstance();
		JETExecutionHelper jetHelper = JETExecutionHelper.getInstance();
		
		//evaluate the condition statement (should result on "true" or "false")
		String conditionResult;
		try {
			conditionResult = jetHelper.generate( "<%=" + substitutionHelper.substitute(getCondition(), varReg).replace('\'', '\"') + "%>" );
		} catch (Exception e) {
			e.printStackTrace();
			IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"Error evaluating the operation's conditional <"+getCondition()+">.", e);
			
			return status;
		}
		
		//only execute if the String returned represents true
		if (Boolean.valueOf(conditionResult.trim()).booleanValue() ){
			
			for (Iterator iter = operations.iterator(); iter.hasNext();) {
				IOperation operation = (IOperation) iter.next();
				
				IStatus status = operation.execute(varReg);
				if( !status.isOK() )
					return status;  //exit as soon as one of the operations fail
				
			}
			
		}
		
		return Status.OK_STATUS; //success
	}
	
	public void addOperation( IOperation operation ){
		operations.add( operation );
	}

	public String getCondition() {
		return condition;
	}

	public void setCondition(String condition) {
		this.condition = condition;
	}
}
