/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.operation;

import java.io.ByteArrayInputStream;

import org.eclipse.atf.adapter.IWebResourceLocator;
import org.eclipse.atf.templates.FileTemplate;
import org.eclipse.atf.templates.TemplatesPlugin;
import org.eclipse.atf.templates.util.JETExecutionHelper;
import org.eclipse.atf.templates.util.VariableSubtitutionHelper;
import org.eclipse.atf.templates.variable.IVariableRegistry;
import org.eclipse.atf.templates.variable.ProjectVariable;
import org.eclipse.atf.templates.variable.StringVariable;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;


public class CreateFileOperation implements IOperation {

	protected String filename = "";
	protected String location = "";
	protected String project = "";
	
	protected FileTemplate fileTemplate;
	
	public IStatus execute( IVariableRegistry varReg ) {
		
		//check the values of the members
		if( "".equals(filename) || "".equals(project) ){
			System.err.println( "ERROR: CreateFileOperation - filename and project must be set!" );
			IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"ERROR: CreateFileOperation - filename and project must be set!", null);
			
			return status;
		}
		
		VariableSubtitutionHelper substitutionHelper = VariableSubtitutionHelper.getInstance();
		JETExecutionHelper jetHelper = JETExecutionHelper.getInstance();
		
		String filename;
		try {
			filename = substitutionHelper.substitute( getFilename(), varReg );
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"ERROR: CreateFileOperation - could not value of filename!", e);
			
			return status;
		}
		
		String location;
		try {
			location = substitutionHelper.substitute( getLocation(), varReg );
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"ERROR: CreateFileOperation - could not resolve the value of location!", e);
			
			return status;
		}
		
		//check the project variable
		String projectName;
		try {
			projectName = substitutionHelper.substitute( getProject(), varReg );
		} catch (Exception e) {
			IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"ERROR: CreateFileOperation - could not resolve the value of project!", e);
			
			return status;
		}
		
		IProject destProject = ProjectVariable.getProjectNamed( projectName );
		
		IPath filepath = new Path(filename);
		
		IPath locationPath = new Path( location );
		
		//Prepend the web content folder to the location
		IContainer webContentFolder = null;
		IWebResourceLocator adapter = (IWebResourceLocator)destProject.getAdapter( IWebResourceLocator.class );
		
		//if there is no adapter then we installe it right on the project root
		if( adapter == null ){
			webContentFolder = destProject;
		}
		else{
			webContentFolder = adapter.getWebResourceContainer();
		}				
		
		locationPath = webContentFolder.getProjectRelativePath().append(locationPath);
		createFolders( locationPath, destProject );
		
		//create the implicit _thisLocation variable or change its value
		StringVariable _thisLocation = (StringVariable)varReg.lookupVariable( "_thisLocation" );
		if( _thisLocation == null ){
			_thisLocation = new StringVariable();
			_thisLocation.setName("_thisLocation" );
			varReg.registerVariable( _thisLocation );
		}
		//make the location a path relative to the WebContent directory
		int segments = webContentFolder.getProjectRelativePath().matchingFirstSegments(locationPath);
		_thisLocation.setValue( locationPath.removeFirstSegments(segments).toString() );
		
		//set the "from" and "to" attributes of all Variables of type RelativePath
		/*
		for (Iterator iter = varReg.variables(); iter.hasNext();) {
			IVariable variable = (IVariable) iter.next();
			if( variable.getType() == RelativePathVariable.TYPE ){
				((RelativePathVariable)variable).setFrom( new Path(location).toString() );
				((RelativePathVariable)variable).setFromPrefix(webContentFolder.getProjectRelativePath());
				((RelativePathVariable)variable).setToPrefix(webContentFolder.getProjectRelativePath());
				
			}
			
		}
		*/
		
		//create file
		IPath pathToFile = locationPath.append( filepath );
					
		String content = "";
		if( getFileTemplate() != null ){
			try {
				content = jetHelper.generate( substitutionHelper.substitute(getFileTemplate().getContent(), varReg) );
			} catch (Exception e) {
				IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
						"ERROR: CreateFileOperation - could not process template <"+getFileTemplate().getPath().toString()+">!", e);
				
				return status;
			}
		}
		
		IFile file = destProject.getFile( pathToFile );
		try {
			file.create( new ByteArrayInputStream(content.getBytes()), true, new NullProgressMonitor() );
		} catch (CoreException ce) {
			IStatus status = new Status(IStatus.ERROR, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"ERROR: CreateFileOperation - could not create new file <"+pathToFile.toString()+">!", ce );
			
			return status;
		}
		
		try {
			IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
			IDE.openEditor(page, file, true);
		}
		catch (PartInitException e) {
			IStatus status = new Status(IStatus.WARNING, TemplatesPlugin.getDefault().getBundle().getSymbolicName(), IStatus.ERROR, 
					"WARNING: CreateFileOperation - could not open new file!", e );
			
			return status;
		}
		
		
		return Status.OK_STATUS;
	}

	public String getFilename() {
		return filename;
	}

	public void setFilename(String filename) {
		// protect against NPE
		if ( filename == null )
			this.filename = "";
		else
			this.filename = filename;
	}
	
	public String getLocation() {
		return location;
	}

	public void setLocation(String location) {
//		 protect against NPE
		if( location == null )
			this.location = "";
		else
			this.location = location;
	}

	public FileTemplate getFileTemplate() {
		return fileTemplate;
	}

	public void setFileTemplate(FileTemplate fileTemplate) {
		this.fileTemplate = fileTemplate;
	}

	public String getProject() {
		return project;
	}

	public void setProject(String project) {
//		 protect against NPE
		if( project == null )
			this.project = "";
		else
			this.project = project;
	}
	
	protected void createFolders( IPath locationPath, IProject destProject ){
		//remove the first segment if it is '.'
		if( locationPath.segmentCount() > 0 ){
			if( ".".equals(locationPath.segment(0).toString()) )
				locationPath = locationPath.removeFirstSegments(1);
		}
		
		for( int i=0; i<locationPath.segmentCount(); i++ ){
			
			//each step remove less from the tail of the path
			IPath subpath = locationPath.removeLastSegments( locationPath.segmentCount()-(i+1) );
			
			try{				
				IFolder subFolder = destProject.getFolder( subpath );
				
				if( !subFolder.exists() )
					subFolder.create( false, true, new NullProgressMonitor() );
			
			}
			catch( CoreException ce ){
				ce.printStackTrace();
				System.err.println( "ERROR with location path segment <"+subpath+">!" );
			}
		}	
		
	}
}
