/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.util;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.codegen.CodeGenPlugin;
import org.eclipse.emf.codegen.jet.JETCompiler;
import org.eclipse.emf.codegen.jet.JETException;
import org.eclipse.emf.common.util.BasicMonitor;
import org.eclipse.emf.common.util.Monitor;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.launching.JavaRuntime;


/**
 * A convenience class for compiling and invoking a template dynamically.
 */
public class JETEmitter extends org.eclipse.emf.codegen.jet.JETEmitter
{

  protected InputStream is = null;
  
  public JETEmitter( String templateURI, InputStream is ){
	  super(templateURI);
	  
	  this.is = is;
	  
  }

  public void initialize(Monitor monitor) throws JETException{
	  IProgressMonitor progressMonitor = BasicMonitor.toIProgressMonitor(monitor);
	  initialize(progressMonitor);
  }
  /**
   * @GINO
   * Changes were made to allow to grab the template from an InputStream rather that always from
   * a file.
   * 
   * Compiles the template to {@link #setMethod set} the method will be invoked to generate template results.
   * @param progressMonitor the progress monitor for tracking progress.
   * 
   */
  public void initialize(IProgressMonitor progressMonitor) throws JETException
  {
    progressMonitor.beginTask("", 10);
    progressMonitor.subTask(CodeGenPlugin.getPlugin().getString("_UI_GeneratingJETEmitterFor_message", new Object [] { templateURI }));

    try
    {
      // This ensures that the JRE variables are initialized.
      //
      try
      {
        JavaRuntime.getDefaultVMInstall();
      }
      catch (Throwable throwable)
      {
        // This is kind of nasty to come here.
        //
        URL jreURL = Platform.getBundle("org.eclipse.emf.codegen").getEntry("plugin.xml");
        IPath jrePath = new Path(Platform.asLocalURL(jreURL).getFile());
        jrePath = jrePath.removeLastSegments(1).append(new Path("../../jre/lib/rt.jar"));
        if (!jrePath.equals(JavaCore.getClasspathVariable(JavaRuntime.JRELIB_VARIABLE)))
        {
          JavaCore.setClasspathVariable(JavaRuntime.JRELIB_VARIABLE, jrePath, null);
        }
      }

      
      //Use the InputStream as the input to the JetCompiler
      final JETCompiler jetCompiler = new JETCompiler( templateURI, is, encoding );

          
      progressMonitor.subTask
        (CodeGenPlugin.getPlugin().getString("_UI_JETParsing_message", new Object [] { jetCompiler.getResolvedTemplateURI() }));
      jetCompiler.parse();
      progressMonitor.worked(1);

      ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
      jetCompiler.generate(outputStream);
      final InputStream contents = new ByteArrayInputStream(outputStream.toByteArray());

      final IWorkspace workspace = ResourcesPlugin.getWorkspace();
      IJavaModel javaModel= JavaCore.create(ResourcesPlugin.getWorkspace().getRoot());
      if (!javaModel.isOpen())
      {
        javaModel.open(new SubProgressMonitor(progressMonitor, 1));
      }
      else
      {
        progressMonitor.worked(1);
      }

      final IProject project = workspace.getRoot().getProject(getProjectName());
      progressMonitor.subTask
        (CodeGenPlugin.getPlugin().getString("_UI_JETPreparingProject_message", new Object [] { project.getName() }));

      IJavaProject javaProject;
      if (!project.exists())
      {
        progressMonitor.subTask("JET creating project " + project.getName());
        project.create(new SubProgressMonitor(progressMonitor, 1));
        progressMonitor.subTask
          (CodeGenPlugin.getPlugin().getString("_UI_JETCreatingProject_message", new Object [] { project.getName() }));
        IProjectDescription description = workspace.newProjectDescription(project.getName());
        description.setNatureIds(new String [] { JavaCore.NATURE_ID });
        description.setLocation(null);
        project.open(new SubProgressMonitor(progressMonitor, 1));
        project.setDescription(description, new SubProgressMonitor(progressMonitor, 1));
      }
      else
      {
        project.open(new SubProgressMonitor(progressMonitor, 5));
        IProjectDescription description = project.getDescription();
        description.setNatureIds(new String [] { JavaCore.NATURE_ID });
        project.setDescription(description, new SubProgressMonitor(progressMonitor, 1));
      }

      javaProject = JavaCore.create(project);

      progressMonitor.subTask
        (CodeGenPlugin.getPlugin().getString("_UI_JETInitializingProject_message", new Object [] { project.getName() }));
      IClasspathEntry classpathEntry = 
        JavaCore.newSourceEntry(new Path("/" + project.getName() + "/src"));

      IClasspathEntry jreClasspathEntry = 
        JavaCore.newVariableEntry
          (new Path(JavaRuntime.JRELIB_VARIABLE), 
           new Path(JavaRuntime.JRESRC_VARIABLE), 
           new Path(JavaRuntime.JRESRCROOT_VARIABLE));

      List classpath = new ArrayList();
      classpath.add(classpathEntry);
      classpath.add(jreClasspathEntry);
      classpath.addAll(classpathEntries);

      IFolder sourceFolder = project.getFolder(new Path("src"));
      if (!sourceFolder.exists())
      {
        sourceFolder.create(false, true, new SubProgressMonitor(progressMonitor, 1));
      }
      IFolder runtimeFolder = project.getFolder(new Path("runtime"));
      if (!runtimeFolder.exists())
      {
        runtimeFolder.create(false, true, new SubProgressMonitor(progressMonitor, 1));
      }

      IClasspathEntry [] classpathEntryArray = (IClasspathEntry[])classpath.toArray(new IClasspathEntry [classpath.size()]);

      javaProject.setRawClasspath(classpathEntryArray, new SubProgressMonitor(progressMonitor, 1));

      javaProject.setOutputLocation(new Path("/" + project.getName() + "/runtime"), new SubProgressMonitor(progressMonitor, 1));

      javaProject.close();

      progressMonitor.subTask
        (CodeGenPlugin.getPlugin().getString("_UI_JETOpeningJavaProject_message", new Object [] { project.getName() }));
      javaProject.open(new SubProgressMonitor(progressMonitor, 1));

      IPackageFragmentRoot [] packageFragmentRoots = javaProject.getPackageFragmentRoots();
      IPackageFragmentRoot sourcePackageFragmentRoot = null;
      for (int j = 0; j < packageFragmentRoots.length; ++j)
      {
        IPackageFragmentRoot packageFragmentRoot = packageFragmentRoots[j];
        if (packageFragmentRoot.getKind() == IPackageFragmentRoot.K_SOURCE)
        {
          sourcePackageFragmentRoot = packageFragmentRoot;
          break;
        }
      }

      String packageName = jetCompiler.getSkeleton().getPackageName();
      StringTokenizer stringTokenizer = new StringTokenizer(packageName, ".");
      IProgressMonitor subProgressMonitor = new SubProgressMonitor(progressMonitor, 1);
      subProgressMonitor.beginTask("", stringTokenizer.countTokens() + 4);
      subProgressMonitor.subTask(CodeGenPlugin.getPlugin().getString("_UI_CreateTargetFile_message"));
      IContainer sourceContainer = (IContainer)sourcePackageFragmentRoot.getCorrespondingResource();
      while (stringTokenizer.hasMoreElements())
      {
        String folderName = stringTokenizer.nextToken();
        sourceContainer = sourceContainer.getFolder(new Path(folderName));
        if (!sourceContainer.exists())
        {
          ((IFolder)sourceContainer).create(false, true, new SubProgressMonitor(subProgressMonitor, 1));
        }
      }
      IFile targetFile = sourceContainer.getFile(new Path(jetCompiler.getSkeleton().getClassName() + ".java"));
      if (!targetFile.exists())
      {
        subProgressMonitor.subTask
          (CodeGenPlugin.getPlugin().getString("_UI_JETCreating_message", new Object [] { targetFile.getFullPath() }));
        targetFile.create(contents, true, new SubProgressMonitor(subProgressMonitor, 1));
      }
      else
      {
        subProgressMonitor.subTask
          (CodeGenPlugin.getPlugin().getString("_UI_JETUpdating_message", new Object [] { targetFile.getFullPath() }));
        targetFile.setContents(contents, true, true, new SubProgressMonitor(subProgressMonitor, 1));
      }

      subProgressMonitor.subTask
        (CodeGenPlugin.getPlugin().getString("_UI_JETBuilding_message", new Object [] { project.getName() }));
      project.build(IncrementalProjectBuilder.INCREMENTAL_BUILD, new SubProgressMonitor(subProgressMonitor, 1));

      IMarker [] markers = targetFile.findMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE);
      boolean errors = false;
      for (int i = 0; i < markers.length; ++i)
      {
        IMarker marker = markers[i];
        if (marker.getAttribute(IMarker.SEVERITY, IMarker.SEVERITY_INFO) == IMarker.SEVERITY_ERROR)
        {
          errors = true;
          subProgressMonitor.subTask
            (marker.getAttribute(IMarker.MESSAGE) + " : " + 
               (CodeGenPlugin.getPlugin().getString
                 ("jet.mark.file.line", 
                  new Object []
                  {
                    targetFile.getLocation(), 
                    marker.getAttribute(IMarker.LINE_NUMBER)
                  })));
        }
      }

      if (!errors)
      {
        subProgressMonitor.subTask
          (CodeGenPlugin.getPlugin().getString
             ("_UI_JETLoadingClass_message", new Object [] { jetCompiler.getSkeleton().getClassName() + ".class" }));

        // Construct a proper URL for relative lookup.
        //
        URL url = new File(project.getLocation() + "/" + javaProject.getOutputLocation().removeFirstSegments(1) + "/").toURL();
        URLClassLoader theClassLoader = new URLClassLoader(new URL [] { url }, classLoader);
        Class theClass = 
          theClassLoader.loadClass
            ((packageName.length() == 0 ? "" : packageName + ".") + jetCompiler.getSkeleton().getClassName());
        String methodName = jetCompiler.getSkeleton().getMethodName();
        Method [] methods = theClass.getDeclaredMethods();
        for (int i = 0; i < methods.length; ++i)
        {
          if (methods[i].getName().equals(methodName))
          {
            setMethod(methods[i]);
            break;
          }
        }
      }

      subProgressMonitor.done();
    }
    catch (CoreException exception)
    {
      throw new JETException(exception);
    }
    catch (Exception exception)
    {
      throw new JETException(exception);
    }
    finally
    {
      progressMonitor.done();
    }
  }
}