/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.util;

import java.io.ByteArrayInputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.eclipse.atf.templates.Template;
import org.eclipse.atf.templates.model.dialog.DialogModel;
import org.eclipse.atf.templates.model.dialog.SnippetModel;
import org.eclipse.atf.templates.model.dialog.SnippetModelRegistry;
import org.eclipse.atf.templates.ui.dialog.ModelDrivenDialog;
import org.eclipse.atf.templates.variable.IVariable;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.templates.TemplateBuffer;
import org.eclipse.jface.text.templates.TemplateContext;
import org.eclipse.jface.text.templates.TemplateContextType;
import org.eclipse.jface.text.templates.TemplateException;
import org.eclipse.jface.text.templates.TemplateTranslator;
import org.eclipse.jface.text.templates.TemplateVariable;
import org.eclipse.jface.text.templates.TemplateVariableResolver;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.texteditor.ITextEditor;
import org.eclipse.wst.common.snippets.core.ISnippetItem;
import org.eclipse.wst.common.snippets.internal.Logger;
import org.eclipse.wst.common.snippets.internal.SnippetsPlugin;
import org.eclipse.wst.common.snippets.internal.palette.SnippetPaletteItem;
import org.eclipse.wst.common.snippets.ui.DefaultSnippetInsertion;


/*
 * Implementation that supports a content string that used a mixture of JET template
 * and ${variable} syntax. 
 */
public class TemplateInsertion2 extends DefaultSnippetInsertion {

	protected ISnippetItem item = null;
	
	protected Transfer[] createTransfers() {
		return new Transfer[]{TextTransfer.getInstance()};
	}
	
	/*
	 * This is called during a DnD in order to set the data on the event.
	 * 
	 * @see org.eclipse.wst.common.snippets.internal.provisional.ISnippetsInsertion#dragSetData(org.eclipse.swt.dnd.DragSourceEvent, org.eclipse.wst.common.snippets.internal.provisional.ISnippetItem)
	 */
	public void dragSetData(DragSourceEvent event, ISnippetItem item) {
		TextTransfer transfer = TextTransfer.getInstance();
		
		if( transfer.isSupportedType( event.dataType ) ){
			IWorkbenchWindow window = SnippetsPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow();
			Shell shell = null;
			if (window != null) {
				shell = window.getShell();
			}
			
			event.data = computeInsertString( item, shell );
			
		}
				
	}

	/*
	 * This is called when dbl-click on a snippet in order to insert it.
	 * 
	 * @see org.eclipse.wst.common.snippets.internal.provisional.insertions.AbstractInsertion#getInsertString(org.eclipse.swt.widgets.Shell)
	 */
	public String getInsertString(Shell host) {
		
		return computeInsertString( getItem(), host );
	}
	
	
	protected String computeInsertString( ISnippetItem item, Shell host ){
		
		try {
			//retrieve the SnippetModel for the SnippetItem
			final SnippetModel snippetModel = SnippetModelRegistry.getInstance().getSnippetModel( ((SnippetPaletteItem) item).getId() );
			DialogModel dialogModel = snippetModel.getDialogModel();
			
			ModelDrivenDialog dialog = new ModelDrivenDialog( host, dialogModel, snippetModel );
			
			//open the dialog to gather user input
			if( dialog.open() != Window.OK )
				return "";
			
			Template template = snippetModel.getTemplate();
			
			String pattern = template.getContent();
			
			System.out.println( "READ IN: " + pattern );
			
			TemplateTranslator translator = new TemplateTranslator();
			
			TemplateBuffer templateBuffer = translator.translate( pattern );
			
			TemplateVariable[] variables = templateBuffer.getVariables();
			
			TemplateContextType templateContextType = new TemplateContextType("internal");
			
			for (int i = 0; i < variables.length; i++) {
				System.out.println( "VARIABLE: name<"+variables[i].getName()+"> type<"+variables[i].getType()+">");
				
				final TemplateVariable variable = variables[i];
				
				templateContextType.addResolver( new TemplateVariableResolver(){
					
					public String getDescription() {
						return variable.getType();
					}
		
					public String getType() {
						return variable.getType();
					}
		
					protected String resolve(TemplateContext context) {
						
						IVariable var = snippetModel.lookupVariable( getType() );
						if( var != null )
							return var.resolve();
						
						return "<ERROR!>";
					}
					
				});
			}
			
			class MyTemplateContext extends TemplateContext {
				
				public MyTemplateContext(){
					super(null);
				}
				
				public TemplateBuffer evaluate(org.eclipse.jface.text.templates.Template template) throws BadLocationException, TemplateException {
					return null;
				}
	
				public boolean canEvaluate(org.eclipse.jface.text.templates.Template template) {
					return false;
				}
				
			};
			templateContextType.resolve( templateBuffer, new MyTemplateContext());
			
			System.out.println( "RESOLVE: " + templateBuffer.getString() );
			
			String classDirective = "<%@ jet package=\"translated\" class=\"Foo\" %>\n";
			JETEmitter emitter = new JETEmitter( "foo", new ByteArrayInputStream( (classDirective + templateBuffer.getString()).getBytes()) );
			//JETEmitter emitter = new JETEmitter( "file:C:\\dev\\atf\\plugins\\org.eclipse.atf.templates\\template\\test.jet" );
			
		
			String genCode = emitter.generate( new NullProgressMonitor(), new Object[]{new Object()} );
			System.out.println( "GENCODE: " + genCode );
			
			
			return genCode;
			
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			return "<ERROR in template!>";
		}
		
	}

	
	/*
	 * Copy from org.eclipse.wst.common.snippets.internal.provisional.insertions.VariableInsertion
	 * 
	 * This was done because it supports Multipage Editors
	 * 
	 * @see org.eclipse.wst.common.snippets.internal.provisional.ISnippetsInsertion#insert(org.eclipse.ui.IEditorPart)
	 */
	public void insert(IEditorPart editorPart) {
		IEditorPart anEditorPart = editorPart;
		if (anEditorPart == null)
			return;
		if (anEditorPart instanceof ITextEditor) {
			super.insert(editorPart);
		}
		else {
			// MultiPageEditorPart has no accessors for the source EditorPart
			ITextEditor textEditor = null;
			if (anEditorPart instanceof ITextEditor) {
				textEditor = (ITextEditor) anEditorPart;
			}
			else {
				textEditor = (ITextEditor) anEditorPart.getAdapter(ITextEditor.class);
			}
			if (textEditor != null) {
				if (textEditor.isEditable()) {
					IDocument document = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
					ISelection selection = textEditor.getSelectionProvider().getSelection();
					if (document != null && selection instanceof ITextSelection) {
						ITextSelection textSel = (ITextSelection) selection;
						try {
							doInsert(anEditorPart, textEditor, document, textSel);
						}
						catch (Exception t) {
							Logger.logException("Could not insert " + ((SnippetPaletteItem) getItem()).getId(), t); //$NON-NLS-1$
							textEditor.getSite().getShell().getDisplay().beep();
						}
					}
				}
			}
			else {
				// any errors here probably aren't really exceptional
				Method getTextEditor = null;
				try {
					getTextEditor = anEditorPart.getClass().getMethod("getTextEditor", new Class[0]); //$NON-NLS-1$
				}
				catch (NoSuchMethodException e) {
					// nothing, not unusual
				}
				Object editor = null;
				if (getTextEditor != null) {
					try {
						editor = getTextEditor.invoke(anEditorPart, new Object[0]);
					}
					catch (IllegalAccessException e) {
						// nothing, not unusual for a non-visible method
					}
					catch (InvocationTargetException e) {
						// nothing, not unusual for a protected implementation
					}
					if (editor instanceof IEditorPart && editor != anEditorPart)
						insert((IEditorPart) editor);
				}
			}
		}
	}
	
	public void setItem(ISnippetItem item) {
		super.setItem( item );
		this.item = item;
		
	}
	
	protected ISnippetItem getItem(){
		return item;
	}
	
}
