/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.variable;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.atf.templates.variable.util.IProjectFilter;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;


public class ProjectVariable extends VariableBase {

	public static final String TYPE = "Project";
	
	protected IProject project = null;
	
	protected boolean defaultSelected = false;
	
	protected ArrayList filters = new ArrayList();
	
	public String getType() {
		return TYPE;
	}

	public String resolve() {
		if( project != null )
			return project.getName();
		else
			return "<NO PROJECT>";
	}

	public void reset() {
		project = null;
	}
	
	/*
	 * If the project value is null or does not exist, the value is considered
	 * invalid. The Variable can still be used but in the case that it is a 
	 * required Variable, the containing UI should display an error.
	 */
	public boolean isValidValue(){
		IProject project = getProjectValue();
		if(project == null ) {
			return false;
		}
		else if( !project.exists() )
			return false;
		else
			return true;
	}

	public void setProjectValue( IProject project ){
		//check that there was a change
		if( project != null && !project.equals(this.project) ){
			this.project = project;
			notifyChange();
		}
		//check whether it is being set back to null
		else if ( project == null && this.project != null ){
			this.project = null;
			notifyChange();
		}
	}
	
	public IProject getProjectValue(){
		return project;
	}
	
	public void setDefaultSelected( String defaultSelected ){
		this.defaultSelected = Boolean.valueOf( defaultSelected ).booleanValue();
	}
	
	public boolean isDefaultSelected(){
		return defaultSelected;
	}
	
	public void addProjectFilter( IProjectFilter filter ){
		filters.add( filter );
	}
	
	public IProject[] filterValidProjects( IProject[] allProjects ){
		ArrayList validProjects = new ArrayList( allProjects.length );
		
		for (int i = 0; i < allProjects.length; i++) {
			if( isProjectValid(allProjects[i]) )
				validProjects.add( allProjects[i]);
		}
		
		IProject [] validProjectArray = new IProject[ validProjects.size() ];
		return (IProject[])validProjects.toArray(validProjectArray);
	}
	
	/*
	 * Used IProjectFilter classes to filter the list of all projects
	 * Multiple filters are supported and ANDed together.
	 */
	public boolean isProjectValid( IProject project ){
		boolean valid = true;
		
		for (Iterator iter = filters.iterator(); iter.hasNext() && valid;) {
			IProjectFilter filter = (IProjectFilter) iter.next();
			
			valid = valid && filter.isProjectValid( project );
		}
		
		return valid;
	}
	
	public static IProject[] getAllProjects(){
		return ResourcesPlugin.getWorkspace().getRoot().getProjects();
	}
	
	public static IProject getProjectNamed( String name ){
		return ResourcesPlugin.getWorkspace().getRoot().getProject(name);
	}
}
