/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.templates.variable.value;

import org.eclipse.atf.templates.variable.IVariable;
import org.eclipse.atf.templates.variable.IVariableRegistry;
import org.eclipse.atf.templates.variable.StringVariable;
import org.eclipse.jface.text.templates.TemplateBuffer;
import org.eclipse.jface.text.templates.TemplateTranslator;
import org.eclipse.jface.text.templates.TemplateVariable;


/**
 * This class will take the String set as the value of the XML attribute in the
 * plugin.xml and parses it an IValue.
 */
public class ValueParser {

	protected static ValueParser instance = new ValueParser();

	public static ValueParser getInstance() {
		return instance;
	}

	protected ValueParser() {
		
	}
	
	/*
	 * Parses the attrValue into either a LiteralValue, a VariableValue, or a
	 * CompositeValue.
	 */
	public IValue parse( String attrValue, IVariableRegistry varReg ){
		try{
			TemplateTranslator translator = new TemplateTranslator();
			TemplateBuffer templateBuffer = translator.translate( attrValue );
			
			/*
			 * Need to use the result from templateBuffer.getString() because
			 * the translate step above changes the input string by removing
			 * the ${ and }.
			 */
			String template = templateBuffer.getString();
			
			TemplateVariable[] variables = templateBuffer.getVariables();
			
			//exit fast if there are no variables (only a literal value)
			if( variables.length == 0 ){
				return new LiteralValue( attrValue );
			}
			
			//also exit fast if there is only one Variable and used only once, plus no literals
			if( variables.length == 1 && variables[0].getOffsets().length == 1 && template.equals(variables[0].getName()) ){
				IVariable variable = varReg.lookupVariable( variables[0].getName() );
				return new VariableValue( variable );
			}
			
			//From here down, it is assumes that the attrValue contains a composition
			//of Literals and Variables.
			
			//fast way to sort using Bucket (should be more memory efficient)
			//variables could be used multiple times so they may appear multiple times in the array
			TemplateVariable [] sortedVariables = new TemplateVariable[ templateBuffer.getString().length() ];
			for (int i = 0; i < variables.length; i++) {
				int[] offsets = variables[i].getOffsets();
			
				for (int j = 0; j < offsets.length; j++) {
					sortedVariables[offsets[j]] = variables[i]; //insert in bucket using offset
				}
			}
			
			CompositeValue compositeValue = new CompositeValue();
			
			int start = -1;
			for( int i=0; i<sortedVariables.length; i++ ){
				
				if( sortedVariables[i] == null ){//&& start == -1){
					
					if( start == -1 )
						start = i; //mark the start of simple text
				}
				else{
					//this means that there is some text before the variable
					if( start != -1 ){
						String text = template.substring( start, i ); // i is is the start of the template
						start = -1; //clear the start
						compositeValue.addIValue( new LiteralValue(text) );
					}
					
					//lookup the IVariable with the name
					IVariable variable = varReg.lookupVariable(sortedVariables[i].getName());
					
					//check that the variable exists and that it is of the String type
					if( variable!= null && variable instanceof StringVariable ){
						compositeValue.addIValue( new VariableValue(variable) );
					}
					else
						//do not tie to any variable
						compositeValue.addIValue( new LiteralValue("${"+sortedVariables[i].getName()+"}?") );
					
					i+=(sortedVariables[i].getLength()-1);
				}
				
			}
			
			//check if there was some text at the end
			if( start != -1 ){
				String text = template.substring( start, template.length() );
				compositeValue.addIValue( new LiteralValue(text) );
			}
			
			return compositeValue;
		}
		catch( Exception e ){
			return new LiteralValue( "<!ERROR PARSING>" );
		}
	}
}
