/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.ui.wizard;

import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.ui.util.CssUtils;
import org.eclipse.atf.util.Utils;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;

public class CssWizardPage extends WizardPage implements SelectionListener,
		ModifyListener, ICheckStateListener {

	protected CheckboxTableViewer fTableViewer = null;
	protected Button fGenerateCssFileButton = null;
	protected Button fAdvancedButton = null;
	protected Text fNewCssFileNameText = null;
	protected WebApplicationDataModel fDataModel = null;
	
	protected IFolder fFolder = null;
	protected String fWebModuleName = null;
	protected String fNewCssFileName = "";
	protected boolean advancedOptionsShowing = false;
	protected boolean generateCssFile = true;
	protected boolean initted = false;
	
	protected Group advancedOptionsGroup= null; 
	
	
	public CssWizardPage(WebApplicationDataModel dataModel) {
		super("CssWizardPage", "Cascading style sheets", null); //$NON-NLS-1$
		setDescription("Specify new CSS file name and/or CSS imports.");
		fDataModel = dataModel;
	}
		
	
	/**
	 * Correctly resizes the table so no phantom columns appear
	 * 
	 * @param parent
	 *            the parent control
	 * @param buttons
	 *            the buttons
	 * @param table
	 *            the table
	 * @param column1
	 *            the first column
	 * @param column2
	 *            the second column
	 */
	protected void configureTableResizing(final Composite parent, final Table table, final TableColumn column1, final TableColumn column2) {
		parent.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				Rectangle area = parent.getClientArea();
				Point preferredSize = table.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				int width = area.width - 2 * table.getBorderWidth();
				if (preferredSize.y > area.height) {
					// Subtract the scrollbar width from the total column
					// width
					// if a vertical scrollbar will be required
					Point vBarSize = table.getVerticalBar().getSize();
					width -= vBarSize.x;
				}

				Point oldSize = table.getSize();
				if (oldSize.x > width) {
					// table is getting smaller so make the columns
					// smaller first and then resize the table to
					// match the client area width
					// column1 = 1/4 the width, column2 = remaining width
					column1.setWidth(width / 4);
					column2.setWidth(width - column1.getWidth());
					table.setSize(width, area.height);
				}
				else {
					// table is getting bigger so make the table
					// bigger first and then make the columns wider
					// to match the client area width
					// column1 = 1/4 the width, column2 = remaining width
					table.setSize(width, area.height);
					column1.setWidth(width / 4);
					column2.setWidth(width - column1.getWidth());
				}
			}
		});
	}
	
	public void createControl(Composite ancestor) {
		Composite composite1 = new Composite(ancestor, SWT.NONE);
		GridLayout layout = new GridLayout();
		composite1.setLayout(layout);
		composite1.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		createCssGroup(composite1);
		createAdvancedGroup(composite1);
		
		setControl(composite1);
		
	}
	
	protected void createCssGroup(Composite parent) {
		Composite composite2 = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 15;
		composite2.setLayout(layout);
		composite2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		fGenerateCssFileButton = new Button(composite2, SWT.CHECK);
		fGenerateCssFileButton.setText("Generate application CSS file");
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		fGenerateCssFileButton.setLayoutData(gd);
		fGenerateCssFileButton.addSelectionListener(this);
		
		Label nameLabel = new Label(composite2, SWT.NONE);
		nameLabel.setText("Name:");
		fNewCssFileNameText = new Text(composite2, SWT.BORDER);
		fNewCssFileNameText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		fNewCssFileNameText.addModifyListener(this);
	}
	
	protected void createAdvancedGroup(Composite parent) {
		
		fAdvancedButton = new Button(parent, SWT.PUSH);
		fAdvancedButton.setText("Advanced >>");
		fAdvancedButton.addSelectionListener(this);
		
		advancedOptionsGroup = new Group(parent, SWT.NONE);
		advancedOptionsGroup.setText("Css Imports");
		advancedOptionsGroup.setLayout(new GridLayout());
		advancedOptionsGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		//Create table
		Table table = new Table(advancedOptionsGroup, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI | SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		//Create column1
		TableColumn column1 = new TableColumn(table, SWT.NONE);
		column1.setText("Filename");
		
		//Create column2
		TableColumn column2 = new TableColumn(table, SWT.NONE);
		column2.setText("Location");
		
		GridData tableData = new GridData(GridData.FILL_BOTH);
		tableData.widthHint = 500;
		tableData.heightHint = 200;
		table.setLayoutData(tableData);
				
		configureTableResizing(advancedOptionsGroup, table, column1, column2);
		
		fTableViewer = new CheckboxTableViewer(table);
		fTableViewer.addCheckStateListener(this);
		fTableViewer.setLabelProvider(new FileIncludeLabelProvider());
		fTableViewer.setContentProvider(new FileIncludeContentProvider());
		fTableViewer.setSorter(new FileIncludeSorter());
		fTableViewer.addFilter(new FileIncludeFilter());
	}
	
	public void setVisible(boolean visible) {
		if(visible)
			initializeControlContents();
		super.setVisible(visible);
	}
	
	public boolean isInitialized() {
		return initted;
	}
	
	public void initializeControlContents() {
		if(!initted)
			initializeCssPageControlsContent();
		initted = true;
	}
	
	protected void initializeCssPageControlsContent() {
		
		fGenerateCssFileButton.setSelection(generateCssFile);
		
		if(fDataModel.getCssFileName() == null) {
			String appName = fDataModel.getAppName();
			if(appName != null) {
				String newCssFileName = Utils.capitalize(appName).replaceAll(" ", "") + ".css";
				fDataModel.setCssFileName(newCssFileName);
			} else {
				fDataModel.setCssFileName("Styles.css");
			}
		} 
		fNewCssFileNameText.setText(fDataModel.getCssFileName());
		
		String appDir = fDataModel.getAppDir();
		IFolder appFolder = fDataModel.getAppDirFolder();
		String module = fDataModel.getModuleName();
		if(appDir != null && appFolder != null) {
			try {
				updateTableContents(appFolder, module);
				
			} catch(Exception e) {
				e.printStackTrace();
			}
		} else {
			try {
				updateTableContents();
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
		
	}
	
	protected String[] getSelectedCssLocations() {
		
		Object[] objs = fTableViewer.getCheckedElements();
		if(objs == null)
			return null;
		String[] locations = new String[objs.length];
		for(int i=0; i<objs.length; i++) {
			FileIncludeInfo info = (FileIncludeInfo)objs[i];
			locations[i] = new Path(info.getLocation()).makeAbsolute().toString();
		}
		return locations;
	}
	
	public void updateTableContents(IFolder folder, String webModuleName) throws Exception {
		fFolder = folder;
		fWebModuleName = webModuleName;
		if(fFolder != null && fWebModuleName != null) {
			fTableViewer.setInput(CssUtils.getPotentialCssImageImportPaths(fFolder.getProject()));
			setDefaultSelections();
		}
	}
	
	public void updateTableContents() throws Exception {
		if(fFolder != null && fWebModuleName != null) {
			fTableViewer.setInput(CssUtils.getPotentialCssImageImportPaths(fFolder.getProject()));
			setDefaultSelections();
		}
	}
	
	protected String getNewCssFileName() {
		return fNewCssFileName;
	}
	
	protected void setDefaultSelections() {
		Object[] checkedElements = fTableViewer.getCheckedElements();
		if(checkedElements != null && checkedElements.length > 0)
			return;
		
		int tableSize = fTableViewer.getTable().getItemCount();
		for(int i=0; i<tableSize; i++) {
			FileIncludeInfo info = (FileIncludeInfo)fTableViewer.getElementAt(i);
			if(isPathInSelectedFolderCssTree(new Path(info.getLocation())))
				fTableViewer.setChecked(info, true);
			else if(isPathInSelectedFolderTree(new Path(info.getLocation())))
				fTableViewer.setChecked(info, true);
			else 
				fTableViewer.setChecked(info, false);
		}
	}
	
	protected boolean isPathInSelectedFolderCssTree(IPath cssPath) {
		
		IPath cssAreaRelPath = cssPath.removeFirstSegments(5);
		if(cssAreaRelPath.segmentCount() == 2)
			return true;
		return false;
	}
	
	protected boolean isPathInSelectedFolderTree(IPath cssPath) {
		
		return FlexibleProjectUtils.isPathInSelectedFolderTree(fFolder, cssPath);
				
	}
	
	protected boolean validatePage() {
		if(generateCssFile) {
			if(fNewCssFileName == null || fNewCssFileName.equals("")) {
				setErrorMessage("New CSS filename must be specified.");
				return false;
			}
			else if(!fNewCssFileName.endsWith(".css")) {
				setErrorMessage("New CSS filename must end with the \".css\" extension.");
				return false;
			}
		}
		
		setErrorMessage(null);
		return true;
	}

	public void widgetSelected(SelectionEvent e) {
		Widget w = e.widget;
		if(w.equals(fAdvancedButton)) {
			if(advancedOptionsShowing) {
				fAdvancedButton.setText("Advanced >>");
				advancedOptionsGroup.setVisible(false);
				advancedOptionsShowing = false;
			}
			else {
				fAdvancedButton.setText("Advanced <<");
				advancedOptionsGroup.setVisible(true);
				advancedOptionsShowing = true;
			}
		} else if(w.equals(fGenerateCssFileButton)) {
			generateCssFile = fGenerateCssFileButton.getSelection();
			fNewCssFileNameText.setEnabled(generateCssFile);
			if(!generateCssFile) {
				fDataModel.setCssFileName(null);
			} else {
				fDataModel.setCssFileName(fNewCssFileNameText.getText());
			}
			
		} 
		setPageComplete(validatePage());
		
	}

	public void widgetDefaultSelected(SelectionEvent e) {}

	public void modifyText(ModifyEvent e) {
		Widget w = e.widget;
		if(w.equals(fNewCssFileNameText)) {
			fNewCssFileName = fNewCssFileNameText.getText();
		}
		setPageComplete(validatePage());
		
	}
		
	public void updateDataModel() {
		fDataModel.setCssFileName(generateCssFile ? getNewCssFileName() : null);
    	fDataModel.setCssImports(getSelectedCssLocations());
	}
	
	public void checkStateChanged(CheckStateChangedEvent event) {
		fDataModel.setCssImports(getSelectedCssLocations());
		
	}

}
