/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.ui.wizard;

import java.util.ArrayList;

import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.util.Utils;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;

public class FrontWizardPage extends WizardPage implements SelectionListener, ModifyListener {

	protected Text nameText;
	protected Combo projectCombo;
	protected Text appDirText;
	//protected Button projectNewButton;
	protected Combo driverTypeCombo;
	//protected Combo moduleCombo;
	protected Text driverFileNameText;
	protected WebApplicationDataModel dataModel;
	protected String[] targetProjectRequiredNatures;
	protected String[] j2eeControllerTypes = new String[] {"html", "jsp"};
	protected String[] basicControllerTypes = new String[] {"html"};
	protected boolean j2eeControllerMode = false;
		
	protected String appName;
	protected String projectName;
	protected String moduleName;
	protected String appDir;
	protected String controllerType;
	protected String controllerFileName;
	
	boolean initted = false;
	
	public FrontWizardPage(String pageName, WebApplicationDataModel dataModel, String[] targetProjectRequiredNatures) {
		super(pageName);
		this.dataModel = dataModel;
		this.targetProjectRequiredNatures = targetProjectRequiredNatures;
		setTitle("Name and Location"); //$NON-NLS-1$
	    setDescription("Specify name and location of new web application"); //$NON-NLS-1$
	}
	
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setFont(parent.getFont());
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
				
		createFrontPageControls(composite);
				
		setControl(composite);
		Dialog.applyDialogFont(composite);

	}
	
	public void initializeControlContents() {
		if(!initted)
			initializeFrontPageControlsContent();
		initted = true;
	}
	
	protected void initializeFrontPageControlsContent() {
		 
		String[] projectNames = getProjectNames();
		projectCombo.setItems(projectNames);
		
		IProject selectedProject = ((WebApplicationWizard)getWizard()).getInitiallySelectedProject();
		if(selectedProject != null && projectHasRequiredNatures(selectedProject)) {
			projectName = selectedProject.getName();
			projectCombo.setText(projectName);
		}
		else if(projectNames.length > 0){
			projectName = projectCombo.getItem(0);
			projectCombo.setText(projectName);
		}
				
		/*if(projectName != null && moduleCombo != null) {
			//String[] modules = J2EEProjectUtils.getJ2EEModuleNames(projectName);
			try {
				String[] modules = FlexibleProjectUtils.getModuleNames(projectName);
				if(modules != null) {
					moduleCombo.setItems(modules);
					moduleCombo.setText(modules[0]);
					moduleName = modules[0];
				}
			} catch(CoreException e) {
				UIPlugin.log(e);
			}
		}
		else*/ 
		if(projectName != null) {
			moduleName = projectName;
		}
		
		try {
			if(selectedProject != null) {
				//if(selectedProject.hasNature(J2EEProjectUtils.J2EE_NATURE_ID)) {
				if(FlexibleProjectUtils.isJ2EEProject(selectedProject)) {
					driverTypeCombo.setItems(j2eeControllerTypes);
					j2eeControllerMode = true;
					driverTypeCombo.setText(j2eeControllerTypes[0]);
					controllerType = j2eeControllerTypes[0];
				}
				else {
					driverTypeCombo.setItems(basicControllerTypes);
					j2eeControllerMode = false;
					driverTypeCombo.setText(basicControllerTypes[0]);
					controllerType = basicControllerTypes[0];
				}
			}
		} catch(CoreException e) {
			e.printStackTrace();
		}
		
	}
	
	private boolean projectHasRequiredNatures(IProject project) {
		if(targetProjectRequiredNatures == null)
			return true;
		
		boolean match = true;
		for(int i=0; i<targetProjectRequiredNatures.length; i++) {
			try {
				if(project.hasNature(targetProjectRequiredNatures[i])) {
					continue;
				}
				else {
					match = false;
					break;
				}
			} catch(CoreException e) {
				e.printStackTrace();
			}
		}
		return match;
	}
	
	private String[] getProjectNames() {
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		ArrayList list = new ArrayList();
		for(int i=0; i<projects.length; i++) {
			if(projectHasRequiredNatures(projects[i])) {
				list.add(projects[i].getName());
			}
		}
		String[] p = new String[list.size()];
		p = (String[])list.toArray(p);
		return p;
	}
	
	protected void createFrontPageControls(Composite parent) {
		
		Group group1 = new Group(parent, SWT.NONE);
		group1.setFont(parent.getFont());
		group1.setText("Location");
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		group1.setLayout(layout);
		group1.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
				
		Label nameLabel = new Label(group1, SWT.NONE);
		nameLabel.setText("Name:");
		nameText = new Text(group1, SWT.BORDER);
		nameText.addModifyListener(this);
		GridData gData = new GridData(GridData.FILL_HORIZONTAL);
		nameText.setLayoutData(gData);
		//new Label(group1, SWT.NONE);
				
		Label projectLabel = new Label(group1, SWT.NONE);
		projectLabel.setText("Project:");
		projectCombo = new Combo(group1, SWT.DROP_DOWN | SWT.READ_ONLY);
		gData = new GridData(GridData.FILL_HORIZONTAL);
		projectCombo.setLayoutData(gData);
		projectCombo.addModifyListener(this);
		//projectNewButton = new Button(group1, SWT.PUSH);
		//projectNewButton.setText("New...");
		//projectNewButton.addSelectionListener(this);
		
		//if(FlexibleJavaProjectPreferenceUtil.getMultipleModulesPerProjectProp()) {
			//Label moduleLabel = new Label(group1, SWT.NONE);
			//moduleLabel.setText("Module:");
			//moduleCombo = new Combo(group1, SWT.DROP_DOWN | SWT.READ_ONLY);
			//gData = new GridData(GridData.FILL_HORIZONTAL);
			//moduleCombo.setLayoutData(gData);
			//moduleCombo.addModifyListener(this);
			//new Label(group1, SWT.NONE);
		//}
		
		Label appDirLabel = new Label(group1, SWT.NONE);
		appDirLabel.setText("Root directory:");
		appDirText = new Text(group1, SWT.BORDER);
		appDirText.addModifyListener(this);
		gData = new GridData(GridData.FILL_HORIZONTAL);
		appDirText.setLayoutData(gData);
		//new Label(group1, SWT.NONE);
		
		Group group2 = new Group(parent, SWT.NONE);
		group2.setFont(parent.getFont());
		group2.setText("Controller");
		layout = new GridLayout();
		layout.numColumns = 2;
		group2.setLayout(layout);
		group2.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Label typeLabel = new Label(group2, SWT.NONE);
		typeLabel.setText("Type:");
		driverTypeCombo = new Combo(group2, SWT.DROP_DOWN | SWT.READ_ONLY);
		driverTypeCombo.addModifyListener(this);
		
		Label fileNameLabel = new Label(group2, SWT.NONE);
		fileNameLabel.setText("File Name:");
		driverFileNameText = new Text(group2, SWT.BORDER);
		driverFileNameText.addModifyListener(this);
		gData = new GridData(GridData.FILL_HORIZONTAL);
		driverFileNameText.setLayoutData(gData);
				
	}
	
	public void setVisible(boolean visible) {
		initializeControlContents();
		super.setVisible(visible);
	}
	
	protected String getAppDir() {
		return appDir;
	}

	protected String getControllerFileName() {
		return controllerFileName;
	}

	protected String getControllerType() {
		return controllerType;
	}
	
	protected String getAppName() {
		return appName;
	}
	
	protected String getProjectName() {
		return projectName;
	}
	
	protected String getModuleName() {
		if(moduleName == null)
			return projectName;
		return moduleName;
	}
	
	protected IFolder getAppDirFolder() {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		if(project == null || appDir == null)
			return null;
		/*try {
			if(project.hasNature(J2EEProjectUtils.J2EE_NATURE_ID)) {
				if("true".equals(PersistentProjectProperties.isMultipleWebModulesEnabled(project))) {
					return project.getFolder(moduleName).getFolder("WebContent").getFolder(appDir);
				}
				else {
					return project.getFolder("WebContent").getFolder(appDir);
				}
			} else {
				IStaticWebNature sNature = StaticWebProjectUtils.getStaticWebNature(project);
				if(sNature != null) {
					return sNature.getRootPublishableFolder().getFolder(new Path(appDir));
				}
				return null;
			}
		} catch(CoreException e) {
			e.printStackTrace();
			return null;
		}*/
		return project.getFolder("WebContent").getFolder(appDir);
	}

	public void widgetSelected(SelectionEvent e) {
		//Widget w = e.widget;
			
		//if(w.equals(projectNewButton)) {
		//	newProjectButtonSelected(e);
			
		//}
		
	}

	public void widgetDefaultSelected(SelectionEvent e) {}

	public void modifyText(ModifyEvent e) {
		Widget w = e.widget;
		if(w.equals(projectCombo)) {
			projectComboModified(e);
		}
		//else if(w.equals(moduleCombo)) {
		//	moduleComboModified(e);
		//}
		else if(w.equals(driverTypeCombo)) {
			driverTypeComboModified(e);
		}
		else if(w.equals(nameText)) {
			nameTextModified(e);
			
		}
		else if(w.equals(appDirText)) {
			appDirTextModified(e);
			
		}
		else if(w.equals(driverFileNameText)) {
			driverFileNameTextModified(e);
		}
		
		setPageComplete(validatePage());
	}
	
	protected void projectComboModified(ModifyEvent e) {
		projectName = projectCombo.getText();
				
		if(!projectName.equals("")) {
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
			try {
				//if(project.hasNature(J2EEProjectUtils.J2EE_NATURE_ID)) {
				if(FlexibleProjectUtils.isJ2EEProject(project)) {
					if(!j2eeControllerMode) {
						driverTypeCombo.setItems(j2eeControllerTypes);
						j2eeControllerMode = true;
						driverTypeCombo.setText(j2eeControllerTypes[0]);
					}
				}
				else if(j2eeControllerMode) {
					driverTypeCombo.setItems(basicControllerTypes);
					j2eeControllerMode = false;
					driverTypeCombo.setText(basicControllerTypes[0]);
				}
				
			} catch(CoreException ce) {
				ce.printStackTrace();
			}
			//if(moduleCombo != null) {
			//	String[] modules = J2EEProjectUtils.getJ2EEModuleNames(projectName);
			//	try {
			//		String[] modules = FlexibleProjectUtils.getModuleNames(projectName);
			//		if(modules != null) {
			//			moduleCombo.setItems(modules);
			//			moduleName = modules[0];
			//			moduleCombo.setText(moduleName);
			//		}
			//	} catch(CoreException ex) {
			//		UIPlugin.log(ex);
			//	}
			//}
			//else {
			//	moduleName = projectName;
			//}
		}
	}
	
	//protected void moduleComboModified(ModifyEvent e) {
	//	moduleName = moduleCombo.getText();
	//}
	
	protected void driverTypeComboModified(ModifyEvent e) {
		controllerType = driverTypeCombo.getText();
		if(controllerFileName != null && !controllerFileName.equals("")) {
			controllerFileName = Utils.capitalize(appName.replaceAll(" ", "")) + "." + controllerType;
			driverFileNameText.setText(controllerFileName);
		}
	}
	
	protected void nameTextModified(ModifyEvent e) {
		appName = nameText.getText();
		appDir = Utils.uncapitalize(appName.replaceAll(" ", ""));
		appDirText.setText(appDir);
		controllerFileName = Utils.capitalize(appName.replaceAll(" ", "")) + "." + controllerType;
		driverFileNameText.setText(controllerFileName);
	}
	
	protected void appDirTextModified(ModifyEvent e) {
		appDir = appDirText.getText();
	}
	
	protected void driverFileNameTextModified(ModifyEvent e) {
		controllerFileName = driverFileNameText.getText();
	}
	
	protected boolean validatePage() {
		
		if(appName == null || appName.equals("")) {
			setErrorMessage("New application name must be specified.");
			return false;
		}
		else if(projectName == null || projectName.equals("")) {
			setErrorMessage("Target project must be specified.");
			return false;
		}
		else if(appDir == null || appDir.equals("")) {
			setErrorMessage("Application root directory must be specified.");
			return false;
		}
				
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		try {
			
			IContainer webContentFolder = FlexibleProjectUtils.getWebContentFolder(project);
			
			if(!webContentFolder.getFullPath().isValidPath(appDir)) {
				setErrorMessage("Specified application directory root is not a valid path.");
				return false;
			}
			else if(webContentFolder.getFolder(new Path(appDir)).exists()) {
				setErrorMessage("Specified application root directory already exists.");
				return false;
			}
		} catch(Exception e) {
			setErrorMessage(e.getLocalizedMessage());
			return false;
		}
		
		if(controllerFileName == null || controllerFileName.equals("")) {
			setErrorMessage("Controller file name must be specified.");
			return false;
		}
		else if(!controllerFileName.endsWith("." + controllerType)) {
			setErrorMessage("Controller file name must end with the \"."+controllerType+"\" extension.");
			return false;
		}
		
		setErrorMessage(null);
		return true;
	}
	
	public void setPageComplete(boolean complete) {
        super.setPageComplete(complete);
        if(complete) {
        	updateDataModel();
		}
    }
	
	public void updateDataModel() {
		dataModel.setAppDir(getAppDir());
    	dataModel.setAppDirFolder(getAppDirFolder());
    	dataModel.setAppName(getAppName());
    	dataModel.setControllerFileName(getControllerFileName());
    	dataModel.setControllerType(getControllerType());
    	dataModel.setProjectName(getProjectName());
    	dataModel.setModuleName(getModuleName());
    	if(getAppName() != null) {
			String newJsFileName = Utils.capitalize(getAppName()).replaceAll(" ", "") + ".js";
			dataModel.setJavaScriptFileName(newJsFileName);
			String newCssFileName = Utils.capitalize(getAppName()).replaceAll(" ", "") + ".css";
			dataModel.setCssFileName(newCssFileName);
		}
	}

}
