/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.ui.wizard;

import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.ui.util.JsUtils;
import org.eclipse.atf.util.Utils;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;

public class JsWizardPage extends WizardPage implements SelectionListener, ModifyListener, ICheckStateListener {

	protected CheckboxTableViewer fTableViewer = null;
	protected Button fAdvancedButton = null;
	protected Text fNewJsFileNameText = null;
	protected Button fGenerateJsFileButton = null;
	protected Group advancedOptionsGroup= null; 
	
	protected WebApplicationDataModel fDataModel = null;
	
	protected IFolder fFolder = null;
	protected String fWebModuleName = null;
	protected String fNewJsFileName = "";
	protected boolean generateJsFile = true;
	protected boolean advancedOptionsShowing = false;
	protected boolean initted = false;
	
	
	public JsWizardPage(WebApplicationDataModel dataModel) {
		super("JsWizardPage", "JavaScript", null); //$NON-NLS-1$
		setDescription("Specify new JavaScript application file name and/or JavaScript imports.");
		fDataModel = dataModel;
	}
	
	/**
	 * Correctly resizes the table so no phantom columns appear
	 * 
	 * @param parent
	 *            the parent control
	 * @param buttons
	 *            the buttons
	 * @param table
	 *            the table
	 * @param column1
	 *            the first column
	 * @param column2
	 *            the second column
	 */
	protected void configureTableResizing(final Composite parent, final Table table, final TableColumn column1, final TableColumn column2) {
		parent.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				Rectangle area = parent.getClientArea();
				Point preferredSize = table.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				int width = area.width - 2 * table.getBorderWidth();
				if (preferredSize.y > area.height) {
					// Subtract the scrollbar width from the total column
					// width
					// if a vertical scrollbar will be required
					Point vBarSize = table.getVerticalBar().getSize();
					width -= vBarSize.x;
				}

				Point oldSize = table.getSize();
				if (oldSize.x > width) {
					// table is getting smaller so make the columns
					// smaller first and then resize the table to
					// match the client area width
					column1.setWidth(width / 4);
					column2.setWidth(width - column1.getWidth());
					table.setSize(width, area.height);
				}
				else {
					// table is getting bigger so make the table
					// bigger first and then make the columns wider
					// to match the client area width
					table.setSize(width, area.height);
					column1.setWidth(width / 4);
					column2.setWidth(width - column1.getWidth());
				}
			}
		});
	}
	
	public void createControl(Composite ancestor) {
		
		Composite composite1 = new Composite(ancestor, SWT.NONE);
		GridLayout layout = new GridLayout();
		composite1.setLayout(layout);
		composite1.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		createJsApplicationGroup(composite1);
		createAdvancedGroup(composite1);
		
		setControl(composite1);
		
	}
	
	protected void createJsApplicationGroup(Composite parent) {
		
		Group jsApplicationGroup = new Group(parent, SWT.NONE);
		jsApplicationGroup.setText("JavaScript Application");
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 15;
		jsApplicationGroup.setLayout(layout);
		jsApplicationGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		fGenerateJsFileButton = new Button(jsApplicationGroup, SWT.CHECK);
		fGenerateJsFileButton.setText("Generate JavaScript application file");
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		fGenerateJsFileButton.setLayoutData(gd);
		fGenerateJsFileButton.addSelectionListener(this);
		
		Label nameLabel = new Label(jsApplicationGroup, SWT.NONE);
		nameLabel.setText("Filename:");
		fNewJsFileNameText = new Text(jsApplicationGroup, SWT.BORDER);
		fNewJsFileNameText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		fNewJsFileNameText.addModifyListener(this);
	}
	
	protected void createAdvancedGroup(Composite parent) {
		
		fAdvancedButton = new Button(parent, SWT.PUSH);
		fAdvancedButton.setText("Advanced >>");
		fAdvancedButton.addSelectionListener(this);
		
		advancedOptionsGroup = new Group(parent, SWT.NONE);
		advancedOptionsGroup.setText("JavaScript Imports");
		advancedOptionsGroup.setLayout(new GridLayout());
		advancedOptionsGroup.setLayoutData(new GridData(GridData.FILL_BOTH));
		advancedOptionsGroup.setVisible(false);
		
		//Create table
		Table table = new Table(advancedOptionsGroup, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI | SWT.CHECK | SWT.H_SCROLL | SWT.V_SCROLL);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		//Create column1
		TableColumn column1 = new TableColumn(table, SWT.NONE);
		column1.setText("Filename");
		
		//Create column2
		TableColumn column2 = new TableColumn(table, SWT.NONE);
		column2.setText("Location");
		
		GridData tableData = new GridData(GridData.FILL_BOTH);
		tableData.widthHint = 500;
		tableData.heightHint = 200;
		table.setLayoutData(tableData);
				
		configureTableResizing(advancedOptionsGroup, table, column1, column2);
		
		fTableViewer = new CheckboxTableViewer(table);
		fTableViewer.addCheckStateListener(this);
		fTableViewer.setLabelProvider(new FileIncludeLabelProvider());
		fTableViewer.setContentProvider(new FileIncludeContentProvider());
		fTableViewer.setSorter(new FileIncludeSorter());
		fTableViewer.addFilter(new FileIncludeFilter());
	}
	
	public void setVisible(boolean visible) {
		if(visible)
			initializeControlContents();
		super.setVisible(visible);
	}
	
	public boolean isInitialized() {
		return initted;
	}
	
	public void initializeControlContents() {
		if(!initted)
			initializeJsPageControlsContent();
		initted = true;
	}
	
	protected void initializeJsPageControlsContent() {
		
		
		fGenerateJsFileButton.setSelection(generateJsFile);
		
		if(fDataModel.getJavaScriptFileName() == null) {
			String appName = fDataModel.getAppName();
			if(appName != null) {
				String newJsFileName = Utils.capitalize(appName).replaceAll(" ", "") + ".js";
				fDataModel.setJavaScriptFileName(newJsFileName);
			} else {
				fDataModel.setJavaScriptFileName("App.js");
			}
		} 
		fNewJsFileNameText.setText(fDataModel.getJavaScriptFileName());
		
		String appDir = fDataModel.getAppDir();
		IFolder appFolder = fDataModel.getAppDirFolder();
		String module = fDataModel.getModuleName();
		if(appDir != null && appFolder != null) {
			try {
				updateTableContents(appFolder, module);
				
			} catch(Exception e) {
				e.printStackTrace();
			}
		} else {
			try {
				updateTableContents();
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void setNewJsFileName(String filename) {
		fNewJsFileName = filename;
	}
	
	public String[] getSelectedJsLocations() {
		
		Object[] objs = fTableViewer.getCheckedElements();
		if(objs == null)
			return null;
		String[] locations = new String[objs.length];
		for(int i=0; i<objs.length; i++) {
			FileIncludeInfo info = (FileIncludeInfo)objs[i];
			locations[i] = new Path(info.getLocation()).makeAbsolute().toString();
		}
		return locations;
	}
	
	public void updateTableContents(IFolder folder, String webModuleName) throws Exception {
		fFolder = folder;
		fWebModuleName = webModuleName;
		if(fFolder != null && fWebModuleName != null) {
			fTableViewer.setInput(JsUtils.getPotentialJsIncludePaths(fFolder.getProject()));
			setDefaultSelections();
		}
	}
	
	public void updateTableContents() throws Exception {
		if(fFolder != null && fWebModuleName != null) {
			fTableViewer.setInput(JsUtils.getPotentialJsIncludePaths(fFolder.getProject()));
			setDefaultSelections();
		}
	}
	
	protected String getNewJsFileName() {
		return fNewJsFileName;
	}
	
	protected void setDefaultSelections() {
		Object[] checkedElements = fTableViewer.getCheckedElements();
		if(checkedElements != null && checkedElements.length > 0)
			return;
		
		int tableSize = fTableViewer.getTable().getItemCount();
		for(int i=0; i<tableSize; i++) {
			FileIncludeInfo info = (FileIncludeInfo)fTableViewer.getElementAt(i);
			if(isPathInSelectedFolderTree(new Path(info.getLocation())))
				fTableViewer.setChecked(info, true);
			else 
				fTableViewer.setChecked(info, false);
		}
	}
	
	protected boolean isPathInSelectedFolderTree(IPath jsPath) {
		
		/*try {
			if(fFolder.getProject().hasNature(J2EEProjectUtils.J2EE_NATURE_ID)) {
				return J2EEProjectUtils.isPathInSelectedFolderTree(fFolder, jsPath);
			} else {
				return StaticWebProjectUtils.isPathInSelectedFolderTree(fFolder, jsPath);
			}
		} catch(CoreException e) {
			e.printStackTrace();
			return false;
		}*/
		return FlexibleProjectUtils.isPathInSelectedFolderTree(fFolder, jsPath);
				
	}
		
	protected boolean validatePage() {
		
		if(generateJsFile) {
			if(fNewJsFileName == null || fNewJsFileName.equals("")) {
				setErrorMessage("New JavaScript application filename must be specified.");
				return false;
			}
			else if(!fNewJsFileName.endsWith(".js")) {
				setErrorMessage("New JavaScript application filename must end with the \".js\" extension.");
				return false;
			}
		}
				
		setErrorMessage(null);
		return true;
	}

	public void widgetSelected(SelectionEvent e) {
		Widget w = e.widget;
		if(w.equals(fAdvancedButton)) {
			if(advancedOptionsShowing) {
				fAdvancedButton.setText("Advanced >>");
				advancedOptionsGroup.setVisible(false);
				advancedOptionsShowing = false;
			}
			else {
				fAdvancedButton.setText("Advanced <<");
				advancedOptionsGroup.setVisible(true);
				advancedOptionsShowing = true;
			} 
		} else if(w.equals(fGenerateJsFileButton)) {
			generateJsFile = fGenerateJsFileButton.getSelection();
			fNewJsFileNameText.setEnabled(generateJsFile);
			if(!generateJsFile) {
				fDataModel.setJavaScriptFileName(null);
			} else {
				fDataModel.setJavaScriptFileName(fNewJsFileNameText.getText());
			}
			
		}
		setPageComplete(validatePage());
		
	}

	public void widgetDefaultSelected(SelectionEvent e) {}

	public void modifyText(ModifyEvent e) {
		Widget w = e.widget;
		if(w.equals(fNewJsFileNameText)) {
			fNewJsFileName = fNewJsFileNameText.getText();
		}
		setPageComplete(validatePage());
		
	}
		
	public void updateDataModel() {
		fDataModel.setJavaScriptFileName(generateJsFile ? getNewJsFileName() : null);
    	fDataModel.setJavaScriptIncludes(getSelectedJsLocations());
	}

	public void checkStateChanged(CheckStateChangedEvent event) {
		fDataModel.setJavaScriptIncludes(getSelectedJsLocations());
		
	}

}
