/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.ui.wizard;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;

import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.ui.UIPlugin;
import org.eclipse.atf.ui.templates.TemplateRegistry;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.templates.DocumentTemplateContext;
import org.eclipse.jface.text.templates.SimpleTemplateVariableResolver;
import org.eclipse.jface.text.templates.Template;
import org.eclipse.jface.text.templates.TemplateBuffer;
import org.eclipse.jface.text.templates.TemplateContext;
import org.eclipse.jface.text.templates.TemplateContextType;
import org.eclipse.jface.text.templates.persistence.TemplateStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jst.jsp.core.internal.JSPCorePlugin;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.ContainerGenerator;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.wst.sse.core.internal.encoding.CommonEncodingPreferenceNames;

/**
 * Note: this wizard creates an application either based on JSP or HTML.
 * The JSP options introduces a dependency on JST and the JSp support for the 
 * whole plugin.
 * TODO: Consider refactoring the JST specific bist in another plugin 
 * or fragment so that this plugin can be used even when JST is not installed
 *
 */
public abstract class WebApplicationWizard extends Wizard implements INewWizard {

	protected IWorkbench mWorkbench;
	protected IStructuredSelection mSelection;
	protected FrontWizardPage mFrontPage;
	protected CssWizardPage mCssPage;
	protected JsWizardPage mJsPage;
	protected WebApplicationDataModel mDataModel;
		
	protected static final String JSP_CONTROLLER_TYPE = "jsp";
	protected static final String HTML_CONTROLLER_TYPE = "html";
	
	public WebApplicationWizard() {
		super();
	    setNeedsProgressMonitor(true);
	    setWindowTitle("New Web Application"); //$NON-NLS-1$
	}
	public boolean performFinish() {
		
		IFile controllerFile = null;
		if(!mCssPage.isInitialized()) {
			mCssPage.initializeControlContents();
			mCssPage.updateDataModel();
		}
		if(!mJsPage.isInitialized()) {
			mJsPage.initializeControlContents();
			mJsPage.updateDataModel();
		}
			
		//create the controller file
		if(mDataModel.getControllerType().equals(JSP_CONTROLLER_TYPE)) {
			controllerFile = createNewJspFile();
		} else if(mDataModel.getControllerType().equals(HTML_CONTROLLER_TYPE)) {
			controllerFile = createNewHtmlFile();
		}
		//create the JavaScript file
		IFile jsFile = null;
		//if(mJsPage.doGenerateNewJsFile())
			jsFile = createNewJavaScriptFile();
		//create the css file if necessary
		//if(mCssPage.doGenerateNewCssFile())
			createNewCssFile();
		//open the controller file in editor
		if(controllerFile != null)
			openEditor(controllerFile);
		if(jsFile != null) 
			openEditor(jsFile);
		return true;
	}

	public void init(IWorkbench workbench, IStructuredSelection selection) {
		mWorkbench = workbench;
	    mSelection = selection;
	    addTemplateContextTypes();
	}
	
	protected abstract void addTemplateContextTypes();
	
	protected abstract String getJsTemplateId();
	
	protected abstract String getCssTemplateId();
	
	protected abstract String getHtmlTemplateId();
	
	protected abstract String getJspTemplateId();
	
	public void addPages()
	{
	  super.addPages();
	  mDataModel = new WebApplicationDataModel();
	  mFrontPage = new FrontWizardPage("New Web Application", mDataModel, new String[0]);
	  mJsPage = new JsWizardPage(mDataModel);
	  mCssPage = new CssWizardPage(mDataModel);
	  addPage(mFrontPage);
	  addPage(mJsPage);
	  addPage(mCssPage);
	  
	  
	}
		
	public IProject getInitiallySelectedProject() {
		if(mSelection != null && mSelection.getFirstElement() instanceof IResource) {
			return ((IResource)mSelection.getFirstElement()).getProject();
		}
		return null;
	}
	
	public IProject getSelectedProject() {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(mDataModel.getProjectName());
		return project;
	}
	
	public IWorkbench getWorkbench() {
		return mWorkbench;
	}
	
	public IStructuredSelection getSelection() {
		return mSelection;
	}
	
	protected IFile createNewJavaScriptFile() {
		
		IFile file = null;
		try {
			file = createNewFile(mDataModel.getJavaScriptFileName());
			
		} catch(Exception e) {
			e.printStackTrace();
			return null;
		}
		
		TemplateStore templateStore = TemplateRegistry.getInstance().getTemplateStore();
		Template template = templateStore.findTemplateById(getJsTemplateId());
		String templateString = getTemplateString(template);
		
		try {
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			OutputStreamWriter outputStreamWriter = null;
			outputStreamWriter = new OutputStreamWriter(outputStream);
			
			outputStreamWriter.write(templateString);
			outputStreamWriter.flush();
			outputStreamWriter.close();
			ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
			file.setContents(inputStream, true, false, null);
			inputStream.close();
		}
		catch (Exception e) {
			UIPlugin.log(IStatus.ERROR, e, "Could not create contents for new JavaScript file"); //$NON-NLS-1$
		}
		

		return file;
	}

	protected IFile createNewJspFile() {
		
		IFile file = null;
		try {
			file = createNewFile(mDataModel.getControllerFileName());
		} catch(Exception e) {
			e.printStackTrace();
			return null;
		}
		
		//determine the encoding for the new file
		Preferences preference = JSPCorePlugin.getDefault().getPluginPreferences();
		String charSet = preference.getString(CommonEncodingPreferenceNames.OUTPUT_CODESET);

		TemplateStore templateStore = TemplateRegistry.getInstance().getTemplateStore();
		Template template = templateStore.findTemplateById(getJspTemplateId());
		String templateString = getTemplateString(template);
		
		try {
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			OutputStreamWriter outputStreamWriter = null;
			if (charSet == null || charSet.trim().equals("")) { //$NON-NLS-1$
				// just use default encoding
				outputStreamWriter = new OutputStreamWriter(outputStream);
			} else {
				outputStreamWriter = new OutputStreamWriter(outputStream, charSet);
			}
			outputStreamWriter.write(templateString);
			outputStreamWriter.flush();
			outputStreamWriter.close();
			ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
			file.setContents(inputStream, true, false, null);
			inputStream.close();
		}
		catch (Exception e) {
			UIPlugin.log(IStatus.ERROR, e, "Could not create contents for new JSP file"); //$NON-NLS-1$
		}
		

		return file;
	}
	
	protected IFile createNewHtmlFile() {
		
		IFile file = null;
		try {
			file = createNewFile(mDataModel.getControllerFileName());
		} catch(Exception e) {
			e.printStackTrace();
			return null;
		}
		
		TemplateStore templateStore = TemplateRegistry.getInstance().getTemplateStore();
		Template template = templateStore.findTemplateById(getHtmlTemplateId());
		String templateString = getTemplateString(template);
		
		try {
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			OutputStreamWriter outputStreamWriter = null;
			outputStreamWriter = new OutputStreamWriter(outputStream);
			
			outputStreamWriter.write(templateString);
			outputStreamWriter.flush();
			outputStreamWriter.close();
			ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
			file.setContents(inputStream, true, false, null);
			inputStream.close();
		}
		catch (Exception e) {
			UIPlugin.log(IStatus.ERROR, e, "Could not create contents for new JSP file"); //$NON-NLS-1$
		}
		

		return file;
	}
	
	protected IFile createNewCssFile() {
		
		IFile file = null;
		try {
			file = createNewFile(mDataModel.getCssFileName());
		} catch(Exception e) {
			e.printStackTrace();
			return null;
		}
		
		TemplateStore templateStore = TemplateRegistry.getInstance().getTemplateStore();
		Template template = templateStore.findTemplateById(getCssTemplateId());
		String templateString = getTemplateString(template);
		
		try {
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			OutputStreamWriter outputStreamWriter = null;
			outputStreamWriter = new OutputStreamWriter(outputStream);
			
			outputStreamWriter.write(templateString);
			outputStreamWriter.flush();
			outputStreamWriter.close();
			ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
			file.setContents(inputStream, true, false, null);
			inputStream.close();
		}
		catch (Exception e) {
			UIPlugin.log(IStatus.ERROR, e, "Could not create contents for new CSS file"); //$NON-NLS-1$
		}
		

		return file;
	}
	
	/**
	 * Evaluates the provided template, calling setResolverEvaluationString for all resolvers associated
	 * with the template, and returns the resulting string. 
	 * @param template
	 * @return
	 */
	protected String getTemplateString(Template template) {
		
		String templateString = null;
		String contextTypeId = template.getContextTypeId();
		
		if (template != null) {
			TemplateContextType contextType = TemplateRegistry.getInstance().getTemplateContextRegistry().getContextType(contextTypeId);
			Iterator i = contextType.resolvers();
			while(i.hasNext()) {
				Object resolver = i.next();
				if(resolver instanceof SimpleTemplateVariableResolver)
					setResolverEvaluationString((SimpleTemplateVariableResolver)resolver);
			}
			IDocument document = new Document();
			TemplateContext context = new DocumentTemplateContext(contextType, document, 0, 0);
			try {
				TemplateBuffer buffer = context.evaluate(template);
				templateString = buffer.getString();
			}
			catch (Exception e) {
				UIPlugin.log(IStatus.ERROR, e, "Error creating template"); //$NON-NLS-1$
			}
		}

		return templateString;
	}
	
	protected abstract void setResolverEvaluationString(SimpleTemplateVariableResolver resolver);
		
	
	protected IFile createNewFile(String filename) throws Exception {
		
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(mDataModel.getProjectName());
		String module = mDataModel.getModuleName();
		
        // create the new file and cache it if successful
		IPath container;
		/*if(project.hasNature(J2EEProjectUtils.J2EE_NATURE_ID)) 
			container = J2EEProjectUtils.getWebContentFolder(project, module).getFolder(new Path(mDataModel.getAppDir())).getFullPath();
		else 
			container = StaticWebProjectUtils.getWebContentFolder(project, module).getFolder(new Path(mDataModel.getAppDir())).getFullPath();
		*/
		container = FlexibleProjectUtils.getWebContentFolder(project).getFolder(new Path(mDataModel.getAppDir())).getFullPath();
		
		final IPath containerPath = container;
		IPath newFilePath = containerPath.append(filename);
        final IFile newFileHandle = IDEWorkbenchPlugin.getPluginWorkspace().getRoot().getFile(newFilePath);
       
        WorkspaceModifyOperation op = new WorkspaceModifyOperation(createRule(newFileHandle)) {
            protected void execute(IProgressMonitor monitor)
                    throws CoreException {
                try {
                    monitor.beginTask(IDEWorkbenchMessages.WizardNewFileCreationPage_progress, 2000);
                    ContainerGenerator generator = new ContainerGenerator(containerPath);
                    generator.generateContainer(new SubProgressMonitor(monitor, 1000));
                    createFile(newFileHandle, null, new SubProgressMonitor(monitor, 1000));
                } finally {
                    monitor.done();
                }
            }
        };

        try {
            getContainer().run(true, true, op);
        } catch (InterruptedException e) {
            return null;
        } catch (InvocationTargetException e) {
            if (e.getTargetException() instanceof CoreException) {
                ErrorDialog
                        .openError(
                                getContainer().getShell(), // Was Utilities.getFocusShell()
                                IDEWorkbenchMessages.WizardNewFileCreationPage_errorTitle,
                                null, // no special message
                                ((CoreException) e.getTargetException())
                                        .getStatus());
            } else {
                // CoreExceptions are handled above, but unexpected runtime exceptions and errors may still occur.
                IDEWorkbenchPlugin.log(getClass(),
                        "createNewFile()", e.getTargetException()); //$NON-NLS-1$
                MessageDialog
                        .openError(
                                getContainer().getShell(),
                                IDEWorkbenchMessages.WizardNewFileCreationPage_internalErrorTitle, NLS.bind(IDEWorkbenchMessages.WizardNewFileCreationPage_internalErrorMessage, e.getTargetException().getMessage()));
            }
            return null;
        }

        return newFileHandle;
	}
	
	protected void createFile(IFile fileHandle, InputStream contents,
            IProgressMonitor monitor) throws CoreException {
        if (contents == null)
            contents = new ByteArrayInputStream(new byte[0]);

        try {
            IPath path = fileHandle.getFullPath();
            IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
            int numSegments= path.segmentCount();
            if (numSegments > 2 && !root.getFolder(path.removeLastSegments(1)).exists()) {
            	// If the direct parent of the path doesn't exist, try to create the
                // necessary directories.
                for (int i= numSegments - 2; i > 0; i--) {
                	IFolder folder = root.getFolder(path.removeLastSegments(i));
                    if (!folder.exists()) {
                        folder.create(false, true, monitor);
                    }
                }
            }
            fileHandle.create(contents, false, monitor);
            
        } catch (CoreException e) {
            // If the file already existed locally, just refresh to get contents
            if (e.getStatus().getCode() == IResourceStatus.PATH_OCCUPIED)
                fileHandle.refreshLocal(IResource.DEPTH_ZERO, null);
            else
                throw e;
        }

        if (monitor.isCanceled())
            throw new OperationCanceledException();
    }
	
	protected ISchedulingRule createRule(IResource resource) {
		IResource parent = resource.getParent();
    	while (parent != null) {
    		if (parent.exists())
    			return resource.getWorkspace().getRuleFactory().createRule(resource);
    		resource = parent;
    		parent = parent.getParent();
    	}
		return resource.getWorkspace().getRoot();
	}
	
	/**
	 * Opens the specified file with the associated editor. 
	 * @param file
	 */
	protected void openEditor(final IFile file) {
		if (file != null) {
			getShell().getDisplay().asyncExec(new Runnable() {
				public void run() {
					try {
						IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
						IDE.openEditor(page, file, true);
					}
					catch (PartInitException e) {
						UIPlugin.log(IStatus.ERROR, e, e.getMessage());
					}
				}
			});
		}
	}

}
