/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.ui.debug;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;

import org.eclipse.atf.mozilla.ide.core.IXPCOMThreadProxyHelper;
import org.eclipse.atf.mozilla.ide.debug.INestedEventLoop;
import org.eclipse.atf.mozilla.ide.debug.MozillaDebugPlugin;
import org.eclipse.atf.mozilla.ide.debug.internal.model.JSDebugTarget;
import org.eclipse.atf.mozilla.ide.ui.browser.MozBrowserProcess;
import org.eclipse.atf.mozilla.ide.ui.browser.util.MozBrowserUtil;
import org.eclipse.atf.ui.UIPlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.internal.Workbench;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.internal.ServerPreferences;


public class MozillaDebugJob extends Job {

	protected IServer server;
	protected IModule[] module;
	protected ILaunch launch;
	protected URL url;
			
	public MozillaDebugJob(String name) {
		super(name);
	}
	
	public MozillaDebugJob(String name, IServer server, IModule[] module, ILaunch launch, URL url) {
		super(name);
		this.server = server;
		this.module = module;
		this.launch = launch;
		this.url = url;
	}
	
	protected IStatus run(IProgressMonitor monitor) {
		
		IStatus status = Status.OK_STATUS;

		if(server != null && module != null) {
			// wait for up to 5 minutes
			int state = server.getModuleState(module);
			int count = ServerPreferences.getInstance().getModuleStartTimeout();
			while (state == IServer.STATE_STARTING && count > 0) {
				if (monitor.isCanceled())
					return status;
				try {
					Thread.sleep(2000);
				} catch (Exception e) {
					// ignore
				}
				count -= 2000;
				state = server.getModuleState(module);
			}
			
			if (monitor.isCanceled())
				return status;
			
			if (state == IServer.STATE_STARTING)
				return status;
		}
		
		
		try {
			launchApp();
		} catch(Exception e) {
			status = new Status(IStatus.ERROR, 
								UIPlugin.PLUGIN_ID,
								IStatus.OK, 
								"Error launching Mozilla application in debug mode", 
								e);
		}
		
		return status;
	}
	
	Browser debugBrowser;
	protected void launchApp() throws CoreException {
		
		boolean useInternalBrowser = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.IS_INTERNAL_BROWSER, true);
		
		if(useInternalBrowser) {
			
			String appPath = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.APP_PATH, (String)null);
			String projectName = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.PROJECT, (String)null);
			IProject project = null;
			if(projectName != null && !projectName.equals(""))
				project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
						
			// XPCOM calls in the debugger will be delegated to the Display thread by this helper object
			Display display = Display.getDefault();
			IXPCOMThreadProxyHelper proxyHelper = new XPCOMThreadProxyHelper(display);
			
			proxyHelper.syncExec(new Runnable() {
				public void run() {
					
					try {					
						debugBrowser = MozBrowserUtil.openMozillaBrowser( "about:blank" );
					} catch (CoreException e) {
						e.printStackTrace();
						return;
					}
					
				}
			});
			
			// Start the debugger on the Display thread
			final JSDebugTarget debugTarget = (JSDebugTarget)MozillaDebugPlugin.launchDebugTarget(launch, url, project, appPath, new NestedEventLoop(display), proxyHelper);
			launch.addDebugTarget(debugTarget);

			// Open the application in a browser on the same Display thread
			proxyHelper.syncExec(new Runnable() {
				public void run() {
					
					System.out.println(" url to string " + url.toString());
					System.out.println(" url to toExternalForm " + url.toExternalForm());
					
					debugBrowser.setUrl( url.toString() );
					//Process adds itself to the launch
					debugTarget.setProcess(new MozBrowserProcess(debugBrowser, launch));
					
				}
			});
			
		}
		else {
			String browserExec = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.BROWSER_EXEC, "");
			String[] args = new String[]{browserExec,url.toString()};
			String[] args2 = new String[]{browserExec,"-venkman"};
			
			try {
				Process myProcess = Runtime.getRuntime().exec(args);
				String myLabel = url.toExternalForm();
				HashMap attrs = new HashMap();
				attrs.put(IProcess.ATTR_PROCESS_TYPE, getName());
				attrs.put(IProcess.ATTR_PROCESS_LABEL, getName() + " Application"); // label on the console
				DebugPlugin.newProcess(launch, myProcess, myLabel, attrs);
				Runtime.getRuntime().exec(args2);
			} catch(IOException e) {
				throw new CoreException(
						new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								IStatus.OK, "Error launching application", e));
			}
		}
	}
		
}

/**
 * Provides an event loop so that when you are on the same thread as an event
 * loop, you can block the event you're in, but continue to process other
 * scheduled system events.
 * 
 * Note on runEventLoop/stopEventLoop: These methods are meant to be called in
 * pairs. However, since they may be called by different threads, the ordering
 * constraints are a bit relaxed. Basically, for a given runEventLoop (<b>rEL</b>) /
 * stopEventLoop (<b>sEL</b>) call pair, this implementation guarantees
 * correct results even if the sEL call is called before its corresponding rEL.
 * There cannot be, however, two consecutive calls to either rEL or sEL in the
 * same pair of calls. For instance, an example of a valid sEL/rEL call sequence
 * is: <BR>
 * <BR>
 * (sEL,rEL);(rEL,sEL);(rEL,sEL);(sEL,rEL)<BR>
 * <BR>
 * An example of an invalid sEL/rEL call sequence:<BR>
 * <BR>
 * (sEL,rEL)<b>(sEL,sEL)</b><BR>
 * <b>(rEL,rEL)</b><BR>
 * <BR>
 * Both will result in an IllegalStateException being thrown.
 * 
 * @author peller
 */
class NestedEventLoop implements INestedEventLoop {
	
	private static final Object INITIAL = new Object();
	private static final Object RUN = new Object();
	private static final Object STOPPING =  new Object();
	
	private final Display _display;
	private volatile Object _state = INITIAL;
	private volatile long _result;

	public NestedEventLoop(Display display) {
		_display = display;
	}

	/**
	 * Blocks the current thread in the event loop until its corresponding
	 * {@link #stopEventLoop(long)} call is performed.
	 * 
	 * @throws SWTException
	 *             if this thread is not the UI thread.
	 * 
	 * @throws IllegalStateException
	 *             if the current state doesn't allow this method to be called
	 *             (see class docs).
	 * 
	 * @return a numerical value communicated by the thread that calls
	 *         {@link #stopEventLoop(long)}.
	 */
	public long runEventLoop() {
		
		synchronized (this) {
			if (_state == RUN) {
				// A start->start pair is illegal.
				throw new IllegalStateException("Nested event loop already running.");
			} else if (_state == STOPPING) {
				// End of stop->start call pair.
				_state = INITIAL;
				return _result;
			}
			_state = RUN;
		}
				
		while (_state == RUN && !Workbench.getInstance().isClosing()) {
			try {
				if (!_display.readAndDispatch()) {
					_display.sleep();
				}
			} catch (Throwable t) {
				t.printStackTrace();
				if (t instanceof ThreadDeath)
					throw (ThreadDeath) t;

				if (t instanceof RuntimeException)
					;

				if (t instanceof Error)
					throw (Error) t;

				//	TODO	ExceptionHandler.getInstance().handleException(t);
			}
		}
		
		synchronized(this) {
			// End of start->stop call pair.
			_state = INITIAL;
		}

		return _result;
	}

	/**
	 * Signals to the thread blocked in {@link #runEventLoop()} that it may quit
	 * the event loop.
	 * 
	 * @param rv
	 *            A numerical value that will be returned by the
	 *            {@link #stopEventLoop(long)} call that is the pair of this
	 *            call.
	 * 
	 * @throws IllegalStateException
	 *             if the current state doesn't allow this method to be called
	 *             (see class docs).
	 */
	public synchronized void stopEventLoop(long rv) {
		// A stop->stop pair is illegal.
		if (_state == STOPPING) {
			//throw new IllegalStateException("Stop already issued for this start/stop cycle.");
			return;
		}
		_state = STOPPING;
		_result = rv;
	}
}

/**
 * Defines an environment in which to execute Runnables.  Used
 * by XPCOM code to delegate proxies such that they are all executed
 * on the same thread.
 *
 * @author peller
 */
class XPCOMThreadProxyHelper implements IXPCOMThreadProxyHelper {
	private Display _display;

	public XPCOMThreadProxyHelper(Display display) {
		_display = display;
	}

	public Thread getThread() {
		return _display.getThread();
	}

	public void syncExec(Runnable runnable) {
		_display.syncExec(runnable);
	}
};
