/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.atf.ui.debug;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;

import org.eclipse.atf.mozilla.ide.ui.MozIDEUIPlugin;
import org.eclipse.atf.mozilla.ide.ui.browser.MozBrowserProcess;
import org.eclipse.atf.mozilla.ide.ui.browser.util.MozBrowserUtil;
import org.eclipse.atf.mozilla.ide.ui.perspective.MozillaPerspective;
import org.eclipse.atf.ui.UIPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.internal.ServerPreferences;


public class MozillaRunJob extends Job {

	protected IServer server;
	protected IModule[] module;
	protected ILaunch launch;
	protected URL url;
	protected IStatus status;
	
	public MozillaRunJob(String name, IServer server, IModule[] module, ILaunch launch, URL url) {
		super(name);
		this.server = server;
		this.module = module;
		this.launch = launch;
		this.url = url;
		
	}
	
	protected IStatus run(IProgressMonitor monitor) {
		
		status = Status.OK_STATUS;
			
		if(server != null && module != null) {
			// wait for up to 5 minutes
			int state = server.getModuleState(module);
			int count = ServerPreferences.getInstance().getModuleStartTimeout();
			while (state == IServer.STATE_STARTING && count > 0) {
				if (monitor.isCanceled())
					return status;
				try {
					Thread.sleep(2000);
				} catch (Exception e) {
					// ignore
				}
				count -= 2000;
				state = server.getModuleState(module);
			}
					
			if (monitor.isCanceled())
				return status;
			
			if (state == IServer.STATE_STARTING)
				return status;
		}
		
		//display client on UI thread
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				try {
					launchApp();
				} catch (Exception e) {
					e.printStackTrace();
					Status s = new Status(IStatus.ERROR, 
								UIPlugin.PLUGIN_ID,
								IStatus.OK, 
								"Error launching Mozilla application in run mode", 
								e);
					setStatus(s);
				}
			}
		});
						
		return status;
	}
	
	protected void setStatus(IStatus status) {
		this.status = status;
	}
	
	protected void launchApp() throws CoreException {
		
		boolean useInternalBrowser = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.IS_INTERNAL_BROWSER, true);
				
		if(useInternalBrowser) {
			Display.getDefault().syncExec(new Runnable() {

				public void run() {
					
					try {
						Browser browser = MozBrowserUtil.openMozillaBrowser( url.toString() );
						//Process adds itself to the launch
						new MozBrowserProcess(browser, launch);
					} catch (CoreException e) {
						e.printStackTrace();
						return;
					}
					
				}
					
			});
			
			final IWorkbenchWindow window = getActiveWorkbenchWindow();
			//Switch to Mozilla perspective
			if(window != null) {
				Display.getDefault().syncExec(new Runnable() {

					public void run() {
						try {
							MozIDEUIPlugin.getDefault().getWorkbench().showPerspective( MozillaPerspective.ID, window );
						} catch(WorkbenchException e) {
							e.printStackTrace();
						}
					}
						
				});
				
			}	
		}
		else {
			String browserExec = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.BROWSER_EXEC, "");
			String[] args = new String[]{browserExec,url.toString()};
			String[] args2 = new String[]{browserExec,"-venkman"};
			
			try {
				Process myProcess = Runtime.getRuntime().exec(args);
				String myLabel = url.toExternalForm();
				HashMap attrs = new HashMap();
				attrs.put(IProcess.ATTR_PROCESS_TYPE, getName());
				attrs.put(IProcess.ATTR_PROCESS_LABEL, getName() + " Application"); // label on the console
				DebugPlugin.newProcess(launch, myProcess, myLabel, attrs);
				Runtime.getRuntime().exec(args2);
			} catch(IOException e) {
				throw new CoreException(
						new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								IStatus.OK, "Error launching application", e));
			}
		}
	}
	
	private IWorkbenchWindow getActiveWorkbenchWindow() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if(window == null) {
			IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
			if(windows.length > 0)
				window = windows[0];
		}
		return window;
	}

}
