/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/


package org.eclipse.atf.ui.debug;

import org.eclipse.atf.ui.UIPlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.IJobChangeListener;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IModuleArtifact;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerWorkingCopy;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.core.ServerUtil;
import org.eclipse.wst.server.core.internal.IClient;
import org.eclipse.wst.server.core.internal.ILaunchableAdapter;
import org.eclipse.wst.server.core.internal.PublishServerJob;
import org.eclipse.wst.server.core.internal.RestartServerJob;
import org.eclipse.wst.server.core.internal.ServerPlugin;
import org.eclipse.wst.server.core.internal.ServerType;
import org.eclipse.wst.server.core.internal.StartServerJob;
import org.eclipse.wst.server.http.core.internal.HttpServer;
import org.eclipse.wst.server.ui.internal.Messages;
import org.eclipse.wst.server.ui.internal.Trace;
import org.eclipse.wst.server.ui.internal.wizard.ClosableWizardDialog;
import org.eclipse.wst.server.ui.internal.wizard.RunOnServerWizard;



public class ServerManager {

	public static final int WAITING_FOR_START = 1;
	public static final int STARTED = 2;
	public static final int WAITING_FOR_STOP = 3;
	public static final int WAITING_FOR_RESTART = 4;
	public static final int WAITING_FOR_PUBLISH = 5;
	public static final int CANCELLED = 6;
	
	private static ServerManager manager = null;
	//private int dialogResult;
	private IServer server = null;
	private ILaunchableAdapter launchableAdapter;
	private IModuleArtifact moduleArtifact;
	
/*	public static ServerManager getInstance() {
		if(manager == null) {
			manager = new ServerManager();
		}
		return manager;
	} */
	
	ServerManager() {
		super();
	}
	
	public static void startServer(IServer server, String mode, Job launchJob, Shell shell) {
		
		//start server if it's not already started
		// and cue the client to start
		
		int state = server.getServerState();
		String serverMode = server.getMode();
		boolean publish = true;
		PublishServerJob publishJob = null;
		//In the case of the HTTP server, don't publish on start if auto publish is disabled
		if(server.getServerType().getId().equals(HttpServer.ID)) {
			HttpServer httpServer = (HttpServer)server.loadAdapter(HttpServer.class, null);
			if( httpServer.dontPublish() ) {
				publish = false;
			}
		}
		if (state == IServer.STATE_STARTED) {
			boolean restart = false;
			if (ILaunchManager.DEBUG_MODE.equals(mode) && !ILaunchManager.DEBUG_MODE.equals(serverMode)) {
				int result = openWarningDialog(shell, Messages.dialogModeWarningDebug);
				if (result == 1)
					mode = serverMode;
				else if (result == 0)
					restart = true;
				else
					return;
			} 
			
			if(publish) {
				publishJob = new PublishServerJob(server, IServer.PUBLISH_INCREMENTAL, false);
			}
			if (restart) {
				RestartServerJob restartJob = new RestartServerJob(server, mode);
				if(publishJob != null) {
					restartJob.setNextJob(publishJob);
					publishJob.setNextJob(launchJob);
				} else
					restartJob.setNextJob(launchJob);
				restartJob.schedule();
			} else if(publishJob != null) {
				final Job launchAfterPublish = launchJob;
				publishJob.addJobChangeListener(new IJobChangeListener() {
					public void sleeping(IJobChangeEvent event) {}
					public void awake(IJobChangeEvent event) {}
					public void aboutToRun(IJobChangeEvent event) {}
					public void scheduled(IJobChangeEvent event) {}
					public void running(IJobChangeEvent event) {}
					public void done(IJobChangeEvent event) {
						launchAfterPublish.schedule();
					}
				});
				publishJob.schedule();
			} else {
				launchJob.schedule();
			}
			 
		} else if (state != IServer.STATE_STOPPING && state != IServer.STATE_STARTING) {
			
			if(publish) {
				publishJob = new PublishServerJob(server, IServer.PUBLISH_INCREMENTAL, true);
			}
			
			StartServerJob startServerJob = new StartServerJob(server, mode);
			if (((ServerType)server.getServerType()).startBeforePublish()) {
				if(publishJob != null) { 
					publishJob.setNextJob(launchJob);
					final Job finalPublish = publishJob;
					startServerJob.addJobChangeListener(new IJobChangeListener() {
						public void sleeping(IJobChangeEvent event) {}
						public void awake(IJobChangeEvent event) {}
						public void aboutToRun(IJobChangeEvent event) {}
						public void scheduled(IJobChangeEvent event) {}
						public void running(IJobChangeEvent event) {}
						public void done(IJobChangeEvent event) {
							finalPublish.schedule();
						}
					});
				} else {
					final Job finalLaunch = launchJob;
					startServerJob.addJobChangeListener(new IJobChangeListener() {
						public void sleeping(IJobChangeEvent event) {}
						public void awake(IJobChangeEvent event) {}
						public void aboutToRun(IJobChangeEvent event) {}
						public void scheduled(IJobChangeEvent event) {}
						public void running(IJobChangeEvent event) {}
						public void done(IJobChangeEvent event) {
							finalLaunch.schedule();
						}
					});
				}
				startServerJob.schedule();
			} else {
				if(publishJob != null) {
					final Job finalStart = startServerJob;
					publishJob.addJobChangeListener(new IJobChangeListener() {
						public void sleeping(IJobChangeEvent event) {}
						public void awake(IJobChangeEvent event) {}
						public void aboutToRun(IJobChangeEvent event) {}
						public void scheduled(IJobChangeEvent event) {}
						public void running(IJobChangeEvent event) {}
						public void done(IJobChangeEvent event) {
							finalStart.schedule();
						}
					});
				}
				final Job finalLaunch = launchJob;
				startServerJob.addJobChangeListener(new IJobChangeListener() {
					public void sleeping(IJobChangeEvent event) {}
					public void awake(IJobChangeEvent event) {}
					public void aboutToRun(IJobChangeEvent event) {}
					public void scheduled(IJobChangeEvent event) {}
					public void running(IJobChangeEvent event) {}
					public void done(IJobChangeEvent event) {
						finalLaunch.schedule();
					}
				});
				if(publishJob != null) {
					publishJob.schedule();
				} else {
					startServerJob.schedule();
				}
			}
		}
	}
		
	protected static int openWarningDialog(Shell shell, String message) {
		MessageDialog dialog = new MessageDialog(shell, Messages.errorDialogTitle, null,
			message,	MessageDialog.WARNING, new String[] {Messages.dialogModeWarningRestart,
			Messages.dialogModeWarningContinue, IDialogConstants.CANCEL_LABEL}, 0);
		return dialog.open();
	}
	
	public  IServer getServer(IProject project, String moduleName, String mode, 
			IProgressMonitor monitor) throws CoreException {

		
		IModuleArtifact[] moduleArtifacts = ServerPlugin.getModuleArtifacts(project);
		
		// TODO - multiple module artifacts
		moduleArtifact = moduleArtifacts[0];
		IModule module = null;
		module = moduleArtifact.getModule();
		
		if(module == null)
			return null;
		
		IWorkbenchWindow window = null;
		window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if(window == null) {
			IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
			if(windows.length != 0)
				window = windows[0];
		}
		server = ServerCore.getDefaultServer(module);
					
		if(/*server == null && */ window != null) {
			final String mMode = mode;
			final IProgressMonitor mMonitor = monitor;
			final IWorkbenchWindow mWindow = window;
			final IModule mModule = module;
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					if (server == null ) {				
					//try the full wizard
						RunOnServerWizard wizard = new RunOnServerWizard(mModule, mMode, moduleArtifact);
						ClosableWizardDialog dialog = new ClosableWizardDialog(mWindow.getShell(), wizard);
						if (dialog.open() == Window.CANCEL) {
							mMonitor.setCanceled(true);
							return;
						}
						launchableAdapter = wizard.getLaunchableAdapter();
						try {
							Platform.getJobManager().join("org.eclipse.wst.server.ui.family", null);
						} catch (Exception e) {
							Trace.trace(Trace.WARNING, "Error waiting for job", e);
						}
						server = wizard.getServer();
						boolean preferred = wizard.isPreferredServer();
						//tasksRun = true;
			
						// set preferred server if requested
						if (server != null && preferred) {
							try {
								ServerCore.setDefaultServer(mModule, server, mMonitor);
							} catch (CoreException ce) {
								String message = Messages.errorCouldNotSavePreference;
								ErrorDialog.openError(mWindow.getShell(), Messages.errorDialogTitle, message, ce.getStatus());
							}
						}
					} else {
						RunOnServerWizard wizard = new RunOnServerWizard(server, mMode, moduleArtifact);
						if (wizard.shouldAppear()) {
							WizardDialog dialog = new WizardDialog(mWindow.getShell(), wizard);
							if (dialog.open() == Window.CANCEL)
								return;
						} else
							wizard.performFinish();
						launchableAdapter = wizard.getLaunchableAdapter();
					}
				} 
			});
		}
		// ignore preference if the server doesn't support this mode.
		if (server != null && !isCompatibleWithLaunchMode(server, mode))
			server = null;
		
		if (server != null && !ServerUtil.containsModule(server, module, monitor)) {
			IServerWorkingCopy wc = server.createWorkingCopy();
			ServerUtil.modifyModules(wc, new IModule[] { module }, new IModule[0], monitor);
			wc.save(false, monitor);
		}
		
		try {
			Platform.getJobManager().join("org.eclipse.wst.server.ui.family", new NullProgressMonitor());
		} catch (Exception e) {
			throw new CoreException(
					new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
							IStatus.OK, "Error waiting for job", e));
		}
				
		return server;
	}
	
	private IServer getEligibleServer(IProject project, IModule module) {
		
		IServer[] servers = ServerCore.getServers();
		for(int i=0; i<servers.length; i++) {
			IModule[] modulesAdded = servers[i].getModules();
			for(int j=0; j<modulesAdded.length; j++) {
				if(modulesAdded[j].equals(module) &&
						modulesAdded[j].getProject().equals(project)) {
					return servers[i];
				}
			}
		}
		return null;
	}
	
	private boolean isCompatibleWithLaunchMode(IServer server, String launchMode) {
		if (server == null || launchMode == null)
			return false;

		int state = server.getServerState();
		if (state == IServer.STATE_STARTED && launchMode.equals(server.getMode()))
			return true;

		if (server.getServerType().supportsLaunchMode(launchMode))
			return true;
		return false;
	}

	public ILaunchableAdapter getLaunchableAdapter() {
		return launchableAdapter;
	}

	public IModuleArtifact getModuleArtifact() {
		return moduleArtifact;
	}
	
}
