/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.ui.runtime;

import java.util.ArrayList;

import org.eclipse.atf.runtime.IRuntime;
import org.eclipse.atf.runtime.IRuntimeContainer;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.atf.runtime.validator.IRuntimeValidator;
import org.eclipse.atf.ui.runtime.standins.RuntimeContainerStandin;
import org.eclipse.atf.ui.runtime.standins.RuntimeStandin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckable;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;

public class ToolkitSelectionField implements ICheckable, ISelectionProvider
{
	private static final String PLUGIN_ID =
		"org.eclipse.atf.ui.runtime.ToolkitSelectionField";

	private static final String ADD_TOOLKIT_TEXT = "Add New Toolkit...";

	private Composite composite;
	private CheckboxTableViewer viewer;

	private static final int WIDTH_HINT = 600;

	private InstalledToolkitsLabelProvider labelProvider;

	private boolean displayHorizontal;
	private int checkBoxStyle;

	public static final int ERROR_NO_TOOLKIT_SELECTED = -1;
	public static final String ERROR_NO_TOOLKIT_SELECTED_MSG =
		"Select at least one JavaScript toolkit to install.";

	public static final int ERROR_MULTIPLE_TOOLKITS_SELECTED = -2;
	public static final String ERROR_MULTIPLE_TOOLKITS_SELECTED_MSG =
		"Select a maximum of only one JavaScript toolkit per type/kind.";

	private static final String INVALID_TOOLKIT_MSG =
		"One or more toolkits did not pass validation.  Please open the AJAX Toolkits preferences and check that the toolkit information is correct.";


	public ToolkitSelectionField( boolean displayHorizontal, int checkBoxStyle )
	{
		this.displayHorizontal = displayHorizontal;
		this.checkBoxStyle = checkBoxStyle;
	}

	public void dispose()
	{
		composite.dispose();
	}

	public void createControl( Composite parent )
	{
		// create enclosing composite and layout
		composite = new Composite( parent, SWT.NONE );
		int numColumns = 1;
		if ( displayHorizontal )
			numColumns = 2;
		composite.setLayout( new GridLayout( numColumns, false ) );

		// create composite for table - we need this since we are using TableCoumnLayout, and it
		// can only be set on the table parent
		Composite tableComposite = new Composite( composite, SWT.NONE );
		
		// create table
		int style = checkBoxStyle | SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FULL_SELECTION;
		Table table = new Table( tableComposite, style );
		table.setLinesVisible( true );
		table.setHeaderVisible( true );

		// create table layout
		TableColumnLayout tableLayout = new TableColumnLayout();
		tableComposite.setLayout( tableLayout );
		final String[] headings = { "Name", "Kind", "Version", "Location" };

		// add NAME column
		TableColumn column = new TableColumn( table, SWT.LEFT );
		column.setText( headings[ 0 ] );
		column.setResizable( true );
		tableLayout.setColumnData( column, new ColumnWeightData( 4 ) );

		// add KIND column
		column = new TableColumn( table, SWT.LEFT );
		column.setText( headings[ 1 ] );
		column.setResizable( true );
		tableLayout.setColumnData( column, new ColumnWeightData( 4 ) );

		// add VERSION column
		column = new TableColumn( table, SWT.LEFT );
		column.setText( headings[ 2 ] );
		column.setResizable( true );
		tableLayout.setColumnData( column, new ColumnWeightData( 4 ) );

		// add LOCATION column
		column = new TableColumn( table, SWT.LEFT );
		column.setText( headings[ 3 ] );
		column.setResizable( true );
		tableLayout.setColumnData( column, new ColumnWeightData( 12 ) );

		// create viewer for existing table
		viewer = new CheckboxTableViewer( table )
		{
			public void handleSelect( SelectionEvent event )
			{
		        ToolkitSelectionField.this.handleSelect( event );
				super.handleSelect( event );
		    }
		};
		viewer.setContentProvider( new InstalledToolkitsContentProvider()
		{
			public void inputChanged( Viewer viewer, Object oldInput, Object newInput )
			{
				ToolkitSelectionField.this.inputChanged( newInput );
			}
		});
		labelProvider = new InstalledToolkitsLabelProvider();
		viewer.setLabelProvider( labelProvider );

		// add events for viewer
		viewer.addCheckStateListener( new ICheckStateListener()
		{
			public void checkStateChanged( CheckStateChangedEvent event )
			{
				ToolkitSelectionField.this.checkStateChanged( event );
			}
		} );

		// set grid layout for table
		GridData data = new GridData( SWT.FILL, SWT.FILL, true, true );
		data.heightHint = 300;
		data.widthHint = WIDTH_HINT;
		tableComposite.setLayoutData( data );
		
		// create any buttons
		createButtons( composite );
	}

	private void createInvalidToolkitsMsg()
	{
		Group group = new Group( composite, SWT.NONE );
		group.setText( "Invalid Toolkit(s)" );

		GridLayout gridLayout = new GridLayout( 2, false );
		group.setLayout( gridLayout );

		GridData data = new GridData( SWT.FILL, SWT.DEFAULT, true, false );
		data.verticalIndent = 10;
		group.setLayoutData( data );

		// invalid toolkit image
		Label label = new Label( group, SWT.WRAP );
		label.setImage( labelProvider.getInvalidToolkitImage() );

		data = new GridData( SWT.DEFAULT, SWT.DEFAULT, false, false );
		label.setLayoutData( data );

		// invalid toolkit message
		label = new Label( group, SWT.WRAP );
		label.setText( INVALID_TOOLKIT_MSG );

		data = new GridData( SWT.FILL, SWT.DEFAULT, false, false );
		data.widthHint = WIDTH_HINT;
		label.setLayoutData( data );
	}

	public void setLayoutData( GridData data )
	{
		composite.setLayoutData( data );
	}

	public void setInput( IRuntimeContainer input )
	{
		viewer.setInput( input );
	}

	/**
	 * Validates the toolkits ("runtime instances") provided by the new input.
	 * Invalid toolkits are displayed with an error icon.
	 */
	private void inputChanged( Object newInput )
	{
		if ( newInput == null )
			return;
		
		IRuntimeContainer input = (IRuntimeContainer) newInput;
		IRuntime[] toolkits = input.getRuntimes();

		setInvalidToolkits( toolkits );
	}

	private void setInvalidToolkits( IRuntime[] toolkitTypes )
	{
		final ArrayList invalidToolkits = new ArrayList();
		for ( int i = 0; i < toolkitTypes.length; i++ )
		{
			IRuntimeValidator validator = toolkitTypes[ i ].getValidator();
			if ( validator != null )
			{
				IRuntimeInstance[] instances = toolkitTypes[ i ].getRuntimeInstances();
				
				for ( int j = 0; j < instances.length; j++ )
				{
			    	//check if it is valid
		    		IStatus validationStatus = validator.validate( instances[ j ] );
		    		
					if ( !validationStatus.isOK() )
		    			invalidToolkits.add( instances[ j ] );
				}
			}
		}

		if ( invalidToolkits.isEmpty() )
		{
			labelProvider.setInvalidToolkits( null );
			return;
		}
		
		// Tell the label provider to display an error icon next to the given
		// invalid toolkits.
		labelProvider.setInvalidToolkits( invalidToolkits );

		// For tables with checkboxes, set the "disabled" property on
		// invalid toolkits.  See handleSelect().
		if ( checkBoxStyle == SWT.CHECK )
		{
			Display.getCurrent().asyncExec( new Runnable()
			{
				public void run()
				{
					Table table = viewer.getTable();
					
					for ( int i = 0; i < table.getItemCount(); i++ )
					{
						TableItem item = table.getItem( i );
						if ( invalidToolkits.contains( item.getData() ) )
							item.setData( "disabled", Boolean.TRUE );
						else
							item.setData( "disabled", Boolean.FALSE );
					}
				}
			});
		}

		// show invalid toolkits warning message
		createInvalidToolkitsMsg();
	}

	/**
	 * 'Disable' table items.  Since SWT Table does not support disabling of
	 * individual items, this code looks for the "disabled" property on an
	 * item.  If that property is set, then the item cannot be checked.
	 * 
	 * @see https://bugs.eclipse.org/bugs/show_bug.cgi?id=76509
	 */
	private void handleSelect( SelectionEvent event )
	{
		if ( event.detail == SWT.CHECK )
		{
		    TableItem item = (TableItem) event.item;
		    Object data = item.getData( "disabled" );
		    if ( data != null && data == Boolean.TRUE )
		    {
		    	item.setChecked( false );
		    }
		}
	}

	/**
	 * Handles the checkStateChanged event.  Validates the user's selections,
	 * insuring that at least one toolkit is selected and that no more than
	 * one toolkit of a given type is selected.  Multiple unique toolkits
	 * are allowed.
	 */
	protected void checkStateChanged( CheckStateChangedEvent event )
	{
		Object[] checkedElements = viewer.getCheckedElements();
		if ( checkedElements.length == 0 )
		{
			setStatus( new Status( IStatus.ERROR, PLUGIN_ID,
					ERROR_NO_TOOLKIT_SELECTED , ERROR_NO_TOOLKIT_SELECTED_MSG, null ) );
			return;
		}
		
		// Allow only one toolkit of a given type to be selected.  For example,
		// a project can have a Dojo and a Scriptaculous toolkit, but not
		// two different Dojo toolkits.
		for ( int i = 0; i < checkedElements.length - 1; i++ )
		{
			IRuntimeInstance instanceA = (IRuntimeInstance) checkedElements[ i ];
			IRuntime typeA = instanceA.getType();
			
			for ( int j = i + 1; j < checkedElements.length; j ++ )
			{
				IRuntimeInstance instanceB = (IRuntimeInstance) checkedElements[ j ];
				IRuntime typeB = instanceB.getType();
				
				if ( runtimesAreEqual( typeA, typeB ) )
				{
					ArrayList instances = new ArrayList( checkedElements.length );
					
					// Notify the label provider that these toolkit instances
					// must display an error image.
					instances.add( instanceA );
					instances.add( instanceB );
					
					// If there are more toolkit instances of this same type
					// that have been checked by the user, they should also
					// display an error image.
					for ( int k = j + 1; k < checkedElements.length; k++ )
					{
						IRuntimeInstance ri = (IRuntimeInstance) checkedElements[ k ];
						IRuntime type = ri.getType();
						if ( runtimesAreEqual( type, typeA ) )
							instances.add( ri );
					}

					// Set these toolkits to display an error image.
					labelProvider.setErrorToolkits( instances );
					
					// Update the viewer
					refreshViewer( event, checkedElements );

					// Display user error
					setStatus( new Status( IStatus.ERROR, PLUGIN_ID,
							ERROR_MULTIPLE_TOOLKITS_SELECTED,
							ERROR_MULTIPLE_TOOLKITS_SELECTED_MSG, null));
					return;
				}
			}
		}

		labelProvider.setErrorToolkits( null );
		refreshViewer( event, checkedElements );
		setStatus( new Status( IStatus.OK, PLUGIN_ID, "OK" ) );
	}

	/**
	 * Compares toolkit types ("runtimes"), whether they are represented by
	 * Runtime or RuntimeStandin.
	 * 
	 * @return <code>true</code> if the two objects represent the same toolkit
	 *         type, <code>false</code> otherwise.
	 */
	private boolean runtimesAreEqual( IRuntime typeA, IRuntime typeB )
	{
		if ( typeA instanceof RuntimeStandin )
		{
			// RuntimeStandin.equals() handles both RuntimeStandin and RuntimeInstance
			return typeA.equals( typeB );
		}

		return typeB.equals( typeA );
	}

	/**
	 * Refreshes the table view in order to display any errors.
	 */
	private void refreshViewer( CheckStateChangedEvent event, Object[] checkedElements )
	{
		Object[] elements = checkedElements;
		
		// if element was unchecked, add it to list of elements to be updated
		if ( event != null && ! event.getChecked() )
		{
			Object[] newElements = new Object[ elements.length + 1 ];
			System.arraycopy( elements, 0, newElements, 0, elements.length );
			newElements[ elements.length ] = event.getElement();

			elements = newElements;
		}
		
		viewer.update( elements, new String[] { "error-state" } );

		setInvalidToolkits( ((IRuntimeContainer) viewer.getInput()).getRuntimes() );
	}

	public void setCheckedToolkits( Object[] installedToolkits )
	{
		if ( installedToolkits != null )
		{
			viewer.setCheckedElements( installedToolkits );
		}
		
		checkStateChanged( null );
	}

	/**
	 * Users of ToolkitSelectionField may override this method in order to
	 * display an error message when no toolkit is selected.
	 */
	protected void setStatus( IStatus status )
	{
	}

	protected Control createButtons( Composite composite )
	{
		// create 'Add Toolkit' button
		Button addButton = new Button( composite, SWT.PUSH );
		addButton.setFont( composite.getFont() );
		addButton.setText( ADD_TOOLKIT_TEXT );
		addButton.addListener( SWT.Selection, new Listener()
		{
			public void handleEvent( Event event )
			{
				handleAddButton( event );
			}
		} );

		GridData buttonGrid = new GridData();
		if ( displayHorizontal )
			buttonGrid.verticalAlignment = SWT.BEGINNING;
		else
			buttonGrid.horizontalAlignment = SWT.END;
		
		GC gc = new GC( composite );
		gc.setFont( composite.getFont());
		FontMetrics fontMetrics = gc.getFontMetrics();
		gc.dispose();
		int widthHint = Dialog.convertHorizontalDLUsToPixels( fontMetrics,
				IDialogConstants.BUTTON_WIDTH );
		Point minSize = addButton.computeSize( SWT.DEFAULT, SWT.DEFAULT, true );
		buttonGrid.widthHint = Math.max( widthHint, minSize.x );

		addButton.setLayoutData( buttonGrid );

		return addButton;
	}

	protected void handleAddButton( Event event )
	{
		Shell shell = composite.getShell();
		final RuntimeContainerStandin input = (RuntimeContainerStandin) viewer
				.getInput();

		AddRuntimeInstanceDialog dialog = new AddRuntimeInstanceDialog( shell,
				input, null );
		dialog.setTitle( "New AJAX Toolkit" );

		if ( dialog.open() != Window.OK )
		{
			return;
		}

		// save new addition to prefs
		BusyIndicator.showWhile( null, new Runnable()
		{
			public void run()
			{
				input.commitChanges();
			}
		} );

		refresh();
	}

	public Object[] getCheckedElements()
	{
		if ( checkBoxStyle == SWT.CHECK )
			return viewer.getCheckedElements();

		return null;
	}

	public void refresh()
	{
		refresh( null );
	}

	public void refresh( Object element )
	{
		viewer.refresh( element );

		// List of toolkits may have changed (a toolkit could have been added).
		// Recheck for invalid toolkits.
		// XXX This is not good for performance (although most users won't notice).
		//     We should only revalidate newly added toolkits.
		setInvalidToolkits( ((IRuntimeContainer) viewer.getInput()).getRuntimes() );
	}

	/* ICheckable */
	
	public void addCheckStateListener( ICheckStateListener listener )
	{
		viewer.addCheckStateListener( listener );
	}

	public boolean getChecked( Object element )
	{
		return viewer.getChecked( element );
	}

	public void removeCheckStateListener( ICheckStateListener listener )
	{
		viewer.removeCheckStateListener( listener );
	}

	public boolean setChecked( Object element, boolean state )
	{
		return viewer.setChecked( element, state );
	}

	/* ISelectionProvider */
	
	public void addSelectionChangedListener( ISelectionChangedListener listener )
	{
		viewer.addSelectionChangedListener( listener );
	}

	public ISelection getSelection()
	{
		return viewer.getSelection();
	}

	public void removeSelectionChangedListener( ISelectionChangedListener listener )
	{
		viewer.removeSelectionChangedListener( listener );
	}

	public void setSelection( ISelection selection )
	{
		viewer.setSelection( selection );
	}

}
