package org.eclipse.atf.project;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.atf.runtime.IRuntime;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.atf.runtime.RuntimeInstance;
import org.eclipse.atf.runtime.RuntimeManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.helpers.DefaultHandler;

public class ProjectUtils
{
	private static final String EL_TOOLKIT = "toolkit"; //$NON-NLS-1$
	private static final String ATTR_TYPE = "type"; //$NON-NLS-1$
	private static final String ATTR_VERSION = "version"; //$NON-NLS-1$
	private static final String ATTR_NAME = "name"; //$NON-NLS-1$

	private static final String SETTINGS_FILE =
		".settings/org.eclipse.atf.settings.xml"; //$NON-NLS-1$ //$NON-NLS-2$

	public static void saveSettings( IProject project,
			Object[] runtimeInstances ) throws CoreException
	{
		IFile file = project.getFile( SETTINGS_FILE );

		final StringWriter w = new StringWriter();
		final PrintWriter out = new PrintWriter( w );

		final String nl = System.getProperty( "line.separator" ); //$NON-NLS-1$

		out.print( "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" ); //$NON-NLS-1$
		out.print( nl );
		out.print( "<atf-settings>" ); //$NON-NLS-1$
		out.print( nl );

		for ( int i = 0; i < runtimeInstances.length; i++ )
		{
			IRuntimeInstance ri = (IRuntimeInstance) runtimeInstances[ i ];

			out.print( "  <toolkit type=\"" ); //$NON-NLS-1$
			out.print( ri.getType().getId() );
			out.print( "\" version=\"" ); //$NON-NLS-1$
			out.print( ri.getVersion() );
			out.print( "\" name=\"" ); //$NON-NLS-1$
			out.print( ri.getName() );
			out.print( "\"/>" ); //$NON-NLS-1$
			out.print( nl );
		}

		out.print( "</atf-settings>" ); //$NON-NLS-1$
		out.print( nl );

		final byte[] bytes;

		try
		{
			bytes = w.getBuffer().toString().getBytes( "UTF-8" ); //$NON-NLS-1$
		}
		catch ( UnsupportedEncodingException e )
		{
			// Unexpected. All JVMs are supposed to support UTF-8.
			throw new RuntimeException( e );
		}

		final InputStream in = new ByteArrayInputStream( bytes );

		if ( file.exists() )
		{
			validateEdit( new IFile[] { file } );
			file.setContents( in, true, false, null );
		}
		else
		{
			final IFolder parent = (IFolder) file.getParent();

			if ( !parent.exists() )
			{
				parent.create( true, true, null );
			}

			file.create( in, true, null );
		}
	}

	private static void validateEdit( final IFile[] files )
			throws CoreException
	{
		final IWorkspace ws = ResourcesPlugin.getWorkspace();
		final IStatus st = ws.validateEdit( files, IWorkspace.VALIDATE_PROMPT );

		if ( st.getSeverity() == IStatus.ERROR )
		{
			throw new CoreException( st );
		}
	}

	public static Object[] loadSettings( IProject project )
			throws CoreException
	{
		IFile file = project.getFile( SETTINGS_FILE );

		// make sure the settings file exists
		if ( !file.exists() )
		{
			throw new CoreException( new Status( IStatus.ERROR,
					CorePlugin.PLUGIN_ID, IStatus.ERROR,
					"No settings found for project.", null ) );
		}

		// parse the settings file
		Element root = null;

		try
		{
			DocumentBuilder parser = DocumentBuilderFactory.newInstance()
					.newDocumentBuilder();
			parser.setErrorHandler( new DefaultHandler() );
			File f = file.getLocation().toFile();
			root = parser.parse( f ).getDocumentElement();
		}
		catch ( Exception e )
		{
			throw new CoreException( new Status( IStatus.ERROR,
					CorePlugin.PLUGIN_ID, IStatus.ERROR,
					"Could not parse settings.", e ) );
		}

		// iterate over element children, looking for "toolkit" node
		final NodeList nl = root.getChildNodes();
		ArrayList toolkits = new ArrayList();

		for ( int i = 0, n = nl.getLength(); i < n; i++ )
		{
			final Node node = nl.item( i );

			if ( node.getNodeType() == Node.ELEMENT_NODE )
			{
				final Element e = (Element) node;
				final String name = e.getNodeName();

				if ( name.equals( EL_TOOLKIT ) )
				{
					String type = e.getAttribute( ATTR_TYPE );
					IRuntime r = RuntimeManager.getInstance().getRuntime( type );

					RuntimeInstance ri = new RuntimeInstance();
					ri.setType( r );
					ri.setVersion( e.getAttribute( ATTR_VERSION ) );
					ri.setName( e.getAttribute( ATTR_NAME ) );

					toolkits.add( ri );
				}
			}
		}

		if ( ! toolkits.isEmpty() )
			return toolkits.toArray();
		
		throw new CoreException( new Status( IStatus.ERROR,
				CorePlugin.PLUGIN_ID, IStatus.ERROR,
				"No settings found for project.", null ) );
	}

}
