/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/

package org.eclipse.atf.resources;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;


public abstract class ResourceDeltaVisitor implements IResourceDeltaVisitor {

	protected int resourceChangeType;
		
	public ResourceDeltaVisitor() {
		super();
	}
	
	public void setChangeType(int resourceChangeType) {
		this.resourceChangeType = resourceChangeType;
	}
	
	public boolean visit(IResourceDelta subdelta) throws CoreException {
		IResource resource = subdelta.getResource();
		int kind = subdelta.getKind();
		
		switch (kind) {
			case IResourceDelta.ADDED :
			{
				try {
					handleResourceAdded(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.ADDED_PHANTOM :
			{
				try {
					handleResourcePhantomAdded(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.CHANGED :
			{
				try {
					handleResourceChanged(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.MOVED_FROM :
			{
				try {
					handleResourceMovedFrom(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.MOVED_TO :
			{
				try {
					handleResourceMovedTo(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.OPEN :
			{
				try {
					handleResourceOpen(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.REMOVED :
			{
				try {
					handleResourceRemoved(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.REMOVED_PHANTOM :
			{
				try {
					handleResourcePhantomRemoved(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			case IResourceDelta.REPLACED :
			{
				try {
					handleResourceReplaced(resource);
				}
				catch(Exception e) {
					handleException(e, "Error handling resource change");
				}
				return true;
			}
			
			
			
		}
					
		return true;
	
	}
	
	public abstract void handleResourceAdded(IResource resource) throws Exception;
	
	public abstract void handleResourcePhantomAdded(IResource resource) throws Exception;
	
	public abstract void handleResourceChanged(IResource resource) throws Exception;
	
	public abstract void handleResourceMovedFrom(IResource resource) throws Exception;
	
	public abstract void handleResourceMovedTo(IResource resource) throws Exception;
	
	public abstract void handleResourceOpen(IResource resource) throws Exception;
	
	public abstract void handleResourceRemoved(IResource resource) throws Exception;
	
	public abstract void handleResourcePhantomRemoved(IResource resource) throws Exception;
	
	public abstract void handleResourceReplaced(IResource resource) throws Exception;
	
	protected void handleException(Exception e, String message) throws CoreException {
		throw new CoreException(
				new Status(IStatus.ERROR, CorePlugin.PLUGIN_ID, 
						IStatus.OK, message, e));
	}
		

}
