package org.eclipse.atf.runtime;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.atf.core.CorePlugin;
import org.eclipse.atf.natures.ATFNature;
import org.eclipse.atf.runtime.installer.IRuntimeInstaller;
import org.eclipse.atf.runtime.validator.AlwaysValidRuntimeValidator;
import org.eclipse.atf.runtime.validator.IRuntimeValidator;
import org.eclipse.atf.runtime.version.IVersionFinder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Reads the Extensions to detect all the different supported Runtimes and
 * available instances.
 * 
 * Persists changes made by end-user through UI.
 */
public class RuntimeManager implements IRuntimeContainer, IRuntimeInstanceContainer {
	
	protected static final String RUNTIME_PLUGIN_POINT = "runtime";
	protected static final String RUNTIME_INSTANCE_PLUGIN_POINT = "runtimeInstance";
	
	protected static final String USER_RUNTIME_PREF = "AJAX_RUNTIME_USER_SETTINGS";
		
	protected static final String ATTR_ID = "id";
	protected static final String ATTR_NAME = "name";
	protected static final String ATTR_TYPE = "type";
	protected static final String ATTR_VERSION = "version";
	protected static final String ATTR_LOCATION = "location";
	protected static final String ATTR_ALLOWUSERINSTANCES = "allowUserInstances";
	
	protected static final String TAG_RUNTIME = "runtime";
	protected static final String TAG_RUNTIME_INSTANCE = "runtime-instance";
	
	protected static final String TAG_HANDLER = "handler";
	protected static final String TAG_INSTALLER = "installer";
	protected static final String TAG_VALIDATOR = "validator";
	protected static final String TAG_VERSION = "version";
	protected static final String ATTR_CLASS = "class";
	
	
	/*
	 * Map of all the supported Runtimes keyed by id
	 */
	protected HashMap runtimeMap = new HashMap();
	
	private static RuntimeManager instance = null;
	
	public static RuntimeManager getInstance(){
		if( instance == null )
			instance = new RuntimeManager();
		
		return instance;
	}
	
	private RuntimeManager(){
		
		loadSupportedRuntimes();
		loadInstancesFromExtension();
		loadEndUserConfigurations();
	} 
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntimeContainer#getRuntimes()
	 */
	public IRuntime[] getRuntimes(){
		return (IRuntime [])runtimeMap.values().toArray(new IRuntime[0]);
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntimeContainer#getRuntime(java.lang.String)
	 */
	public IRuntime getRuntime( String type ){
		return (IRuntime)runtimeMap.get( type );
	}
	
	/*
	 * Add the IRuntime to the map only if it is a new id.  
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntimeContainer#addRuntime(org.eclipse.atf.runtime.IRuntime)
	 */
	public void addRuntime( IRuntime runtime ) {
		if( !runtimeMap.containsKey(runtime.getId()) ){
			runtimeMap.put( runtime.getId(), runtime );
		}		
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntimeContainer#removeRuntime(org.eclipse.atf.runtime.IRuntime)
	 */
	public void removeRuntime(IRuntime runtime) {
		runtimeMap.remove( runtime.getId() );
	}

	/*
	 * Groups all the instances into a single array
	 * 
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntimeInstanceContainer#getRuntimeInstances()
	 */
	public IRuntimeInstance[] getRuntimeInstances() {
		ArrayList instanceList = new ArrayList();
		
		IRuntime [] runtimes = getRuntimes();
		for (int i = 0; i < runtimes.length; i++) {
			
			IRuntimeInstance [] instances = runtimes[i].getRuntimeInstances();
			for (int j = 0; j < instances.length; j++) {
				instanceList.add( instances[j] );
			}
		
		}
		
		return (IRuntimeInstance []) instanceList.toArray( new IRuntimeInstaller[0] );
	}

	public IRuntimeInstance[] getRuntimeInstancesOfType( IRuntime runtime ) {
		
		return getRuntimeInstancesOfType( runtime.getId() );
	}

	public IRuntimeInstance[] getRuntimeInstancesOfType( String type ) {
		
		if( runtimeMap.containsKey(type) ){
			IRuntime runtime = (IRuntime)runtimeMap.get( type );
			return runtime.getRuntimeInstances();
		}
		
		return new IRuntimeInstance[0];
	}
	
	public void addRuntimeInstance( IRuntimeInstance instance, String type ) {
		Runtime r = (Runtime)getRuntime( type );
		
		//There could be a runtime instance in the extensions that don't have a Runtime
		if( r != null )
			r.addInstance( (RuntimeInstance)instance );
	}

	public void removeRuntimeInstance( IRuntimeInstance instance ) {
		((Runtime)instance.getType()).removeInstance( (RuntimeInstance)instance );	
	}	
	
	/*
	 * Load all the Runtime definition
	 */
	protected void loadSupportedRuntimes(){
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint(CorePlugin.PLUGIN_ID, 
				RUNTIME_PLUGIN_POINT);
		
		if (point == null)
			return;
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements =
				extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				if (elements[j].getName().equals(TAG_RUNTIME)) {
					
					Runtime r = new Runtime( elements[j].getAttribute(ATTR_ID), elements[j].getAttribute(ATTR_NAME) );
					
					//check allowUserInstances
					String allowUserInstancesValue = elements[j].getAttribute( ATTR_ALLOWUSERINSTANCES);
					if( allowUserInstancesValue != null ){
						r.setAllowUserInstances( Boolean.valueOf(allowUserInstancesValue).booleanValue() );
					}
					
					//get the installer
					//@GINO: Change to do lazy loading
					try{
						if( elements[j].getChildren(TAG_INSTALLER).length > 0 ){
							IRuntimeInstaller installer = (IRuntimeInstaller) elements[j].createExecutableExtension(TAG_INSTALLER);
							r.setInstaller(installer);
						}
					}
					catch( CoreException ce ){
						/*
						 * It is crucial that the Runtime has an installer. There are errors if there are not installers set so it
						 * won't be added to the list of Runtimes
						 */
						CorePlugin.log( ce, "No installer defined for runtime. Will not add to configuration" );
						continue;
					}
					
					//get the validator
					//@GINO: Change to do lazy loading
					try{
						if( elements[j].getChildren(TAG_VALIDATOR).length > 0 ){
							IRuntimeValidator validator = (IRuntimeValidator) elements[j].createExecutableExtension(TAG_VALIDATOR);
							r.setValidator(validator);
						}
						else{
							//setting a default validator that always return true
							r.setValidator( new AlwaysValidRuntimeValidator() );
						}
					}
					catch( CoreException ce ){
						CorePlugin.log( ce, "No validator defined for runtime." );
					}
					
					//get the version finder
					//@GINO: Change to do lazy loading
					try{
						if( elements[j].getChildren(TAG_VERSION).length > 0 ){
							IVersionFinder versionFinder = (IVersionFinder) elements[j].createExecutableExtension(TAG_VERSION);
							r.setVersionFinder(versionFinder);
						}
					}
					catch( CoreException ce ){
						CorePlugin.log( ce, "No version finder defined for runtime." );
					}
					
					addRuntime( r );
				}
			}
		}
		
	}
	
	/*
	 * Loads from the extension registry instances of Runtimes.
	 */
	protected void loadInstancesFromExtension(){
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint(CorePlugin.PLUGIN_ID, 
				RUNTIME_INSTANCE_PLUGIN_POINT);
		
		if (point == null)
			return;
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements =
				extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				if (elements[j].getName().equals(TAG_RUNTIME)) {
					
					RuntimeInstance ri = new RuntimeInstance();
					ri.setName( elements[j].getAttribute(ATTR_NAME));
					ri.setLocation( elements[j].getAttribute(ATTR_LOCATION));
					ri.setVersion( elements[j].getAttribute(ATTR_VERSION));
					
					
					//matches to the id of Runtime
					String type = elements[j].getAttribute(ATTR_TYPE);
					
					addRuntimeInstance( ri, type );
				}
			}
		}
	}
	
	/*
	 * Reads the XML stored in the RUNTIME_PREF to retrieve user defined Runtime and RuntimeInstance
	 *
	 * FORMAT:
	 * 
	 * <runtime-settings>
	 * 	 <runtime name="" id="" />
	 *   <runtime-instance name="" type="" location="" version=""/>
	 * </runtime-settings>
	 */
	protected void loadEndUserConfigurations(){
		
		if( CorePlugin.getDefault().getPluginPreferences().contains(USER_RUNTIME_PREF) ){
			
			String userSettings = CorePlugin.getDefault().getPluginPreferences().getString( USER_RUNTIME_PREF );
				
			Element config= null;	
			
			InputStream stream = null;
			try {
				stream = new BufferedInputStream( new ByteArrayInputStream(userSettings.getBytes("UTF8")) );
				DocumentBuilder parser= DocumentBuilderFactory.newInstance().newDocumentBuilder();
				parser.setErrorHandler(new DefaultHandler());
				config = parser.parse(new InputSource(stream)).getDocumentElement();
			
			} catch ( Exception e) {
				if( stream != null ){
					try {
						stream.close();
					} catch (IOException e1) {
						e1.printStackTrace();
					}
				}
			} finally {
				if( stream != null ){
					try {
						stream.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
			}
			
			//If the top-level node wasn't what we expected, bail out
			if ( config == null || !config.getNodeName().equalsIgnoreCase("runtime-settings")) {
				return; //@GINO: error condition
			}
			
			//recreate all runtimes defined by end-user
			NodeList userRuntimeConfigList = config.getElementsByTagName( TAG_RUNTIME );
			for( int i=0; i<userRuntimeConfigList.getLength(); i++ ){
				
				Element runtimeElt = (Element)userRuntimeConfigList.item(i);
				
				//creates the instance and adds it to the Runtime type
				Runtime r = new Runtime( runtimeElt.getAttribute(ATTR_ID), runtimeElt.getAttribute(ATTR_NAME) );		
				r.setIsUser( true );
				
				addRuntime( r );
			}
			
			
			//look for all instances defined by user
			NodeList userInstanceConfigList = config.getElementsByTagName( TAG_RUNTIME_INSTANCE );
			for( int i=0; i<userInstanceConfigList.getLength(); i++ ){
				
				Element runtimeInstanceElt = (Element)userInstanceConfigList.item(i);
				
				//creates the instance and adds it to the Runtime type
				RuntimeInstance ri = new RuntimeInstance();
				ri.setName( runtimeInstanceElt.getAttribute(ATTR_NAME));
				ri.setLocation( runtimeInstanceElt.getAttribute(ATTR_LOCATION));
				ri.setVersion( runtimeInstanceElt.getAttribute(ATTR_VERSION));
				
				//matches to the id of Runtime
				String type = runtimeInstanceElt.getAttribute(ATTR_TYPE);
				
				//set as user defined
				ri.setIsUser( true );
				
				addRuntimeInstance( ri, type );
				
			}
			
		}
		
	}

	/*
	 * Serializes user defined RuntimeInstance and changes to the default settings into
	 * a an XML string that can be stored in a Pref.
	 * 
	 * FORMAT:
	 * 
	 * <runtime-settings>
	 *	 <runtime name="" id="" />
	 *   <runtime-instance name="" type="" location="" version=""/>
	 * </runtime-settings>
	 */
	protected String serializeSettings() {
		StringBuffer buffer = new StringBuffer();
		buffer.append( "<runtime-settings>" );
		
		//serialize default settings
		IRuntime [] runtimeIter = getRuntimes();
		for (int i = 0; i < runtimeIter.length; i++) {
			IRuntime r = runtimeIter[i];
			
			//serialize user runtimes
			if( r.isUser() ){
				buffer.append( serializeRuntime(r) );	
			}
			
			//serialize user instances
			IRuntimeInstance[] instanceArray = r.getRuntimeInstances();
			for (int j = 0; j < instanceArray.length; j++) {
				IRuntimeInstance ri = instanceArray[j];
				if( ri.isUser() ){
					buffer.append( serializeRuntimeInstance(ri) );					
				}				
			}
		}		
		
		buffer.append( "</runtime-settings>" );
		return buffer.toString();
	}
	
	/**
	 * This method is used to serialized an IRuntime into XML for
	 * persistence. This is necessary for IRuntime that are configured
	 * by the end-user and are not loaded from the extension point.
	 * 
	 * <runtime name="" id="" />
	 * 
	 * @param r IRuntime to serialize
	 * @return An XML string representing the IRuntime
	 */
	public String serializeRuntime( IRuntime r ){
		StringBuffer buffer = new StringBuffer();
		
		buffer.append( "<" );
		
		buffer.append( TAG_RUNTIME );
		
		buffer.append( " " );
		
		buffer.append( ATTR_NAME );
		
		buffer.append("=\"");
		
		buffer.append( r.getName() );
		
		buffer.append( "\" ");
		
		buffer.append( ATTR_ID );
		
		buffer.append("=\"");
		
		buffer.append( r.getId() );
		
		buffer.append( "\" />");
		
		return buffer.toString();
	}
	
	/**
	 * This method is used to serialized an IRuntimeInstance into XML for
	 * persistence. This is necessary for IRuntimeInstances that are configured
	 * by the end-user and are not loaded from the extension point.
	 * 
	 * <runtime-instance name="" type="" location="" version=""/>
	 * 
	 * @param ri IRuntimeInstance to serialize
	 * @return An XML string representing the IRuntimeInstance
	 */
	public String serializeRuntimeInstance( IRuntimeInstance ri ){
		StringBuffer buffer = new StringBuffer();
		
		buffer.append( "<" );
		
		buffer.append( TAG_RUNTIME_INSTANCE );
		
		buffer.append( " " );
		
		buffer.append( ATTR_NAME );
		
		buffer.append("=\"");
		
		buffer.append( ri.getName() );
		
		buffer.append( "\" ");
		
		buffer.append( ATTR_TYPE );
		
		buffer.append("=\"");
		
		buffer.append( ri.getType().getId() );
		
		buffer.append( "\" ");
		
		buffer.append( ATTR_LOCATION );
		
		buffer.append("=\"");
		
		buffer.append( ri.getLocation() );
		
		buffer.append( "\" ");
		
		buffer.append( ATTR_VERSION );
		
		buffer.append("=\"");
		
		buffer.append( ri.getVersion() );
		
		buffer.append( "\" />");
		
		return buffer.toString();
	}
	
	/*
	 * Serialized the user settings and persist them in a Pref
	 * 
	 * Call should be wrapped in a Runnable to prevent blocking UI.
	 */
	public void saveSettings(){
		String serializedUserPrefs = serializeSettings();
		CorePlugin.getDefault().getPluginPreferences().setValue( USER_RUNTIME_PREF, serializedUserPrefs );
		CorePlugin.getDefault().savePluginPreferences();
	}
	
	/*
	 * Runtime install
	 * 
	 * install the runtime and store the settings using the nature
	 */
	public void installRuntimes( final Object[] runtimeInstances, final IProject project, IProgressMonitor monitor ) throws CoreException
	{
		monitor.beginTask( "Installing runtime instances to project " + project.getName() + "...", 100 );
		
		try
		{
			/*
			 * Batching resource changes to prevent unnecessary auto-builds
			 */
			project.getWorkspace().run( new IWorkspaceRunnable()
			{
				public void run( IProgressMonitor monitor ) throws CoreException
				{
					int individualWork = 100 / runtimeInstances.length;

					// install each runtime instance
					for ( int i = 0; i < runtimeInstances.length; i++ )
					{
						IRuntimeInstance ri = (IRuntimeInstance) runtimeInstances[ i ];
						IRuntime runtime = ri.getType();
						IRuntimeInstaller installer = runtime.getInstaller();
						
						SubProgressMonitor sub = new SubProgressMonitor( monitor, individualWork );
						sub.setTaskName( "Installing " + runtime.getName() + " runtime assets..." );
						
						installer.install( ri, project, sub );
					}
				}
			}, monitor);
		}
		finally{
			monitor.done();
		}
	}
	
	/*
	 * Runtime uninstall
	 * 
	 * uninstall the runtime and remove the settings using the nature
	 */
	public void uninstallRuntimes( final Object[] runtimeInstances, final IProject project, IProgressMonitor monitor ) throws CoreException
	{
		monitor.beginTask( "Uninstalling runtime instance from project " + project.getName() + "...", 100 );
		
		//check if the nature is available (if not we assume that it is already uninstalled and do nothing
		if (!project.hasNature(ATFNature.ATF_NATURE_ID))
		{
			monitor.done();
			throw new CoreException( new Status(
					IStatus.ERROR,
					CorePlugin.PLUGIN_ID,
					IStatus.OK,
					"Error uninstalling runtime! Nature not found. Without the nature, the runtime cannot be unistalled.", null) );
		}
		
		try
		{
			/*
			 * Batching resource changes to prevent unnecessary auto-builds
			 */
			project.getWorkspace().run( new IWorkspaceRunnable()
			{
				public void run( IProgressMonitor monitor ) throws CoreException
				{
					int individualWork = 100 / runtimeInstances.length;
					
					// install each runtime instance
					for ( int i = 0; i < runtimeInstances.length; i++ )
					{
						IRuntimeInstance ri = (IRuntimeInstance) runtimeInstances[ i ];
						IRuntime runtime = ri.getType();
						IRuntimeInstaller installer = runtime.getInstaller();
						
						SubProgressMonitor sub = new SubProgressMonitor( monitor, individualWork );
						sub.setTaskName( "Uninstalling " + runtime.getName() + " runtime assets..." );
						
						installer.uninstall( ri, project, sub );
					}
				}
			}, monitor );
		}
		finally{
			monitor.done();
		}
	}
}
