/*******************************************************************************
 * Copyright (c) 2006 nexB Inc and EasyEclipse.org. All rights reserved. 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html.
 *
 * Contributors:
 *     nexB Inc and EasyEclipse.org - initial API and implementation
 *******************************************************************************/
package org.eclipse.atf.mozilla.ide.core;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.osgi.framework.Bundle;

public class XulRunnerExtensionReader {

	public static final String EXTENSION_POINT_ID = "xulrunner"; //$NON-NLS-1$

	public static final String XULRUNNER_ELEMENT = "xulrunner"; //$NON-NLS-1$

	public static final String XULRUNNER_PATH_ATTRIBUTE = "path"; //$NON-NLS-1$

	public static final String XULRUNNER_VERSION_ATTRIBUTE = "version"; //$NON-NLS-1$

	public XulRunnerExtensionReader() {
		super();
	}

	public Map getXulRunners() {
		IExtensionPoint point = getExtensionPoint();
		if (point == null) {
			MozideCorePlugin.logInfo("No plugin contributing a private Xulrunner. " //$NON-NLS-1$
					+ "Using the default platform registered Xulrunner."); //$NON-NLS-1$
			return null;
		}

		Map xulRunnerRoots = new HashMap();
		IConfigurationElement[] configElements = point.getConfigurationElements();
		for (int i = 0; i < configElements.length; i++) {
			IConfigurationElement element = configElements[i];
			String pluginId = element.getContributor().getName();
			String xulRunner = configElements[i].getName();
			if (!XULRUNNER_ELEMENT.equals(xulRunner)) {
				continue;
			}

			String path = element.getAttribute(XULRUNNER_PATH_ATTRIBUTE);
			if (path == null) {
				MozideCorePlugin.logError("Invalid extension contributed by:" + pluginId //$NON-NLS-1$
						+ " Missing or empty path attribute in extension: " + MozideCorePlugin.PLUGIN_ID + "." //$NON-NLS-1$ //$NON-NLS-2$
						+ EXTENSION_POINT_ID);
				continue;
			}
			String version = element.getAttribute(XULRUNNER_VERSION_ATTRIBUTE);
			if (version == null) {
				MozideCorePlugin.logError("Invalid extension contributed by:" + pluginId //$NON-NLS-1$
						+ " Missing or empty version attribute in extension: " + MozideCorePlugin.PLUGIN_ID + "." //$NON-NLS-1$ //$NON-NLS-2$
						+ EXTENSION_POINT_ID);
				continue;
			}

			File xulRunnerRootDir = getXulRunnerRootDir(pluginId, path);

			if (xulRunnerRootDir != null && version.length() > 0) {
				xulRunnerRoots.put(xulRunnerRootDir, version);
			} else {
				MozideCorePlugin
						.logError("Invalid extension contributed by:" + pluginId //$NON-NLS-1$
								+ " Missing, empty or invalid version or path attribute in extension: " + MozideCorePlugin.PLUGIN_ID + "." //$NON-NLS-1$ //$NON-NLS-2$
								+ EXTENSION_POINT_ID);
			}
		}
		return xulRunnerRoots;
	}

	private IExtensionPoint getExtensionPoint() {
		IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();
		IExtensionPoint point = extensionRegistry.getExtensionPoint(MozideCorePlugin.PLUGIN_ID, EXTENSION_POINT_ID);
		return point;
	}

	/**
	 * Returns a java File pointing to a directory inside a plugin installation
	 * location
	 * 
	 * @param pluginId
	 * @param pluginRelativePath
	 * @return a java File pointing to a xulrunner root directory
	 */
	File getXulRunnerRootDir(String pluginId, String pluginRelativePath) {
		Bundle bundle = Platform.getBundle(pluginId);
		if (bundle == null) {
			return null;
		}
		try {
			URL filePath = bundle.getEntry(pluginRelativePath);
			if (filePath == null) {
				return null;
			}
			URL localPath = FileLocator.resolve(filePath);
			File xulRunnerRoot = new File(FileLocator.toFileURL(localPath).getFile());
			if (xulRunnerRoot.isDirectory()) {
				return xulRunnerRoot;
			}
		} catch (IOException e) {
			String message = "Invalid extension contributed by:" + pluginId //$NON-NLS-1$
					+ " Invalid extension location contribution in extension: " + MozideCorePlugin.PLUGIN_ID + "." //$NON-NLS-1$ //$NON-NLS-2$
					+ EXTENSION_POINT_ID + "with location: " + pluginRelativePath + " : " + e.getMessage(); //$NON-NLS-1$ //$NON-NLS-2$;

			Status status = new Status(IStatus.ERROR, pluginId, IStatus.ERROR, message, e);
			MozideCorePlugin.logStatus(status);
		}
		return null;
	}
}
