package org.eclipse.atf.mozilla.ide.core.util;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;

import org.eclipse.atf.adapter.IWebResourceLocator;
import org.eclipse.atf.mozilla.ide.core.MozideCorePlugin;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.core.runtime.Status;

public class SourceLocatorUtil {

	protected static SourceLocatorUtil instance = null;
	public static SourceLocatorUtil getInstance(){
		if( instance == null )
			instance = new SourceLocatorUtil();
		
		return instance;
	}
	
	protected SourceLocatorUtil(){}
	
	/**
	 * Returns the element that represented by the URL. It tries to match it to a local resource and if
	 * not found, then it returns a reference to a remote artifact.
	 * 
	 * @param sourceURL the ulr to the resource (no host and port)
	 * @param basePathHint (optional) portion of the ulr's path that maps to the root of the project
	 * @param projectHint (optional) project where the resources should be available
	 */
	public Object getSourceElement( URL sourceURL, String basePathHint, IProject projectHint ){
		//first try to find it locally
		Object source = findLocalResource( sourceURL, basePathHint, projectHint );
		
		//try to get as remote resource
		if( source == null ){
			source = findRemoteResource( sourceURL );
		}
		
		return source;
		
	}
	
	public Object getSourceElement( URL sourceURL ){
		return getSourceElement( sourceURL, null, null );		
	}
	
	/**
	 * This method tries to resolve the URL to a local resource. Returns
	 * null if not found.
	 * 
	 * @param url the ulr to the resource (no host and port)
	 * @param basePathHint (optional) portion of the ulr's path that maps to the root of the project
	 * @param projectHint (optional) project where the resources should be available
	 */
	public IResource findLocalResource( URL url, String basePathHint, IProject projectHint ){
		IResource res = null;
		String decodedString;
		try {
			decodedString = URLDecoder.decode(url.toString(), "UTF-8");
		} catch (UnsupportedEncodingException e) {
			decodedString = url.toString();
		}
		Path urlAsPath = new Path(decodedString);
		//if the basePathHint is not null and contained in the urlPath, convert to relative
		if( basePathHint != null && !"".equals(basePathHint) ){
			
			Path basePath = new Path( basePathHint );		
			
			//contruct the relative path by removing the base
			//assumming that basepath maps to project's root
			if( basePath.isPrefixOf(urlAsPath) ){
				IPath relativePathToResource = urlAsPath.removeFirstSegments( basePath.segmentCount() );
				
				if( projectHint != null ){
					
					//find the resource in the project
					res = getResource( projectHint, relativePathToResource );
					
				}
				else{
					//assume basePathHint last segment contains the name of the project.
					String possibleProjectName = basePath.lastSegment();
					IProject possibleProject = ResourcesPlugin.getWorkspace().getRoot().getProject(possibleProjectName);
					
					if( possibleProject != null && possibleProject.exists()){
						
						//find the resource in the project
						res = getResource( possibleProject, relativePathToResource );
						
					}
				}
			}
			
			if( res != null ){
				return res;
			}
		}
		
		//at this point the basePathHint did not work, try finding using the name of the project as the basePath
		if( projectHint != null ){
			Path basePath = new Path( projectHint.getName() );
			
			//contruct the relative path by removing the base
			//assumming that basepath maps to project's root
			if( basePath.isPrefixOf(urlAsPath) ){
				IPath relativePathToResource = urlAsPath.removeFirstSegments( basePath.segmentCount() );
						
				//find the resource in the project
				res = getResource( projectHint, relativePathToResource );
				
			}
			//else{} maybe do an else here to search in the project with the entire url
			
			if( res != null ){
				return res;
			}
		}
		
		//if there are no hints, then assume the first segment of the path is the project name and the rest is the path to resource
		IPath pathPortionOfURL = new Path( url.getPath() );
		String possibleProjectName = pathPortionOfURL.segment(0);
		if( possibleProjectName != null ){
			IProject possibleProject = ResourcesPlugin.getWorkspace().getRoot().getProject(possibleProjectName);
			
			if( possibleProject != null && possibleProject.exists()){
				IPath relativePathToResource = pathPortionOfURL.removeFirstSegments( 1 ); //remove only the first segment
				
				//find the resource in the project
				res = getResource( possibleProject, relativePathToResource );
			}
		}
		
		return res;
	}
	
	/**
	 * This method calculates the full absolute path relative to the Project. In the case that the project has a designated folder that
	 * contains all the Web content (i.e. WebContent), this is determined and added to the path so that we can find the resource.
	 * 
	 * @param project project that should contain the resource
	 * @param pathToResource path to a resource derived from the URL that might not contain the full path relative to the project root
	 * @return
	 */
	protected IResource getResource( IProject project, IPath pathToResource ){
		
		IResource res = null;
		IContainer webContentContainer;
		
		IWebResourceLocator adapter = (IWebResourceLocator)project.getAdapter( IWebResourceLocator.class );
		
		if( adapter != null ){
			//find the destionation folder within the project to install runtime
			webContentContainer = adapter.getWebResourceContainer();			
		}
		else{
			webContentContainer = project;
		}
		
		if( webContentContainer != null ){
			IPath projectPath = webContentContainer.getProjectRelativePath().append(pathToResource);
			res = project.findMember(projectPath);
		}
		return res;
	}
	
	public IStorage findRemoteResource( URL url ){
		return new URLStorage( url );		
	}
	
	/**
	 * This method gets the content of the resource that maps to the URL. It tried to find it in the local workspace and
	 * if not found, it used a URL connection.
	 * 
	 * @param sourceURL the ulr to the resource (no host and port)
	 * @param basePathHint (optional) portion of the ulr's path that maps to the root of the project
	 * @param projectHint (optional) project where the resources should be available
	 * @return stream with content
	 * @throws CoreException
	 */
	public InputStream getSourceContent( URL sourceURL, String basePathHint, IProject projectHint ) throws CoreException{

		//first try to find it locally
		InputStream stream = getLocalSourceContent( sourceURL, basePathHint, projectHint );
		
		//try to get as remote resource
		if( stream == null ){
			stream = getRemoteSourceContent( sourceURL );
		}
		
		return stream;
		
	}
	
	/**
	 * This method gets the content of the local resource that maps to the URL. 
	 * 
	 * @param sourceURL the ulr to the resource (no host and port)
	 * @param basePathHint (optional) portion of the ulr's path that maps to the root of the project
	 * @param projectHint (optional) project where the resources should be available
	 * @return stream with content
	 * @throws CoreException
	 */
	public InputStream getLocalSourceContent( URL sourceURL, String basePathHint, IProject projectHint ) throws CoreException{
		//try to find Source in a local resource
		IResource resource = findLocalResource( sourceURL, basePathHint, projectHint);

		if( resource != null ){
			
			if ( resource instanceof IFile ) {
				try {
					return new BufferedInputStream( ((IFile)resource).getContents() );
					
					
				} catch (CoreException ce) {
					MozideCorePlugin.logStatus(new Status(IStatus.ERROR, MozideCorePlugin.PLUGIN_ID,
							IStatus.ERROR, "I/O Exception reading local source" /*TODO:i18n*/, ce));
					// return null
				}
			}
		}
		return null;
	}
	
	/**
	 * This method gets the content of the local resource that maps to the URL. 
	 * 
	 * @param sourceURL the ulr to the resource (no host and port)
	 * @throws CoreException
	 */
	public InputStream getRemoteSourceContent( URL sourceURL ) throws CoreException{
		//get content using the URL connection
		try {
			URLConnection conn = sourceURL.openConnection();
			conn.connect();
			return new BufferedInputStream(conn.getInputStream());
		} catch (IOException ioe) {
			throw new CoreException(new Status(IStatus.ERROR, MozideCorePlugin.PLUGIN_ID,
					IStatus.ERROR, "I/O Exception reading source" /*TODO:i18n*/, ioe));
		}
	}
	
	public URLStorage getURLStorage(String location) {		
		URLStorage storage = null;
		try {
			URL url = new URL(location);
			storage = new URLStorage(url);
		}catch (MalformedURLException mue) {
			// return null
		}
		return storage;
	}
	
	protected class URLStorage extends PlatformObject implements IStorage {

		private URL _url;

		public URLStorage(URL url) {
			_url = url;
		}

		public InputStream getContents() throws CoreException {
			try {
				URLConnection conn = _url.openConnection();
				conn.connect();
				return new BufferedInputStream(conn.getInputStream());
			} catch (MalformedURLException mue) {
				throw new CoreException(new Status(IStatus.ERROR, MozideCorePlugin.PLUGIN_ID,						
						IStatus.ERROR, "Malformed source URL" /*TODO:i18n*/, mue));
			} catch (IOException ioe) {
				throw new CoreException(new Status(IStatus.ERROR, MozideCorePlugin.PLUGIN_ID,
						IStatus.ERROR, "I/O Exception reading source" /*TODO:i18n*/, ioe));
			}
		}

		public IPath getFullPath() {
			return new Path(this._url.toString());
//			return null;
		}

		public URL getURL() {
			return _url;
		}

		public String getName() {
			return _url.toString();
		}

		public boolean isReadOnly() {
			return true;
		}

		public boolean equals(Object obj) {
			if (obj instanceof URLStorage) {
				return ((URLStorage)obj).getURL().equals(getURL());
			}
			return super.equals(obj);
		}

		public int hashCode() {
			return getURL().hashCode();
		}

		public Object getAdapter(Class adapter) {
			if (URL.class.equals(adapter))
				return getURL();
			return super.getAdapter(adapter);
		}
	}
}
