/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/
package org.eclipse.atf.mozilla.ide.debug.internal.model;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.atf.mozilla.ide.core.IXPCOMThreadProxyHelper;
import org.eclipse.atf.mozilla.ide.core.XPCOMThreadProxy;
import org.eclipse.atf.mozilla.ide.debug.INestedEventLoop;
import org.eclipse.atf.mozilla.ide.debug.JSDebugCoreMessages;
import org.eclipse.atf.mozilla.ide.debug.MozillaDebugPlugin;
import org.eclipse.atf.mozilla.ide.debug.model.JSLineBreakpoint;
import org.eclipse.atf.mozilla.ide.debug.model.JSSourceLocator;
import org.eclipse.atf.mozilla.ide.debug.preferences.JSDebugPreferenceNames;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.Preferences.IPropertyChangeListener;
import org.eclipse.core.runtime.Preferences.PropertyChangeEvent;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointListener;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.IBreakpointManagerListener;
import org.eclipse.debug.core.IBreakpointsListener;
import org.eclipse.debug.core.IDebugEventSetListener;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IMemoryBlock;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IThread;
import org.mozilla.xpcom.Mozilla;
import org.mozilla.interfaces.jsdIDebuggerService;
import org.mozilla.interfaces.jsdIExecutionHook;
import org.mozilla.interfaces.jsdIScript;
import org.mozilla.interfaces.jsdIScriptEnumerator;
import org.mozilla.interfaces.nsIEventQueue;
import org.mozilla.interfaces.nsISupports;


/**
 * @author Adam
 *
 * TODO To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Style - Code Templates
 */
public class JSDebugTarget extends JSDebugElement implements IDebugTarget, IDebugEventSetListener, IBreakpointListener, IPropertyChangeListener, IBreakpointManagerListener {

	private ILaunch _launch;
	private IProcess _process;
	private JSDebugThread[] _threads = new JSDebugThread[0];
	private URL _appURL;
	private boolean _terminated = false;
	private boolean _connected = false;
	private String _errorString = null;
	private ILog _logger = null;

	private jsdIDebuggerService _debuggerService;
	private INestedEventLoop _eventLoop;
	private IXPCOMThreadProxyHelper _proxyHelper;
	private IBreakpointManager _breakpointManager;
	private BreakpointsChangedListener _fBreakpointsChangedListener;

	public JSDebugTarget(ILaunch launch, URL appURL, INestedEventLoop eventLoop,
			IXPCOMThreadProxyHelper proxyHelper) throws DebugException {

		super(null);

		_launch = launch;
		_appURL = appURL;

		//TODO: can we defer creation of the thread until connection is received?
		JSDebugThread thread = new JSDebugThread(this);
		_threads = new JSDebugThread[] { thread };
		// TODO setup listener for terminate

		_proxyHelper = proxyHelper;
		_debuggerService = (jsdIDebuggerService)XPCOMThreadProxy.createProxy(
				MozillaDebugPlugin.createDebuggerService(), proxyHelper);

		_eventLoop = eventLoop;

		setupDebuggerHooks( thread );
		
	    _debuggerService.setFlags(jsdIDebuggerService.ENABLE_NATIVE_FRAMES);

	    handleConnectionStarted();

	    _breakpointManager = DebugPlugin.getDefault().getBreakpointManager();
	    _breakpointManager.addBreakpointListener(this);
	    _breakpointManager.addBreakpointManagerListener(this);
	    _breakpointManager.addBreakpointListener(_fBreakpointsChangedListener = new BreakpointsChangedListener());
	    DebugPlugin.getDefault().addDebugEventListener(this);

	}

	protected void setupDebuggerHooks( JSDebugThread thread ){
		_debuggerService.setScriptHook(thread);
	    _debuggerService.setBreakpointHook(thread);
	    
	    _debuggerService.setDebugHook(thread); //not sure why we need this hook
	    
	    //set remaining hooks based on preferences
	    Preferences prefs = MozillaDebugPlugin.getDefault().getPluginPreferences();
	    
//	    if( prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_DEBUGGER_KEYWORD) ){
//	    	_debuggerService.setDebuggerHook(thread);
//	    }
	    
    	_debuggerService.setErrorHook(thread);
    	_debuggerService.setThrowHook(thread);
    	try {
	    	if( _launch.getLaunchConfiguration().getAttribute(JSDebugPreferenceNames.OVERRIDE_PREFERENCES, false) ) {
	    		if( _launch.getLaunchConfiguration().getAttribute(JSDebugPreferenceNames.SUSPEND_ON_FIRST_LINE, false) ) {
					_debuggerService.setTopLevelHook( thread );
				} else {
					_debuggerService.setTopLevelHook( null );
				}
	    		if( _launch.getLaunchConfiguration().getAttribute(JSDebugPreferenceNames.SUSPEND_ON_DEBUGGER_KEYWORD, false) ) {
	    			_debuggerService.setDebuggerHook(_threads[0]);
	    		} else {
	    			_debuggerService.setDebuggerHook(null);
	    		}
    			_threads[0].setSuspendOnErrors(
    					_launch.getLaunchConfiguration().getAttribute(JSDebugPreferenceNames.SUSPEND_ON_ERRORS, false));
	    		_threads[0].setSuspendOnExceptions(
	    				_launch.getLaunchConfiguration().getAttribute(JSDebugPreferenceNames.SUSPEND_ON_EXCEPTIONS, false));
	    	} else {
	    		if( prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_FIRST_LINE) ) {
	    			_debuggerService.setTopLevelHook( thread );
	    		} else {
	    			_debuggerService.setTopLevelHook( null );
	    		}
	    		if( prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_DEBUGGER_KEYWORD) ) {
	    			_debuggerService.setDebuggerHook(_threads[0]);
	    		} else {
	    			_debuggerService.setDebuggerHook(null);
	    		}
	    		_threads[0].setSuspendOnErrors(prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_ERRORS));
	    		_threads[0].setSuspendOnExceptions(prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_EXCEPTIONS));
	    	}
			
		} catch (CoreException e) {
			
		}
	    
	}
	
	protected void removeDebuggerHooks(){
		_debuggerService.setTopLevelHook(null);
		_debuggerService.setFunctionHook(null);
		_debuggerService.setBreakpointHook(null);
		_debuggerService.setDebuggerHook(null);
		_debuggerService.setErrorHook(null);
		_debuggerService.setThrowHook(null);
		_debuggerService.setScriptHook(null);
		_debuggerService.setInterruptHook(null);
	    
		//remove as listener to changes to the preferences
	    Preferences prefs = MozillaDebugPlugin.getDefault().getPluginPreferences();
	    prefs.removePropertyChangeListener( this );
	}
	
	/*
	 * Listen to changes to the Preferences
	 */
	public void propertyChange(PropertyChangeEvent event) {
		
		if (event.getProperty().equals(JSDebugPreferenceNames.SUSPEND_ON_DEBUGGER_KEYWORD)) {
			Boolean newValue= (Boolean)event.getNewValue();
			
			if ( newValue.booleanValue() ){
				_debuggerService.setDebuggerHook(_threads[0]);
			}
			else{
				_debuggerService.setDebuggerHook( null );
			}
		}
		else if (event.getProperty().equals(JSDebugPreferenceNames.SUSPEND_ON_ERRORS)) {
			Boolean newValue= (Boolean)event.getNewValue();
			JSDebugThread thread = _threads[0];
			thread.setSuspendOnErrors(newValue.booleanValue());
		}
		else if (event.getProperty().equals(JSDebugPreferenceNames.SUSPEND_ON_EXCEPTIONS)) {
			Boolean newValue= (Boolean)event.getNewValue();
			JSDebugThread thread = _threads[0];
			thread.setSuspendOnExceptions(newValue.booleanValue());
		}
		else if (event.getProperty().equals(JSDebugPreferenceNames.SUSPEND_ON_FIRST_LINE)) {
			Boolean newValue= (Boolean)event.getNewValue();
			
			if ( newValue.booleanValue() ){
				_debuggerService.setTopLevelHook(_threads[0]);
			}
			else{
				_debuggerService.setTopLevelHook( null );
				
			}
		}
		
	}
	
	public IXPCOMThreadProxyHelper getProxyHelper() {
		return _proxyHelper;
	}

	public jsdIDebuggerService getDebuggerService() {
		return _debuggerService;
	}
	
	private nsIEventQueue _nestedEventQ;

	public interface INestedCallback {
		public void onNest();
	}

	public long enterNestedEventLoop(INestedCallback callback) {
		//TODO synchronization?
		
		//TODO: does this eventQueue really do anything?
		// eventQueue is intended to stop processing of events while we're in a callback
		_nestedEventQ = MozillaDebugPlugin.getEventQueueService().pushThreadEventQueue();
		if (callback != null)
			callback.onNest();

		long rv = _eventLoop.runEventLoop();
		MozillaDebugPlugin.getEventQueueService().popThreadEventQueue(_nestedEventQ);
		_nestedEventQ = null;
		return rv;
	}

	public void exitNestedEventLoop(long rv) {
		_eventLoop.stopEventLoop(rv);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#getProcess()
	 */
	public IProcess getProcess() {
		return _process;
	}

	public void setProcess(IProcess process) {
		_process = process;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#getThreads()
	 */
	public IThread[] getThreads() throws DebugException {
		return _threads;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#hasThreads()
	 */
	public boolean hasThreads() throws DebugException {
		return getThreads().length > 0;
	}

	protected void removeThread(IThread thread) {
		List newThreads = new ArrayList(_threads.length);
		for (int i = 0; i < _threads.length; i++)
			newThreads.add(_threads[i]);
		newThreads.remove(thread);
		_threads = (JSDebugThread[])newThreads.toArray(new JSDebugThread[newThreads.size()]);

	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#getName()
	 */
	public String getName() throws DebugException {
		return _appURL == null ? JSDebugCoreMessages.JSDebugCore_0 : _appURL.toExternalForm();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugTarget#supportsBreakpoint(org.eclipse.debug.core.model.IBreakpoint)
	 */
	public boolean supportsBreakpoint(IBreakpoint breakpoint) {
		return breakpoint instanceof JSLineBreakpoint;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		return !isTerminated();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		return _terminated;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#terminate()
	 * 
	 * NOTE: This method, at least, needs to be Thread safe because it can get
	 * call multiple times by the separate Threads.
	 */
	synchronized public void terminate() throws DebugException {
		if (isTerminated())
			return;

		shutdown();
		if (_threads.length > 0) {
			for (int i = 0; i < _threads.length; i++)
				_threads[i].terminate();
			_threads = new JSDebugThread[0];
		}
		IProcess proc = getProcess();
		if (proc != null)
			proc.terminate();
		_terminated = true;
		fireTerminateEvent(); 
	}

	/**
	 * DebugTarget object cleanup common to terminate and disconnect
	 */
	private void shutdown() {
		// Unregister as breakpoint listener
		_breakpointManager.removeBreakpointListener(this);
		_breakpointManager.removeBreakpointListener(_fBreakpointsChangedListener);
	    DebugPlugin.getDefault().removeDebugEventListener(this);

	    stopDebugger();
	}
	
	public void resetDebugger() {	
		if (_debuggerService == null)
			return;
		JSDebugThread thread = (JSDebugThread)_threads[0];
		thread.clearCX();
		
		_debuggerService.clearAllBreakpoints();
		_debuggerService.gC();
		
		if (isSuspended())
			thread.resumeAbortScript();
		
	}

	private void stopDebugger() {
		if (_debuggerService == null)
			return;

		removeDebuggerHooks();
		_debuggerService.clearAllBreakpoints();

		if (isSuspended())
			exitNestedEventLoop(jsdIExecutionHook.RETURN_CONTINUE);

		_debuggerService.gC();

	    if (!_debuggerService.getInitAtStartup())
	    	_debuggerService.off();

	    _debuggerService = null;
	    _connected = false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canResume()
	 */
	public boolean canResume() {
		return isSuspended();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canSuspend()
	 */
	public boolean canSuspend() {
		return !isTerminated() && !isDisconnected() && !isSuspended();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#isSuspended()
	 */
	public boolean isSuspended() {
		return _threads.length > 0 ?
				_threads[0].isSuspended() : false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#resume()
	 */
	public void resume() throws DebugException {
		JSDebugThread thread = (JSDebugThread)_threads[0];
		thread.resume();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#suspend()
	 */
	public void suspend() throws DebugException {
		_proxyHelper.syncExec(new Runnable() {
			public void run() {
				_debuggerService.setInterruptHook((JSDebugThread)_threads[0]);
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointListener#breakpointAdded(org.eclipse.debug.core.model.IBreakpoint)
	 */
	public void breakpointAdded(final IBreakpoint breakpoint) {
		if (supportsBreakpoint(breakpoint)) {	
			// RSG - change to get source locator from target
			final JSSourceLocator locator = this.locator;
			_debuggerService.enumerateScripts(new jsdIScriptEnumerator() {
				public void enumerateScript(jsdIScript script) {
					establishBreakpoint(true, locator, script, breakpoint);
				}
	
				public nsISupports queryInterface(String id) {
					return Mozilla.queryInterface(this, id);
				}			
			});
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointListener#breakpointRemoved(org.eclipse.debug.core.model.IBreakpoint, org.eclipse.core.resources.IMarkerDelta)
	 */
	public void breakpointRemoved(final IBreakpoint breakpoint, IMarkerDelta delta) {
		if (supportsBreakpoint(breakpoint)) {
			// RSG - change to get source locator from target
			final JSSourceLocator locator = this.locator;
			_debuggerService.enumerateScripts(new jsdIScriptEnumerator() {
				public void enumerateScript(jsdIScript script) {
					establishBreakpoint(false, locator, script, breakpoint);
				}
			
				public nsISupports queryInterface(String id) {
					return Mozilla.queryInterface(this, id);
				}			
			});
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointListener#breakpointChanged(org.eclipse.debug.core.model.IBreakpoint, org.eclipse.core.resources.IMarkerDelta)
	 */
	public void breakpointChanged(final IBreakpoint breakpoint, IMarkerDelta delta) {

		// Don't process change when all breakpoints are disabled or
		// there is no marker delta				
		if (_breakpointManager.isEnabled() && delta != null) {
			if (supportsBreakpoint(breakpoint)) {	
				boolean deltaEnabled;
				deltaEnabled = delta.getAttribute(IBreakpoint.ENABLED, false);

				final int deltaLNumber = delta.getAttribute(IMarker.LINE_NUMBER, 0);
  				IMarker marker = breakpoint.getMarker();
				int lineNumber = -1;
				lineNumber = marker.getAttribute(IMarker.LINE_NUMBER, 0); 
				
				try {
					if (lineNumber != deltaLNumber) {
						breakpoint.setMarker(marker);
 						// TODO: Should breakpoints be removed on line number change?
/*						if (breakpoint.isEnabled()) {
								final JSSourceLocator locator = (JSSourceLocator)getLaunch().getSourceLocator();
								_debuggerService.enumerateScripts(new jsdIScriptEnumerator() {
									public void enumerateScript(jsdIScript script) {
										establishBreakpoint(locator, script, breakpoint, deltaLNumber);
									}
								
									public nsISupports queryInterface(String id) {
										return Mozilla.queryInterface(this, id);
									}			
								});
							} */
					} 
					if (deltaEnabled != breakpoint.isEnabled()){
						if (breakpoint.isEnabled()) {
							breakpointAdded(breakpoint);
						} else {
							breakpointRemoved(breakpoint, null);
						}
					}
				} catch (CoreException ce) {
					// Breakpoint doesn't throw this exception 
					MozillaDebugPlugin.log(ce);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IBreakpointManagerListener#breakpointManagerEnablementChanged(org.eclipse.debug.core.model.IBreakpoint, org.eclipse.core.resources.IMarkerDelta)
	 */	
	public void breakpointManagerEnablementChanged(boolean enabled) {
		IBreakpoint[] breakpoints = DebugPlugin.getDefault().getBreakpointManager().getBreakpoints(getModelIdentifier());
		for (int i = 0; i < breakpoints.length; i++) {
			if (supportsBreakpoint(breakpoints[i])) {
				try {
					if (breakpoints[i].isEnabled()) {
						if (enabled) {
							breakpointAdded(breakpoints[i]);
						} else {
							breakpointRemoved(breakpoints[i], null);
						}
					}
				} catch (CoreException e) {
//					 Breakpoint doesn't throw this exception 
					MozillaDebugPlugin.log(e);
				}
			}
		}
	}

	static public boolean establishBreakpoint(JSSourceLocator locator, jsdIScript script,
											  IBreakpoint breakpoint, int oldLinNumber) {
		
		if (!(breakpoint instanceof JSLineBreakpoint)) return false;
		boolean match = locator.matches(breakpoint, script.getFileName());
		if (match) {
			// Remove old breakpoint
			match = establishBreakpoint(false, locator, script, oldLinNumber);
			
			// TODO: need to figure out what should be done when false is returned.
			// Just return for now. Not logging since I;m not sure it is an error;
			if (!match){
//				MozillaDebugPlugin.getDefault().getLog().log("Unexpected error removing breakpoint");
				return false;
			}
			// Add new breakpoint
			JSLineBreakpoint jsBPoint = (JSLineBreakpoint) breakpoint;
			try {
				int lineNumber = jsBPoint.getLineNumber();
				match = establishBreakpoint(true, locator, script, lineNumber);
			} catch (CoreException e) {
				IStatus status = new Status(IStatus.ERROR, MozillaDebugPlugin.ID,
						IStatus.ERROR, JSDebugCoreMessages.JSDebugCore_2, e);
				MozillaDebugPlugin.getDefault().getLog().log(status);
			}		

			//			 TODO: need to figure out what should be done when false is returned.			
		}
		
		return match;
		
	}
	
	static public boolean establishBreakpoint(boolean activate, JSSourceLocator locator, jsdIScript script, IBreakpoint breakpoint) {

		if (!(breakpoint instanceof JSLineBreakpoint)) return false;
		boolean match = locator.matches(breakpoint, script.getFileName());
		if (match) {
			JSLineBreakpoint jsBPoint = (JSLineBreakpoint) breakpoint;
			try {
				int lineNumber = jsBPoint.getLineNumber();
				match = establishBreakpoint(activate, locator, script, lineNumber);
			} catch (CoreException e) {
				IStatus status = new Status(IStatus.ERROR, MozillaDebugPlugin.ID,
						IStatus.ERROR, JSDebugCoreMessages.JSDebugCore_3, e);
				MozillaDebugPlugin.getDefault().getLog().log(status);
			}
		}
		return match; 
	}

	
	static private boolean establishBreakpoint(boolean activate, JSSourceLocator locator, jsdIScript script, int line) {

		final int PCMAP_SOURCETEXT = 1;
		long baseLineNumber = script.getBaseLineNumber();
		long lineExtent = script.getLineExtent();
		boolean match = line >= baseLineNumber
				&& line <= baseLineNumber + lineExtent
				&& (script.isLineExecutable(line, PCMAP_SOURCETEXT) || baseLineNumber == line);
		if (match) {
			long pc = script.lineToPc(line, PCMAP_SOURCETEXT);
			if (activate) {
				System.err.println("setting breakpoint at " //$NON-NLS-1$
						+ script.getFileName() + " line=" + line + " pc="  //$NON-NLS-2$
						+ pc);
				script.setBreakpoint(pc);
			} else {
				System.err.println("clearing breakpoint at " //$NON-NLS-1$
						+ script.getFileName() + " line=" + line + " pc=" //$NON-NLS-2$
						+ pc);
				script.clearBreakpoint(pc);
			}
		}

		return match;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDisconnect#canDisconnect()
	 */
	public boolean canDisconnect() {
		return !isDisconnected();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDisconnect#disconnect()
	 */
	public void disconnect() throws DebugException {
		shutdown();

		if (_threads.length > 0) {
			for (int i = 0; i < _threads.length; i++)
				;//TODO _threads[i].terminate();
			_threads = new JSDebugThread[0];
		}

		fireChangeEvent(DebugEvent.CONTENT);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDisconnect#isDisconnected()
	 */
	public boolean isDisconnected() {
		return !_connected;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IMemoryBlockRetrieval#supportsStorageRetrieval()
	 */
	public boolean supportsStorageRetrieval() {
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IMemoryBlockRetrieval#getMemoryBlock(long, long)
	 */
	public IMemoryBlock getMemoryBlock(long startAddress, long length)
			throws DebugException {
		notSupported("", null); //$NON-NLS-1$
		return null; // will never get here
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugElement#getLaunch()
	 */
	public ILaunch getLaunch() {
		return _launch;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IDebugElement#getDebugTarget()
	 */

	public IDebugTarget getDebugTarget() {
		return this;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.atf.mozilla.ide.debug.internal.model.JSDebugElement#getLabel()
	 */
	public String getLabel() {
		String urlString = "<unknown>"; //TODO i18n
		try {
			urlString = getName();
		} catch (DebugException e) {
		}

		int paramIndex = urlString.indexOf('?');
		StringBuffer label = new StringBuffer((paramIndex > 0) ? urlString.substring(0, paramIndex) : urlString);
		if (_errorString != null) {
			label.append(" error=");
			label.append(_errorString);
		}
		label.append(" (");
		if (isTerminated()) {
			label.append("Terminated"); //TODO i18n			
		} else if (isSuspended()) {
			label.append("Stopped"); //TODO i18n			
		} else if (isDisconnected()) {
			label.append((_debuggerService == null) ? "Disconnected" : "Waiting"); //TODO i18n
		} else {
			label.append("Running"); //TODO i18n			
		}
		label.append(')');
		return label.toString();
	}

	public void handleConnectionStarted() {
		_connected = true;
		fireChangeEvent(DebugEvent.STATE);
		_threads[0].fireCreationEvent();
	}

	public void handleConnectionTerminated(IStatus status) {
		_connected = false;
		fireChangeEvent(DebugEvent.STATE);

		if (status.getCode() != IStatus.OK)
			_errorString = status.getMessage();

//TODO
//		if (_connector == null) {
//			// Event initiated by a disconnect or terminate.  Do nothing.
//		} else {
//			// User-terminated process or unexpected disconnect.
//			// Corresponding app instance is gone. Terminate the debug target.
//			_connector = null;
			try {
				terminate();
			} catch (DebugException de) {
				// Not much we can do except log it
				MozillaDebugPlugin.log(de);
			}
//		}
	}

	public void handleLogMessage(IStatus status) {
		if (_logger != null)
			_logger.log(status);

		MozillaDebugPlugin.getDefault().getLog().log(status);
	}

	public void setLogger(ILog logger) {
		_logger = logger;
	}

	public void handleDebugEvents(DebugEvent[] events) {
		for (int i=0; i < events.length; i++) {
			DebugEvent event = events[i];
			switch (event.getKind()) {
			case DebugEvent.TERMINATE: {
				Object source = event.getSource();

				if (source.equals(_process) ||
					source.equals(_threads[0])) {

					try {
						terminate();
					} catch (DebugException de) {
						MozillaDebugPlugin.log(de);
					}
				}
				break;
			} case DebugEvent.MODEL_SPECIFIC: {
                    final Object obj = events[i].getSource();
                    if(obj.equals(getLaunch())) {
                    	Object data = events[i].getData();
                    	if (data instanceof String) {
                    		if ( ((String)data).equals("locationChange") ) {
                    			resetDebugger();
                    		}
                    	}
                    }
			}
			default:
			}
		}
	}
	
	/**
	 * Throws a IStatus in a Debug Event
	 * 
	 */
	private void fireError(IStatus status) {
		DebugEvent event = new DebugEvent(this, DebugEvent.MODEL_SPECIFIC);
		event.setData(status);
		fireEvent(event);
	}
	
	// This class could be up in the Debug UI plugin, but will probably need to 
	// be in the model when multiple debug session are supported.
    private class BreakpointsChangedListener implements IBreakpointsListener {

		public void breakpointsAdded(IBreakpoint[] breakpoints) {
			
		}

		public void breakpointsChanged(IBreakpoint[] breakpoints, IMarkerDelta[] deltas) {

			try {
				if (_breakpointManager.isEnabled()) {
					for (int i = 0; i < breakpoints.length; i++) {
						if (supportsBreakpoint(breakpoints[i]) && breakpoints[i].isEnabled() && deltas[i] != null) {
							int deltaLNumber = deltas[i].getAttribute(IMarker.LINE_NUMBER, 0);
							int lineNumber = -1;
							IMarker marker = breakpoints[i].getMarker();
							lineNumber = marker.getAttribute(IMarker.LINE_NUMBER, 0);
							if (lineNumber != deltaLNumber) {
								String errorMessage = JSDebugCoreMessages.JSDebugCore_1;
								Status status = new Status(IStatus.WARNING, MozillaDebugPlugin.ID, DebugPlugin.INTERNAL_ERROR, errorMessage, null);
								fireError(status);
								break;
							}
						}
						
		
					}
				}
			} catch (CoreException ce) {
				// Breakpoint doesn't throw this exception 
				MozillaDebugPlugin.log(ce);
			}
			
		}

		public void breakpointsRemoved(IBreakpoint[] breakpoints, IMarkerDelta[] deltas) {
			
		}
    	
    }
    
    // RSG put Source locator in target for now. 
    private JSSourceLocator locator;
    public JSSourceLocator getSourceLocator() {
    	return locator;
    }
    
    public void setSourceLocator(JSSourceLocator locator) {
    	this.locator = locator;  	
    }
}
