/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/
package org.eclipse.atf.mozilla.ide.debug.internal.model;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Vector;

import org.eclipse.atf.mozilla.ide.debug.MozillaDebugPlugin;
import org.eclipse.atf.mozilla.ide.debug.model.IJSDebugScriptElement;
import org.eclipse.atf.mozilla.ide.debug.model.JSSourceLocator;
import org.eclipse.atf.mozilla.ide.debug.preferences.JSDebugPreferenceNames;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.IStatusHandler;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.mozilla.interfaces.jsdICallHook;
import org.mozilla.interfaces.jsdIContext;
import org.mozilla.interfaces.jsdIErrorHook;
import org.mozilla.interfaces.jsdIExecutionHook;
import org.mozilla.interfaces.jsdIProperty;
import org.mozilla.interfaces.jsdIScript;
import org.mozilla.interfaces.jsdIScriptHook;
import org.mozilla.interfaces.jsdIStackFrame;
import org.mozilla.interfaces.jsdIValue;
import org.mozilla.interfaces.nsIBaseWindow;
import org.mozilla.interfaces.nsIDOMDocument;
import org.mozilla.interfaces.nsIDOMLocation;
import org.mozilla.interfaces.nsIDOMNSDocument;
import org.mozilla.interfaces.nsIDOMWindow;
import org.mozilla.interfaces.nsISupports;
import org.mozilla.interfaces.nsIWebBrowser;
import org.mozilla.xpcom.Mozilla;
import org.mozilla.xpcom.XPCOMException;


/**
 * @author Adam
 *
 * TODO To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Style - Code Templates
 */
public class JSDebugThread extends JSDebugElement implements IThread, jsdIExecutionHook, jsdIErrorHook, jsdICallHook, jsdIScriptHook {

	final private IStackFrame[] _emptyStack = new IStackFrame[0];
	private IBreakpoint[] _breakpoints = new IBreakpoint[0];
	private IStackFrame[] _stackFrames = _emptyStack;
	final static private long TYPE_RUNNING = -1L;
	private long _suspendedType = TYPE_RUNNING;
	private JSDebugVariable _exceptionObject = null;
	private boolean _stepping = false, _terminated = false;
	private IBreakpointManager _breakpointManager;
	private boolean _suspendOnErrors, _suspendOnExceptions;
	private boolean _reachedFirstLine;


	/*
	 * jsdIScript elements are going to be organized in a hierarchical model based on the jsdIScript
	 * filename and the baselinenumber and lineextend members. Only a reference to the Top level
	 * scripts (should map to files) are kept in this map and keyed by the filename URL.
	 */
	protected HashMap topScriptElements = new HashMap();

	/**
	 * @param parent
	 */
	public JSDebugThread(IDebugTarget target) {
		super(target);
		_breakpointManager = DebugPlugin.getDefault().getBreakpointManager();

		Preferences prefs = MozillaDebugPlugin.getDefault().getPluginPreferences();
	    _suspendOnErrors = prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_ERRORS);
	    _suspendOnExceptions = prefs.getBoolean(JSDebugPreferenceNames.SUSPEND_ON_EXCEPTIONS);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getStackFrames()
	 */
	public IStackFrame[] getStackFrames() throws DebugException {
		return isSuspended() ? _stackFrames : _emptyStack;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#hasStackFrames()
	 */
	public boolean hasStackFrames() throws DebugException {
		return _stackFrames != null && (_stackFrames.length > 0);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getPriority()
	 */
	public int getPriority() throws DebugException {
		// TODO Auto-generated method stub
		return 0;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getTopStackFrame()
	 */
	public IStackFrame getTopStackFrame() throws DebugException {
		return _stackFrames.length > 0 ? _stackFrames[0] : null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getName()
	 */
	public String getName() throws DebugException {
		return "Application Thread"; //TODO i18n
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IThread#getBreakpoints()
	 */
	public IBreakpoint[] getBreakpoints() {
		return _breakpoints;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canResume()
	 */
	public boolean canResume() {
		return isSuspended();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#canSuspend()
	 */
	public boolean canSuspend() {
		return !isSuspended();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#isSuspended()
	 */
	public boolean isSuspended() {
		return _suspendedType != TYPE_RUNNING;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#resume()
	 */
	public void resume() throws DebugException {
		resume(false);
	}
	
	/*
	 * Special Method used when a refresh in the browser is done when
	 * suspended. Resume but abort the running script.
	 */
	protected void resumeAbortScript() {
		resume(true);
		_suspendedType = TYPE_RUNNING;
	}
	
	/* 
	 * The logic for resume
	 */
	private void resume(boolean abortScript) {

		if (_stepping) {
			_stepping = false;
	
			final JSDebugTarget target = (JSDebugTarget)getDebugTarget();
			target.getProxyHelper().syncExec(new Runnable() {
				public void run() {
					target.getDebuggerService().setInterruptHook(null);
				}
			});
		}
		
		_reachedFirstLine = false;
		long returnType = (_suspendedType == TYPE_THROW) ? RETURN_CONTINUE_THROW : RETURN_CONTINUE;
//System.err.println("resume type="+((_suspendedType == TYPE_THROW) ? "RETURN_CONTINUE_THROW" : "RETURN_CONTINUE"));
		if (abortScript)returnType = RETURN_ABORT;
		((JSDebugTarget)getDebugTarget()).exitNestedEventLoop(returnType);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ISuspendResume#suspend()
	 */
	public void suspend() throws DebugException {
		getDebugTarget().suspend();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepInto()
	 */
	public boolean canStepInto() {
		return !_reachedFirstLine;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepOver()
	 */
	public boolean canStepOver() {
		return !_reachedFirstLine;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#canStepReturn()
	 */
	public boolean canStepReturn() {
		return !_reachedFirstLine;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#isStepping()
	 */
	public boolean isStepping() {
		return _stepping;
	}

	public void setStepping(int pastDepth, int pastLine, String pastFrame, int overDepth) {
		_stepping = true;
		_stepPastDepth = pastDepth;
		_stepPastLine = pastLine;
		_stepPastFrame = pastFrame;
		_stepOverDepth = overDepth;
	}

	private int _stepPastDepth = -1;
	private int _stepPastLine = -1;
	private String _stepPastFrame = null;
	private int _stepOverDepth = -1;

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepInto()
	 */
	public void stepInto() throws DebugException {
		getTopStackFrame().stepInto();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepOver()
	 */
	public void stepOver() throws DebugException {
		getTopStackFrame().stepOver();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IStep#stepReturn()
	 */
	public void stepReturn() throws DebugException {
		getTopStackFrame().stepReturn();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#canTerminate()
	 */
	public boolean canTerminate() {
		return !isTerminated();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#isTerminated()
	 */
	public boolean isTerminated() {
		return _terminated;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.ITerminate#terminate()
	 */
	public void terminate() throws DebugException {
		if (isTerminated())
			return;

		_terminated = true;
		_stackFrames = null;
		fireTerminateEvent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.atf.mozilla.ide.debug.internal.model.JSDebugElement#getLabel()
	 */
	public String getLabel() {
		StringBuffer label = new StringBuffer("<unknown>"); //TODO i18n

		try {
			label = new StringBuffer(getName());
			label.append(" (");
			try {
				switch ((int)_suspendedType) {
				case (int)TYPE_INTERRUPTED:
					label.append(isStepping() ? "stepping" : "interrupted"); //TODO i18n
					break;
				case (int)TYPE_BREAKPOINT:
					label.append("at breakpoint"); //TODO i18n
					break;
				case (int)TYPE_DEBUG_REQUESTED:
					label.append("debug requested"); //TODO i18n
					break;
				case (int)TYPE_DEBUGGER_KEYWORD:
					if (!_reachedFirstLine) {
						label.append("debugger keyword"); //TODO i18n
					}else {
						label.append("suspended at start"); //TODO i18n
					}
					break;
				case (int)TYPE_THROW:
					label.append("exception: "); //TODO i18n
					if (_exceptionObject != null) {
						jsdIValue exception = (jsdIValue)_exceptionObject.getValue().getAdapter(jsdIValue.class);
						if (exception != null && exception.getJsClassName() == "Error")
			        		label.append(exception.getProperty("message"));
			        	else
			        		label.append(_exceptionObject.getValue().getValueString());
					}
					break;
				case (int)TYPE_RUNNING:
					label.append("running"); //TODO i18n
					break;
				default:
					label.append("unknown"); //TODO i18n
				break;
				}
			} finally {
				label.append(')');
			}
		} catch (DebugException de) {
			de.printStackTrace();
		}

		return label.toString();
	}

	public nsISupports queryInterface(String id) {
		return Mozilla.queryInterface(this, id);
	}

	public void setSuspendOnErrors(boolean suspend) {
		_suspendOnErrors = suspend;
	}

	public void setSuspendOnExceptions(boolean suspend) {
		_suspendOnExceptions = suspend;
	}

	public boolean onError(final String message, String fileName, long line, long pos, long flags, long errnum, final jsdIValue exception) {
//		MozillaDebugPlugin.log("onError: "+message+" filename="+fileName+" exception="+exception.getStringValue());

		boolean resume = !_suspendOnErrors;
		try {
			int severity = IStatus.OK;
			Throwable throwable = null;

			if ((flags & REPORT_WARNING) > 0)
				severity = IStatus.WARNING;
			if ((flags & REPORT_ERROR) > 0) {
				severity = IStatus.ERROR;
			}
			if ((flags & REPORT_EXCEPTION) > 0) {
				severity = IStatus.INFO;
				throwable = new Throwable() {
//					private String _message = exception.getStringValue();
					private String _stackTrace = JSDebugThread.this._stackTrace;

					public void printStackTrace(PrintStream s) {
						s.println(_stackTrace);
					}

					public void printStackTrace(PrintWriter s) {
						s.println(_stackTrace);
					}

					public String getMessage() {
						return "JavaScript stack trace:\n" + _stackTrace;
					}
				};
			}
			if ((flags & REPORT_STRICT) > 0)
				severity = IStatus.INFO;

			try {
				Status status = new Status(severity, MozillaDebugPlugin.ID,
						MozillaDebugPlugin.ERROR_STATUS, message, throwable);
				//TODO: put this in the JavaScript console, perhaps?  In the meantime, send to the error log.
				MozillaDebugPlugin.log(new DebugException(new Status(IStatus.ERROR, MozillaDebugPlugin.ID,
						DebugException.REQUEST_FAILED, "Mozilla JavaScript Exception" /*TODO:i18n*/, throwable)));
				IStatusHandler statusHandler = DebugPlugin.getDefault().getStatusHandler(status);
				//TODO: embellish status line so that you can click to get the stack trace?
				if (statusHandler != null)
					statusHandler.handleStatus(status, this);
			} catch (CoreException ce) {
				MozillaDebugPlugin.log(ce);
			} finally {
				_stackTrace = null;
			}
		} catch (RuntimeException re) {
			// This is called by native code, so exceptions will disappear.  Log them.
			// Most likely, an XPCOMException due to some mismatch with xulrunner.
			MozillaDebugPlugin.log(re);
			throw re;
		} catch (Error e) {
			MozillaDebugPlugin.log(e);
			throw e;
		}

		return resume; /* false to break */
	}

	private String _stackTrace = null;
	private jsdIContext _cx;

	/*
	 * Special Method used when a refresh in the browser is done when
	 * suspended. Allow scripts to run.
	 */
	public void clearCX () {
		if (isSuspended() && _cx != null)
			_cx.setScriptsEnabled(true);
	}

	public long onExecute(jsdIStackFrame frame, long type, jsdIValue[] rv) {
		try {
	//		MozillaDebugPlugin.log("onExecute: type="+type);
	
	//TYPE_INTERRUPTED, TYPE_BREAKPOINT, TYPE_DEBUG_REQUESTED, TYPE_DEBUGGER_KEYWORD, TYPE_THROW
	//RETURN_HOOK_ERROR, RETURN_CONTINUE, RETURN_ABORT, RETURN_RET_WITH_VAL, RETURN_THROW_WITH_VAL, RETURN_CONTINUE_THROW
	
			if (frame.getIsNative())
				return RETURN_CONTINUE;
	
			if (type == TYPE_THROW) {
				if (_stackTrace == null || false) { //TODO: need to determine if this is a new exception?
					// fill in stack frame if this is the first time we've seen this exception
					// If we try to do this after creating JSDebugStackFrames, bad things happen to our
					// state, so we construct this directly from jsdIStackFrame instead
					//TODO: perhaps the trace should be passed around as structured data instead of string for the JS Console?
					StringBuffer trace = new StringBuffer();
					for (jsdIStackFrame f = frame; f != null; f = f.getCallingFrame()) {
						trace.append(f.getFunctionName()); //TODO: need "guess"
						trace.append("()"/*argList*/);
						trace.append(' ');
						boolean isNative = Boolean.valueOf(f.getIsNative()).booleanValue();
						if (isNative) {
							trace.append("[native function]");
						} else {
							jsdIScript script = frame.getScript();
							trace.append(script.getFileName());
							trace.append(':');
							trace.append(f.getLine());
						}
						trace.append('\n');
					}
					_stackTrace = trace.toString();
				}

				if (!_suspendOnExceptions)
					return RETURN_CONTINUE_THROW;
			}

			// Stop scripts from executing while we're suspended
			_cx = frame.getExecutionContext();
			_cx.setScriptsEnabled(false);
	
			long result = RETURN_ABORT;
			nsIBaseWindow window = (nsIBaseWindow)getDebugTarget().getProcess().getAdapter(nsIBaseWindow.class);
			try {
				try {
					// Prevent user from taking further actions within the browser while suspended
					window.setEnabled(false);
				} catch (XPCOMException xpcome) {
					if (xpcome.errorcode != Mozilla.NS_ERROR_NOT_IMPLEMENTED)
						throw xpcome;
				}

				_suspendedType = type;
		//		final jsdIStackFrame aFrame = frame;
		
				switch ((int)type) {
				case (int)TYPE_INTERRUPTED:
					if (isStepping()) {
						if ((_stepPastDepth == getDepth(frame)) &&
						    (_stepPastFrame != null) && (_stepPastFrame.equals(frame.getFunctionName())) &&
						    (_stepPastLine == frame.getLine())) {
//						    System.out.println("stepPast: " + _stepPastFrame + "line: "+_stepPastLine+" depth:"+_stepPastDepth);
						    return RETURN_CONTINUE;
						}
			
				        _stepPastDepth = -1;
					}
			        break;
				case (int)TYPE_BREAKPOINT:
					Vector foundBreakpoints = new Vector();
					//TODO match breakpoints with current line of code
					_breakpoints = (IBreakpoint[]) foundBreakpoints.toArray(new IBreakpoint[0]);
					break;
				default:
				}

				updateFrames(frame);

				if (type == TYPE_THROW) {
					// Add variable reference to exception object
					if (rv.length > 0) {
						jsdIValue exception = rv[0];
						if (exception != null) {
							_exceptionObject = new JSDebugVariable(getDebugTarget(), this,
									"[exception object]", exception);
							JSDebugStackFrame topFrame = (JSDebugStackFrame)_stackFrames[0]; 
							topFrame.addVariable(_exceptionObject);
						}
					}
				}

				result = waitLoop();

				if (_stackFrames != null){
					for (int i = 0; i < _stackFrames.length; i++) {
						JSDebugStackFrame f = (JSDebugStackFrame)_stackFrames[i];
						f.invalidate();
					}
				}
			} finally {
				_suspendedType = TYPE_RUNNING;
				_exceptionObject = null;
				_breakpoints = new IBreakpoint[0];

				// Free up the browser now that we're no longer suspended
				_cx.setScriptsEnabled(true);
				_cx = null;
				try {
					window.setEnabled(true);
				} catch (XPCOMException xpcome) {
					if (xpcome.errorcode != Mozilla.NS_ERROR_NOT_IMPLEMENTED)
						throw xpcome;
				}
			}
			return result;
		} catch (RuntimeException re) {
			// This is called by native code, so exceptions will disappear.  Log them.
			// Most likely, an XPCOMException due to some mismatch with xulrunner.
			MozillaDebugPlugin.log(re);
			throw re;
		} catch (Error e) {
			MozillaDebugPlugin.log(e);
			throw e;
		}
	}

	private long waitLoop() {
		fireSuspendEvent(isStepping() ? DebugEvent.STEP_END : DebugEvent.BREAKPOINT);
		JSDebugTarget target = ((JSDebugTarget)getDebugTarget());
		target.getDebuggerService().setFunctionHook(null);
		fireChangeEvent(DebugEvent.STATE);
		long result = target.enterNestedEventLoop(null/*new JSDebugTarget.INestedCallback() {
			public void onNest() {
//				populateFrames(aFrame);
			}
		}*/);

//System.err.println("nestedEventLoop done.  stepping="+_stepping);
		if (!isStepping() && target.getDebuggerService() != null){
			target.getDebuggerService().setInterruptHook(null);
		}
		
		target.fireChangeEvent(DebugEvent.STATE);
		fireResumeEvent(isStepping() ? DebugEvent.UNSPECIFIED /*TODO*/: DebugEvent.CLIENT_REQUEST);

		return result;
	}

	public void onCall(jsdIStackFrame frame, long type) {
		try {
			MozillaDebugPlugin.log("onCall: type="+type);

		switch ((int)type) {
		case (int)TYPE_TOPLEVEL_START:
			
			if( !_reachedFirstLine ){
				try{
					_reachedFirstLine = true;
					onExecute( frame, TYPE_DEBUGGER_KEYWORD, new jsdIValue[]{} );
				}
				finally{
					((JSDebugTarget)getDebugTarget()).getDebuggerService().setTopLevelHook( null );
				}
			}
			//TODO: optionally break here?
			// target.getDebuggerService().setInterruptHook(null);
			break;

			case (int)TYPE_FUNCTION_CALL: // function enter
				if (isStepping()) {
					JSDebugTarget target = (JSDebugTarget)getDebugTarget();
					// For stepover, since a function call was made, cancel the interrupt and wait
					// for the function return before breaking.
				    target.getDebuggerService().setInterruptHook(null);
				}
				break;

			case (int)TYPE_FUNCTION_RETURN: // function exit
				if (isStepping()) {
			        // we're called *before* the returning frame is popped from the
			        // stack, so we want our depth calculation to be off by one.
			        int depth = getDepth(frame) - 1;

			        //dd ("Returning: " + frame.functionName +
			        //    ", target depth: " + _stepOverDepth +
			        //    ", current depth: " + depth);

			        if (depth <= _stepOverDepth)
			        {
			        	System.out.println("step over at target depth of " + depth);
						JSDebugTarget target = (JSDebugTarget)getDebugTarget();
					    target.getDebuggerService().setInterruptHook(this);
					    target.getDebuggerService().setFunctionHook(null);
			            _stepOverDepth = -1;
			            //TODO: sometimes this leaves us at the end of the calling instruction, rather than at the next line
			        }
				}

				break;
			default:
			}
		} catch (RuntimeException re) {
			// This is called by native code, so exceptions will disappear.  Log them.
			// Most likely, an XPCOMException due to some mismatch with xulrunner.
			MozillaDebugPlugin.log(re);
			throw re;
		} catch (Error e) {
			MozillaDebugPlugin.log(e);
			throw e;
		}
	}

	public void onScriptCreated(jsdIScript script) {
		try {
	//System.err.println("onScriptCreated file="+script.getFileName()+" func="+script.getFunctionName()+" baseline="+script.getBaseLineNumber()+" extent"+script.getLineExtent());
			IBreakpoint[] breakpoints = DebugPlugin.getDefault().getBreakpointManager().getBreakpoints();
			// RSG - change to get source locator from target
			JSSourceLocator locator = ((JSDebugTarget)getDebugTarget()).getSourceLocator();
			if (_breakpointManager.isEnabled()){
				for(int i = 0; i < breakpoints.length; i++) {
					try {
						if (breakpoints[i].isEnabled()) {
							JSDebugTarget.establishBreakpoint(true, locator, script, breakpoints[i]);
						}
					} catch (CoreException e) {
						// Breakpoint doesn't throw this exception
						MozillaDebugPlugin.log(e);
					}
				}
			}


			//TODO: set djconfig.debugAtAllCosts to true for dojo via extension point?

			//create the JSDebugScriptElement
			JSDebugTopScriptElement topScriptElement = null;
			if( topScriptElements.containsKey(script.getFileName()) ){
				topScriptElement = (JSDebugTopScriptElement)topScriptElements.get( script.getFileName() );
			}
			else{
				topScriptElement = new JSDebugTopScriptElement( getDebugTarget() );
				topScriptElement.setName( script.getFileName() );
				topScriptElement.setLocation( script.getFileName() );	

				topScriptElements.put( script.getFileName(), topScriptElement );
			}

			JSDebugScriptElement scriptElement = new JSDebugScriptElement( getDebugTarget() );

			/*
			 * if the function name is empty it represents a top level script area (i.e. <script> tag in an HTML file)
			 * and the name is not set. This name String to display will be handled by the LabelProviderAdapter
			 */
			if( !(script.getFunctionName() == null || "".equals(script.getFunctionName())) ){
				scriptElement.setName( script.getFunctionName() );
			}
			
			scriptElement.setLineStart( (int)script.getBaseLineNumber() );
			scriptElement.setLineTotal( (int)script.getLineExtent() );
			
			topScriptElement.insert( scriptElement );
		} catch (RuntimeException re) {
			// This is called by native code, so exceptions will disappear.  Log them.
			// Most likely, an XPCOMException due to some mismatch with xulrunner.
			MozillaDebugPlugin.log(re);
			throw re;
		} catch (Error e) {
			MozillaDebugPlugin.log(e);
			throw e;
		}
	}
	
	
	
	public IJSDebugScriptElement[] getTopScriptElements(){
		IJSDebugScriptElement[] array = new IJSDebugScriptElement[ topScriptElements.size() ];
		topScriptElements.values().toArray( array );
		return array;
	}

	public void onScriptDestroyed(jsdIScript script) {
		try {
//			System.err.println("onScriptDestroyed file="+script.getFileName()+" func="+script.getFunctionName());
		} catch (RuntimeException re) {
			// This is called by native code, so exceptions will disappear.  Log them.
			// Most likely, an XPCOMException due to some mismatch with xulrunner.
			MozillaDebugPlugin.log(re);
			throw re;
		} catch (Error e) {
			MozillaDebugPlugin.log(e);
			throw e;
		}
	}

	private void updateFrames(jsdIStackFrame topFrame) {
//		System.err.println("STACK FRAME: "+topFrame.getFunctionName()+":"+topFrame.getLine()+" file="+topFrame.getScript().getFileName());
//		List frames = new ArrayList();
//		int depth = getDepth(topFrame);
//		jsdIStackFrame frame = topFrame;
//	    while (frame != null)
//	    {
//	        frames.add(new JSDebugStackFrame(getDebugTarget(), this, this, frame, depth--));
//	        frame = frame.getCallingFrame();
//	    }
//
//		_stackFrames  = (IStackFrame[])frames.toArray(new IStackFrame[0]);

		// Create a new call stack to reflect the chain pointed to by topFrame,
		// but preserve JSDebugStackFrames which persist from the previous call as appropriate

		IStackFrame[] newStack = _emptyStack;

		try {
			final jsdIStackFrame[] newFrames = getFullStack(topFrame);
			final int newLength = newFrames.length;

			// Starting at the bottom of both stacks, keep those which are identical
			int i = _stackFrames.length - 1;
			for (int j = newFrames.length - 1; i >= 0 && j >= 0; i--, j--) {
				JSDebugStackFrame oldFrame = (JSDebugStackFrame)_stackFrames[i];
				jsdIStackFrame newFrame = newFrames[j];

				if (!oldFrame.isUnchangedFrom(newFrame)) {
					if (oldFrame.isSameAs(newFrame)) {
						// Frame is the same, but the pc moved, so include one more
						i--;

						// And refresh its data
						oldFrame.validate(newFrame);
					}
					break;
				}

				// Provide a fresh XPCOM frame pointer and refresh data in the cached object as necessary.
				// This will also update the children (variables) which may have changed since the last break.
				// Even though the unchanged stack frames should not have had any variable changes, we cannot
				// guarantee that the frames are identical, so refresh anyway.
				oldFrame.validate(newFrame);
			}

//System.err.println("call frames the same down to i="+i);

			// If all stack frames are the same, then we're done.
			if ((i < 0) && (_stackFrames.length == newLength))
				return;

			// Allocate a new array of the right size
			newStack = new IStackFrame[newLength];

			// Copy over the shared frame objects
			int n = newLength - 1;
			for (int m = _stackFrames.length - 1; m > i; m--, n--) {
				newStack[n] = _stackFrames[m];
			}

			// Now add new frames, if any
			for (; n >= 0; n--) {
				newStack[n] = new JSDebugStackFrame(getDebugTarget(), this, newFrames[n], newLength - 1 - n, _reachedFirstLine);
			}
		} catch (RuntimeException re) {
			// Under unexpected failures, like XPCOMExceptions, be sure to still set the debugger state
			// and follow through with the wait loop
			re.printStackTrace();
			MozillaDebugPlugin.log(re);
		}

		_stackFrames = newStack;
	}

	private int getDepth(jsdIStackFrame frame) {
		int depth = -1;

		while (frame != null) {
			frame = frame.getCallingFrame();
			depth++;
		}

		return depth;
	}

	private jsdIStackFrame[] getFullStack(jsdIStackFrame topFrame) {
		List frames = new ArrayList();

		for (jsdIStackFrame frame = topFrame; frame != null; frame = frame.getCallingFrame()) {
			frames.add(frame);
		}

		return (jsdIStackFrame[])frames.toArray(new jsdIStackFrame[0]);
	}
	
	private boolean isCorrectContext(jsdIStackFrame frame) {

		nsIWebBrowser browser = (nsIWebBrowser)getDebugTarget().getProcess().getAdapter(nsIWebBrowser.class);
		nsIDOMWindow window = browser.getContentDOMWindow();
		nsIDOMDocument document = window.getDocument();
		nsIDOMNSDocument domNSDoc = (org.mozilla.interfaces.nsIDOMNSDocument)document.queryInterface(org.mozilla.interfaces.nsIDOMNSDocument.NS_IDOMNSDOCUMENT_IID);
		nsIDOMLocation location = domNSDoc.getLocation();

		jsdIContext context = frame.getExecutionContext();
		jsdIValue globalObject = context.getGlobalObject();
		jsdIProperty prop = globalObject.getProperty("location");
		jsdIValue value = prop.getValue();
		String frameLoc = "";
		System.out.println("Window href: " + location.getHref());
		if (value.getIsValid()){
			frameLoc = value.getStringValue();
			System.out.println("Frame Location: " + frameLoc);
			if(frameLoc.equalsIgnoreCase(location.getHref()))return true;
		}
		
		return false;
	}
}
