/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/
package org.eclipse.atf.mozilla.ide.ui.browser;

import org.eclipse.atf.mozilla.ide.ui.MozIDEUIPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.swt.browser.LocationEvent;
import org.eclipse.swt.browser.LocationListener;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.browser.CloseWindowListener;
import org.eclipse.swt.browser.WindowEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.widgets.Display;
import org.mozilla.interfaces.nsIBaseWindow;
import org.mozilla.interfaces.nsIWebBrowser;


/**
 * Implements an IProcess which is associated with a MozillaBrowser.
 * Watches for changes which imply that the original debugging
 * session is no longer active: closing of the browser or changing
 * to another page.  Also able to terminate the browser by triggering
 * a close event which the browser's container (e.g. an EditorPart)
 * will handle.
 * 
 * Could track new browser's spawned by the original browser someday,
 * if that is helpful.
 * 
 * @author peller
 *
 */
public class MozBrowserProcess extends PlatformObject implements IProcess {
	private Browser _browser;
	private CloseWindowListener _closeListener;
	private DisposeListener _disposeListener;
	private LocationListener _locationListener;
	private ILaunch _launch;
	private int _exitValue = 0;
	private boolean _terminated;

	public MozBrowserProcess(Browser browser, ILaunch launch) {
		_browser = browser;
		_launch = launch;
		launch.addProcess(this);
		fireCreationEvent();
		
		_closeListener = new CloseWindowListener() {
			public void close(WindowEvent event) {
				processTermination();
			}
		};
		
		browser.addCloseWindowListener(_closeListener);
		
		_disposeListener = new DisposeListener() {

			public void widgetDisposed(DisposeEvent e) {
				processTermination();				
			}
		};
		
		browser.addDisposeListener(_disposeListener);
		
		//only attach the LocationListner if in DegugMode (ProfileMode needs this also)
		//When doing a run is does not matter if the browser changes
		//to another location.
		if( ILaunchManager.DEBUG_MODE.equals(launch.getLaunchMode()) || ILaunchManager.PROFILE_MODE.equals(launch.getLaunchMode())){
			_locationListener = new LocationListener() {
				public void changed(LocationEvent event) {
					//System.err.println("LocationEvent changed");
					
				}
	
				public void changing(LocationEvent event) {
					//System.err.println("LocationEvent changing");
					//IGNORE ALL CHANGING EVENTS THAT ARE CAUSED BY CLICKING ON AN ANCHOR WITH A JAVASCRIPT URL
					if( event.location.startsWith("javascript:") )
						return;
					processLocationChange();
				}
			};
					
			browser.addLocationListener(_locationListener);
		}
	}

	public boolean canTerminate() {
		return !isTerminated();
	}

	public boolean isTerminated() {
		return _terminated;
	}

	public void terminate() throws DebugException {
		if (isTerminated())
			return;

		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				Browser browser = _browser;
				processTermination();
				browser.dispose();
			}
		});
	}

	protected void processLocationChange() {		
		DebugEvent event = new DebugEvent(getLaunch(), DebugEvent.MODEL_SPECIFIC);
		event.setData("locationChange");
		fireEvent(event); 
	}
	
	protected void processTermination() {
		if (_browser != null && !_browser.isDisposed()) {
			if( _closeListener != null )
				_browser.removeCloseWindowListener(_closeListener);
		
			if( _locationListener != null )
				_browser.removeLocationListener(_locationListener);
		}
		fireTerminateEvent();
		_browser = null;
		_closeListener = null;
		_locationListener = null;

		_terminated = true;
	}

	public Object getAdapter(Class adapter) {
		if (adapter.equals(IProcess.class)) {
			return this;
		}
		if (adapter.equals(IDebugTarget.class)) {
			ILaunch launch = getLaunch();
			IDebugTarget[] targets = launch.getDebugTargets();
			for (int i = 0; i < targets.length; i++) {
				if (this.equals(targets[i].getProcess())) {
					return targets[i];
				}
			}
			return null;
		}
		if (nsIBaseWindow.class.equals(adapter)) {
			return (nsIBaseWindow)((nsIWebBrowser)_browser.getWebBrowser()).queryInterface(nsIBaseWindow.NS_IBASEWINDOW_IID);
		}
		if (nsIWebBrowser.class.equals(adapter)) {
			return _browser.getWebBrowser();
		}
		if (Browser.class.equals(adapter))
			return _browser;

		return super.getAdapter(adapter);
	}

	public String getAttribute(String key) {
		return null;
	}

	public int getExitValue() throws DebugException {
		if (!isTerminated())
			throw new DebugException(new Status(IStatus.ERROR,
					MozIDEUIPlugin.ID,
					IStatus.OK,
					"process still running. exit value unknown",
					null));
		return _exitValue;
	}

	public String getLabel() {
		return "Mozilla Browser"; //TODO
	}

	public ILaunch getLaunch() {
		return _launch;
	}

	public IStreamsProxy getStreamsProxy() {
		return null;
	}

	public void setAttribute(String key, String value) {
	}

	/**
	 * Fires a creation event.
	 */
	protected void fireCreationEvent() {
		fireEvent(new DebugEvent(this, DebugEvent.CREATE));
	}

	/**
	 * Fires the given debug event.
	 * 
	 * @param event debug event to fire
	 */
	protected void fireEvent(DebugEvent event) {
		DebugPlugin manager= DebugPlugin.getDefault();
		if (manager != null) {
			manager.fireDebugEventSet(new DebugEvent[]{event});
		}
	}

	/**
	 * Fires a terminate event.
	 */
	protected void fireTerminateEvent() {
		DebugEvent event = new DebugEvent(this, DebugEvent.TERMINATE);
		event.setData(_browser);
		fireEvent(event);
	}

	/**
	 * Fires a change event.
	 */	
	protected void fireChangeEvent() {
		fireEvent(new DebugEvent(this, DebugEvent.CHANGE));
	}
	
};