package org.eclipse.atf.mozilla.ide.ui.css;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.atf.mozilla.ide.ui.browser.IWebBrowser;
import org.eclipse.atf.mozilla.ide.ui.browser.SelectionBox;
import org.eclipse.atf.mozilla.ide.ui.common.event.IDOMMutationListener;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeListener;
import org.eclipse.core.runtime.jobs.Job;
import org.mozilla.interfaces.inIDOMUtils;
import org.mozilla.interfaces.nsIDOMCSSRule;
import org.mozilla.interfaces.nsIDOMCSSStyleDeclaration;
import org.mozilla.interfaces.nsIDOMCSSStyleRule;
import org.mozilla.interfaces.nsIDOMDocument;
import org.mozilla.interfaces.nsIDOMElement;
import org.mozilla.interfaces.nsIDOMNode;
import org.mozilla.interfaces.nsIDOMNodeList;
import org.mozilla.interfaces.nsISupportsArray;
import org.mozilla.xpcom.Mozilla;

/**
 * Locates and highlights selected CSS properties
 * @author Kevin Sawicki (ksawicki@us.ibm.com)
 *
 */
public class CSSPropertyLocator implements IDOMMutationListener {
	
	private nsIDOMDocument document;
	private boolean on = false;
	private List boxes = new ArrayList();
	private CSSJobSearch cts;
	private IWebBrowser container;
	private nsIDOMNode currNode;
	private nsIDOMElement currElement;
	
	private List removed = new ArrayList();
	
	private class CSSJobSearch extends Job {
		
		private nsIDOMNode node;
		private inIDOMUtils service;
		private Map cssProperties = new HashMap();
		private Map rulesToElements = new HashMap();
		private Map rulesToProperties = new HashMap();
		
		public CSSJobSearch(nsIDOMNode node) {
			super("Indexing CSS for page");
			this.node = node;
		}
		
		protected IStatus run(IProgressMonitor monitor) {
			rulesToElements.clear();
			rulesToProperties.clear();
			cssProperties.clear();
			service = (inIDOMUtils)Mozilla.getInstance().getServiceManager().getServiceByContractID("@mozilla.org/inspector/dom-utils;1", inIDOMUtils.INIDOMUTILS_IID);
			parse(node);
			return Status.OK_STATUS;
		}
		
		private void parse( nsIDOMNode node ) {
			try {
				if( node.getNodeType() == nsIDOMNode.ELEMENT_NODE ){
					nsIDOMElement e = (nsIDOMElement)node.queryInterface(nsIDOMElement.NS_IDOMELEMENT_IID);
					nsISupportsArray rules = null;
					if( e != null && service != null ) {
						rules = service.getCSSStyleRules(e);
					}
					if( rules != null ) {
						long numRules = rules.count();
						nsIDOMCSSStyleDeclaration styleDec = null;
						for( long i = 0; i < numRules; i++ ) {
							nsIDOMCSSRule ret;
							ret = (nsIDOMCSSRule)(rules.getElementAt(i).queryInterface(nsIDOMCSSRule.NS_IDOMCSSRULE_IID));
							nsIDOMCSSStyleRule styleRule = (nsIDOMCSSStyleRule)ret.queryInterface(nsIDOMCSSStyleRule.NS_IDOMCSSSTYLERULE_IID);
							styleDec = styleRule.getStyle();
							String lineNumber = Long.toString(service.getRuleLine(styleRule));
							String url = styleRule.getParentStyleSheet().getHref();
							for( long j = 0; j < styleDec.getLength(); j++ ) {
								String name = styleDec.item(j);
								String value = styleDec.getPropertyValue(name);
								CSSProperty  cssProperty = new CSSProperty( name, value, url , lineNumber, styleRule.getSelectorText() );
								if( rulesToProperties.containsKey(cssProperty.getURL()+cssProperty.getRule()) ) {
									List elements = (List)rulesToProperties.get(cssProperty.getURL()+cssProperty.getRule());
									if( !elements.contains(e) ) {
										elements.add(e);
									}
								} else {
									List elements = new ArrayList();
									elements.add(e);
									rulesToProperties.put(cssProperty.getURL()+cssProperty.getRule(), elements);
								}
								if( rulesToElements.containsKey(cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName()) ) {
									List elements = (List)rulesToElements.get(cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
									elements.add(e);
								} else {
									List elements = new ArrayList();
									elements.add(e);
									rulesToElements.put(cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName(), elements);
								}
								if( cssProperties.containsKey(cssProperty.getName()) ) {
									Map elementsToRules = (Map)cssProperties.get(cssProperty.getName());
									elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
								} else {
									Map elementsToRules = new HashMap();
									elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
									cssProperties.put(cssProperty.getName(),elementsToRules);
								}
							}
						}
					}
					String inlineStyle = e.getAttribute("style");
					if( inlineStyle != null && !inlineStyle.equals("") ) {
						StringTokenizer token = new StringTokenizer(inlineStyle,";");
						if( token.hasMoreTokens() ) {
							while( token.hasMoreTokens() ) {
								String cssDec = (String)token.nextToken();
								if( cssDec.indexOf(':') != -1 ) {
									String name = cssDec.substring(0,cssDec.indexOf(':'));
									name = name.trim();
									String value = cssDec.substring(cssDec.indexOf(':')+1,cssDec.length());
									value = value.trim();
									CSSProperty cssProperty = new CSSProperty(name,value,"","",CSSContentProvider.INLINE_STYLE);
									cssProperty.setInline(true);
									cssProperty.setPresent(true);
									cssProperty.setProperty(true);
									if( cssProperties.containsKey(cssProperty.getName()) ) {
										Map elementsToRules = (Map)cssProperties.get(cssProperty.getName());
										elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
									} else {
										Map elementsToRules = new HashMap();
										elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
										cssProperties.put(cssProperty.getName(),elementsToRules);
									}
								}
							}
						}
					}
					nsIDOMNodeList list = e.getChildNodes();
					for( int i = 0; i < list.getLength(); i++ ) {
						parse( list.item(i) );
					}
				}
			} catch( Exception e ) {
				e.printStackTrace();
			}
		}
		
	}
	
	public void add( nsIDOMNode node ) {
		if( node.getNodeType() == nsIDOMNode.ELEMENT_NODE ) {
			try{
				nsIDOMElement e = (nsIDOMElement)node.queryInterface(nsIDOMElement.NS_IDOMELEMENT_IID);
				inIDOMUtils service = (inIDOMUtils)Mozilla.getInstance().getServiceManager().getServiceByContractID("@mozilla.org/inspector/dom-utils;1", inIDOMUtils.INIDOMUTILS_IID);
				nsISupportsArray rules = service.getCSSStyleRules(e);
				long numRules = rules.count();
				nsIDOMCSSStyleDeclaration styleDec = null;
				for( long i = 0; i < numRules; i++ ) {
					nsIDOMCSSRule ret;
					ret = (nsIDOMCSSRule)(rules.getElementAt(i).queryInterface(nsIDOMCSSRule.NS_IDOMCSSRULE_IID));
					nsIDOMCSSStyleRule styleRule = (nsIDOMCSSStyleRule)ret.queryInterface(nsIDOMCSSStyleRule.NS_IDOMCSSSTYLERULE_IID);
					styleDec = styleRule.getStyle();
					String lineNumber = Long.toString(service.getRuleLine(styleRule));
					String url = styleRule.getParentStyleSheet().getHref();
					for( long j = 0; j < styleDec.getLength(); j++ ) {
						String name = styleDec.item(j);
						String value = styleDec.getPropertyValue(name);
						CSSProperty  cssProperty = new CSSProperty( name, value, url , lineNumber, styleRule.getSelectorText() );
						if( cts.rulesToProperties.containsKey(cssProperty.getURL()+cssProperty.getRule()) ) {
							List elements = (List)cts.rulesToProperties.get(cssProperty.getURL()+cssProperty.getRule());
							if( !elements.contains(e) ) {
								elements.add(e);
							}
						} else {
							List elements = new ArrayList();
							elements.add(e);
							cts.rulesToProperties.put(cssProperty.getURL()+cssProperty.getRule(), elements);
						}
						if( cts.rulesToElements.containsKey(cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName()) ) {
							List elements = (List)cts.rulesToElements.get(cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
							elements.add(e);
						} else {
							List elements = new ArrayList();
							elements.add(e);
							cts.rulesToElements.put(cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName(), elements);
						}
						if( cts.cssProperties.containsKey(cssProperty.getName()) ) {
							Map elementsToRules = (Map)cts.cssProperties.get(cssProperty.getName());
							elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
						} else {
							Map elementsToRules = new HashMap();
							elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
							cts.cssProperties.put(cssProperty.getName(),elementsToRules);
						}
					}
				}
			} catch( Exception e ) {
				e.printStackTrace();
			}
		}
	}
	
	public void remove( nsIDOMNode node ) {
		if( node.getNodeType() == nsIDOMNode.ELEMENT_NODE ) {
			removed.add(node);
		}
	}
	
	public void load( nsIDOMNode node, IJobChangeListener listener ) {
		if( cts != null ) {
			cts.cancel();
		}
		cts = new CSSJobSearch(node);
		cts.setPriority(Job.BUILD);
		cts.addJobChangeListener(listener);
		cts.schedule();
	}
	
	public void disable() {
		Iterator iter = boxes.iterator();
		while( iter.hasNext() ) {
			SelectionBox box = (SelectionBox)iter.next();
			box.hide();
		}
		if( cts != null ) {
			cts.cancel();
		}
		boxes.clear();
		on = false;
	}
	
	public void match( CSSProperty property ) {
		if( on ) {
			disable();
		} else {
			if( cts.getResult() == null ) {
				return;
			}
			if( !property.isRule() && !property.isInline() ) {
				List elements = (List)cts.rulesToElements.get(property.getURL()+property.getRule()+property.getName());
				if( elements != null ) {
					Iterator iter = elements.iterator();
					while( iter.hasNext() ) {
						nsIDOMElement e = (nsIDOMElement)iter.next();
						Map elementsToRules = (Map)cts.cssProperties.get(property.getName());
						if( elementsToRules != null && elementsToRules.get(e) != null ) {
							String rule = property.getURL()+property.getRule()+property.getName();
							if( rule.equals(((String)elementsToRules.get(e))) && !removed.contains(e)) {
								SelectionBox box = new SelectionBox(document);
								box.highlight(e, "#00FF00");
								boxes.add(box);
							}
						}
					}
				}
			} else if( !property.isInline() ){
				List elements = (List)cts.rulesToProperties.get(property.getURL()+property.getRule());
				if( elements != null ) {
					Iterator iter = elements.iterator();
					while( iter.hasNext() ) {
						nsIDOMElement e = (nsIDOMElement)iter.next();
						if( !removed.contains(e)) {
							SelectionBox box = new SelectionBox(document);
							box.highlight(e, "#00FF00");
							boxes.add(box);
						}
					}
				}
			} else {
				if( !removed.contains(currNode)) {
					SelectionBox box = new SelectionBox(document);
					box.highlight((nsIDOMElement)(currNode.queryInterface(nsIDOMElement.NS_IDOMELEMENT_IID)), "#00FF00");
					boxes.add(box);
				}
			}
			on = true;
		}
	}
	
	public void addAttribute( nsIDOMNode node ) {
		if( node.getNodeType() == nsIDOMNode.ELEMENT_NODE ) {
			try{
				nsIDOMElement e = (nsIDOMElement)node.queryInterface(nsIDOMElement.NS_IDOMELEMENT_IID);
				String inlineStyle = e.getAttribute("style");
				if( inlineStyle != null && !inlineStyle.equals("") ) {
					StringTokenizer token = new StringTokenizer(inlineStyle,";");
					if( token.hasMoreTokens() ) {
						while( token.hasMoreTokens() ) {
							String cssDec = (String)token.nextToken();
							if( cssDec.indexOf(':') != -1 ) {
								String name = cssDec.substring(0,cssDec.indexOf(':'));
								name = name.trim();
								String value = cssDec.substring(cssDec.indexOf(':')+1,cssDec.length());
								value = value.trim();
								CSSProperty cssProperty = new CSSProperty(name,value,"","",CSSContentProvider.INLINE_STYLE);
								cssProperty.setInline(true);
								cssProperty.setPresent(true);
								cssProperty.setProperty(true);
								if( cts.cssProperties.containsKey(cssProperty.getName()) ) {
									Map elementsToRules = (Map)cts.cssProperties.get(cssProperty.getName());
									elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
								} else {
									Map elementsToRules = new HashMap();
									elementsToRules.put(e, cssProperty.getURL()+cssProperty.getRule()+cssProperty.getName());
									cts.cssProperties.put(cssProperty.getName(),elementsToRules);
								}
							}
						}
					}
				}
			} catch( Exception e ) {
				e.printStackTrace();
			}
		}
	}
	
	public void removeAttribute( nsIDOMElement domElement ) {
		add( domElement );
	}
	
	public void setDocumentContainer( IWebBrowser doc ) {
		this.document = doc.getDocument();
		if( this.container != null ) {
			this.container.removeDOMMutationListener(this);
		}
		this.container = doc;
		this.container.addDOMMutationListener(this);
	}
	
	public void setNode( nsIDOMNode node ) {
		this.currNode = node;
		currElement = (nsIDOMElement)(currNode.queryInterface(nsIDOMElement.NS_IDOMELEMENT_IID));
	}

	public void attributeAdded(nsIDOMElement ownerElement, String attributeName) {
		if( currElement != null && currElement.equals(ownerElement) && attributeName.equals("style")) {
			addAttribute(ownerElement);
		}
	}

	public void attributeModified(nsIDOMElement ownerElement, String attributeName, String newValue, String previousValue) {
		if( currElement != null && currElement.equals(ownerElement) && attributeName.equals("style")) {
			addAttribute(ownerElement);
		}
	}

	public void attributeRemoved(nsIDOMElement ownerElement, String attributeName) {
		if( currElement != null && currElement.equals(ownerElement) && attributeName.equals("style") ) {
			removeAttribute(ownerElement);
		}
	}

	public void nodeInserted(nsIDOMNode parentNode, nsIDOMNode insertedNode) {
		add(insertedNode);
		addAttribute(insertedNode);
	}

	public void nodeRemoved(nsIDOMNode parentNode, nsIDOMNode removedNode) {
		remove(removedNode);
	}
	
}