/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2007 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/
package org.eclipse.atf.mozilla.ide.ui.netmon;

import org.eclipse.atf.mozilla.ide.ui.browser.IWebBrowser;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.INetworkCallList;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.impl.MozHTTPCall;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.impl.MozXHRCall;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.impl.NetworkCallListImpl;
import org.mozilla.interfaces.nsIChannel;
import org.mozilla.interfaces.nsIDOMEvent;
import org.mozilla.interfaces.nsIDOMEventListener;
import org.mozilla.interfaces.nsIDOMWindow;
import org.mozilla.interfaces.nsIHttpChannel;
import org.mozilla.interfaces.nsIInterfaceRequestor;
import org.mozilla.interfaces.nsIJSXMLHttpRequest;
import org.mozilla.interfaces.nsIObserver;
import org.mozilla.interfaces.nsIObserverService;
import org.mozilla.interfaces.nsIOnReadyStateChangeHandler;
import org.mozilla.interfaces.nsIRequest;
import org.mozilla.interfaces.nsISupports;
import org.mozilla.interfaces.nsIURI;
import org.mozilla.interfaces.nsIWebBrowser;
import org.mozilla.interfaces.nsIWebProgress;
import org.mozilla.interfaces.nsIWebProgressListener;
import org.mozilla.interfaces.nsIXMLHttpRequest;
import org.mozilla.xpcom.Mozilla;
import org.mozilla.xpcom.XPCOMException;

/**
 * This class is used to listen to all the network communication done by the
 * browser.
 * 
 * Each Browser Editor that is opened in the Eclipse environment will have its
 * own instance of this class.
 * 
 * @author Gino Bustelo
 *
 */
public class MozNetworkMonitorAdapter implements INetworkMonitorAdapter, nsIObserver, nsIWebProgressListener{

	protected static final String HTTP_ON_MODIFY_REQUEST_TOPIC = "http-on-modify-request";
	protected static final String HTTP_ON_EXAMINE_RESPONSE_TOPIC = "http-on-examine-response";
	
	/*
	 * This inner class is used to track the response of an XHR Call. It is implemented
	 * as an inner class because it is meant to be used only internally and it has
	 * access to the Adapter's (the host class) handler methods.
	 * 
	 * FOR ASYNC CALLS:
	 * It straps itself to the nsIXMLHTTPRequest
	 * as a nsIOnReadyStateChangeHandler to keep track of state changes of the call.
	 * 
	 * It wraps any existing nsIOnReadyStateChangeHandler and relays events to it. This
	 * is the only way that we can make sure that this object is the first one to get
	 * notified of any changes to the Call.
	 * 
	 * NOTE: This could be risky since this object is using an interface
	 * (nsIOnReadyStateChangeHandler) that is only meant to be used by JavaScript.
	 * 
	 * FOR SYNC CALLS:
	 * It straps itself to the nsIXMLHTMLRequest as a nsIDOMEventListener to the
	 * onerror and onload handlers (like in JavaScript). 
	 * 
	 * If a handler for either exists, the it is wrapped and the event is relayed
	 * after processing the event in the monitor.
	 * 
	 * By looking at the nsXMLHTTPRequest.cpp source (http://lxr.mozilla.org/mozilla/source/extensions/xmlextras/base/src/nsXMLHttpRequest.cpp#1442)
	 * the order of notifications when the call is completed is as follows:
	 * 
	 * - nsIOnReadyStateChangeHandler (onreadystate handler) (only async)
	 * - onload, onerror and onprogress handlers (JavaScript)
	 * - nsIDOMEventListener for load and error
	 */
	protected class XHRHandler implements nsIOnReadyStateChangeHandler, nsIDOMEventListener{

		//Mozilla XHR object
		protected nsIXMLHttpRequest xhr = null;
		
		/*
		 * if the is already an nsIOnReadyStateChangeHandler set on the nsIJSXMLHTTPRequest, then
		 * we keep a reference and wrap it.
		 * 
		 * This is only usable for an ASYNC call
		 */
		protected nsIOnReadyStateChangeHandler wrappedOnReadyStateHandler = null;
		
		/*
		 * if there is already an onerror and onload handlers, then we keep a reference and wrap
		 */
		protected nsIDOMEventListener wrappedOnErrorHandler = null;
		protected nsIDOMEventListener wrappedOnLoadHandler = null;
		
		public XHRHandler( nsIXMLHttpRequest xhr ){
			//cache the xhr to be able to query on ready-state changes
			this.xhr = xhr;	
			
			//Setting this object as a nsIOnReadyStateChangeHandler
			nsIJSXMLHttpRequest jsXHR = (nsIJSXMLHttpRequest)xhr.queryInterface( nsIJSXMLHttpRequest.NS_IJSXMLHTTPREQUEST_IID );
			
			
			/*
			 * since the nsIOnReadyStateChangeHandler is used for ASYNC call, I'm using
			 * its presence to determine that the call is ASYNC (maybe there is a 
			 * better way.
			 * 
			 * Otherwise, we hook to the onload and onerror handlers.
			 */
			
			if( jsXHR.getOnreadystatechange() != null ){
				//Async call and need to do this so that we intercept state before JavaScript handler
				
				//getting the already set handler in order to wrap (could be null)
				this.wrappedOnReadyStateHandler = jsXHR.getOnreadystatechange();
				
				jsXHR.setOnreadystatechange( this );
			
			}
			else{
				/*
				 * Could be either Async or Synch but since there is no onreadystate handler,
				 * it is safe to hook to the regular onerror and onprogress handlers.
				 * 
				 * Wrap existing handlers
				 */
				
				this.wrappedOnLoadHandler = jsXHR.getOnload();
				this.wrappedOnErrorHandler = jsXHR.getOnerror();
				
				jsXHR.setOnload( this );
				jsXHR.setOnerror( this );
			}
		}
		
		protected final static int COMPLETE_READYSTATE = 4;
		protected final static int OK_STATUS = 200;
		/*
		 * @see org.mozilla.xpcom.nsIOnReadyStateChangeHandler#handleEvent()
		 */
		public void handleEvent() {

			try{
				//only care about completed requests
				if( xhr.getReadyState() == COMPLETE_READYSTATE ){  //4 means complete
					
					try{
						handleXHRResponse( xhr, false );
					}
					finally{
						xhr = null;
					}
				}

			}
			/*
			 * this outer try/finally is to protect against exceptions in the Eclipse side
			 * stopping the proper execution of the Web Application. Always make sure
			 * to notify the JS handler if it exists
			 */
			finally{
				//relay to wrapped
				if( wrappedOnReadyStateHandler != null )
					wrappedOnReadyStateHandler.handleEvent();
			}
		}

		protected static final String LOAD_TYPE = "load";
		protected static final String ERROR_TYPE = "error";
		/*
		 *  (non-Javadoc)
		 * @see org.mozilla.xpcom.nsIDOMEventListener#handleEvent(org.mozilla.xpcom.nsIDOMEvent)
		 */
		public void handleEvent(nsIDOMEvent event) {
			//this is always called after the XHR is complete (either an error or load)
			try{

				//check the type of event
				handleXHRResponse(xhr, ERROR_TYPE.equals(event.getType()) );	

			}
			/*
			 * this outer try/finally is to protect against exceptions in the Eclipse side
			 * stopping the proper execution of the Web Application. Always make sure
			 * to notify the JS handler if it exists
			 */
			finally{
				
				xhr = null;
				//relay to wrapped
				if( wrappedOnLoadHandler != null && LOAD_TYPE.equals(event.getType()) )
					wrappedOnLoadHandler.handleEvent(event);

				if( wrappedOnErrorHandler != null && ERROR_TYPE.equals(event.getType()) )
					wrappedOnErrorHandler.handleEvent(event);
			}

		}

		public nsISupports queryInterface(String id) {
			return Mozilla.queryInterface(this, id);
		}		
	}
	
	
	/*
	 * Collects all the ongoing and finished network calls.
	 */
	protected INetworkCallList callList;
	
	/*
	 * This is the context that this observer cares about. The nsIObserver interface is
	 * added into a global topic, so we need some sort of context to filter out the relevant
	 * events.
	 */
	protected IWebBrowser browserContext = null;
	
	public MozNetworkMonitorAdapter( IWebBrowser browserContext ){
		this.browserContext = browserContext;
		this.callList = new NetworkCallListImpl();
	}
	
	private boolean connected = false;
	
	/**
	 * hook to mozilla
	 * 
	 * there are two listeners,
	 * 
	 * webprogresslistener gets everything but not the XHR.
	 * obsever to request topic is used to get to XHR calls.
	 */
	public void connect(){
		if( !connected ){
			
			//web progress connection
			nsIWebBrowser mozBrowser = (nsIWebBrowser)browserContext.getAdapter( nsIWebBrowser.class );
			mozBrowser.addWebBrowserListener( this, nsIWebProgressListener.NS_IWEBPROGRESSLISTENER_IID );
			
			//observer connection
			nsIObserverService observerService = (nsIObserverService) Mozilla.getInstance().getServiceManager().getServiceByContractID( "@mozilla.org/observer-service;1", nsIObserverService.NS_IOBSERVERSERVICE_IID );
			observerService.addObserver( this, HTTP_ON_MODIFY_REQUEST_TOPIC, false );
			observerService.addObserver( this, HTTP_ON_EXAMINE_RESPONSE_TOPIC, false );
			
			connected = true;
		}
	}
	
	/**
	 * unhook from mozilla
	 */
	public void disconnect(){
		if( connected ){
			
			//web progress connection
			nsIWebBrowser mozBrowser = (nsIWebBrowser)browserContext.getAdapter( nsIWebBrowser.class );
			mozBrowser.removeWebBrowserListener( this, nsIWebProgressListener.NS_IWEBPROGRESSLISTENER_IID );
			
			//observer connection
			nsIObserverService observerService = (nsIObserverService) Mozilla.getInstance().getServiceManager().getServiceByContractID( "@mozilla.org/observer-service;1", nsIObserverService.NS_IOBSERVERSERVICE_IID );
			observerService.removeObserver( this, HTTP_ON_MODIFY_REQUEST_TOPIC );
			observerService.removeObserver( this, HTTP_ON_EXAMINE_RESPONSE_TOPIC );
			
			connected = false;
		}
	}
	
	public INetworkCallList getCallList(){
		return callList;
	}
	
	/**
	 * There is one observer registered for each browser open. Since the observers are global, need to check if the notification
	 * is within the context of this listener.
	 */
	public void observe(nsISupports subject, String topic, String data) {
		
		try{
			//need to filter for this browser context
			nsIRequest request = (nsIRequest)subject.queryInterface( nsIRequest.NS_IREQUEST_IID );
			//System.err.println( "MozNetworkMonitorAdaptre:observe() -- " + request.getName() );
			
			//top window for the request
			nsIWebProgress webProgress = null;
			
			try{
				webProgress = (nsIWebProgress)request.getLoadGroup().getGroupObserver().queryInterface( nsIWebProgress.NS_IWEBPROGRESS_IID );
			}
			catch( Exception e ){
				//first try to get webProgress failed, try a different way
				
				nsIChannel channel = (nsIChannel)request.queryInterface( nsIChannel.NS_ICHANNEL_IID );
				
				webProgress = (nsIWebProgress)channel.getNotificationCallbacks().getInterface( nsIWebProgress.NS_IWEBPROGRESS_IID );
			}
			
			
			nsIDOMWindow requestOwnerWindow = webProgress.getDOMWindow().getTop();
			
			nsIDOMWindow contextWindow = (nsIDOMWindow)browserContext.getAdapter(nsIDOMWindow.class);
			
			if( requestOwnerWindow != contextWindow ){
				return; //ignore this notification
			}
			
			if( HTTP_ON_MODIFY_REQUEST_TOPIC.equals(topic) ){
				observeRequest( request );
			}
			
			else if( HTTP_ON_EXAMINE_RESPONSE_TOPIC.equals(topic) ){
				observeResponse( request );
			}
		}
		catch( Exception e ){
			//errors in QI are ignored
			e.printStackTrace();
		}
	}
	
	/*
	 * Handle the request side for the observer
	 * 
	 */
	protected void observeRequest( nsIRequest request ){
		//System.err.println( "MozNetworkMonitorAdaptre:observeRequest() -- " + request.getName() );
		
		//LOAD_BACKGROUND seems to be set for all XHR calls
		if( (request.getLoadFlags() & nsIChannel.LOAD_BACKGROUND) != 0 ){
			
			
//			System.out.println( "XHRMonitor:handleRequest() -- status flags<"+Long.toHexString(httpChannel.getStatus())+">." );
			
			try{
				
				nsIHttpChannel httpChannel = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );
				
				/*
				 * In nsIXMLHTTPRequest.cpp line 1593
				 * mChannel->SetNotificationCallbacks(this); //instance of XHR sets itself as notificationCallback
				 * 
				 * This means that I can QI to nsIXMLHTTPRequest
				 */
				nsIInterfaceRequestor intReq = httpChannel.getNotificationCallbacks();
				nsIXMLHttpRequest xhr = (nsIXMLHttpRequest)intReq.queryInterface( nsIXMLHttpRequest.NS_IXMLHTTPREQUEST_IID );
				
				handleXHRRequest( xhr );
				
//				System.out.println( "XHRMonitor:handleRequest() -- xhr" );
				return;
			} catch( XPCOMException e ){}
		}
		
		try{
			nsIHttpChannel httpRequest = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );

			MozHTTPCall call = new MozHTTPCall();
			
			call.captureRequest( httpRequest );
			
			callList.add( call, request );
			return;
		}
		catch( Exception e ){
			
		}
		
	}
	
	/*
	 * Handle the response side for the observer
	 */
	protected void observeResponse( nsIRequest request ){
		
		//System.err.println( "MozNetworkMonitorAdaptre:observeResponse() -- " + request.getName() );
		
		//ignore xhr calls because they are handled by the XHR Handler
		if( (request.getLoadFlags() & nsIChannel.LOAD_BACKGROUND) != 0 ){
			
			//System.out.println( "XHRMonitor:observe() -- POSSIBLE AJAX attaching webProgressListener" );
//			System.out.println( "XHRMonitor:handleRequest() -- status flags<"+Long.toHexString(httpChannel.getStatus())+">." );
			
			try{
				
				nsIHttpChannel httpChannel = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );
				
				/*
				 * In nsIXMLHTTPRequest.cpp line 1593
				 * mChannel->SetNotificationCallbacks(this); //instance of XHR sets itself as notificationCallback
				 * 
				 * This means that I can QI to nsIXMLHTTPRequest
				 */
				nsIInterfaceRequestor intReq = httpChannel.getNotificationCallbacks();
				
				//can we QI to XHR?
				intReq.queryInterface( nsIXMLHttpRequest.NS_IXMLHTTPREQUEST_IID );
				
				return;
			} catch( XPCOMException e ){}
		}

		MozHTTPCall call = (MozHTTPCall)callList.get( request );
		if( call != null ){
			nsIHttpChannel httpRequest = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );
			call.captureResponse( httpRequest );
			
			//this notifies the call list that it needs to notify its listeners that a call
			//object has changed
			callList.removeMap( request ); //no need to keep around the map
			callList.update( call );
		}
	}
	
	/**
	 * An XHR request has been identified, grab the information, set
	 * handlers to sniff response, and add to call list.
	 * 
	 * @param xhr
	 */
	protected void handleXHRRequest( nsIXMLHttpRequest xhr ){
		try{
			MozXHRCall call = new MozXHRCall();
			
			nsIHttpChannel httpChannel = (nsIHttpChannel) (xhr.getChannel()
					.queryInterface(nsIHttpChannel.NS_IHTTPCHANNEL_IID));
			call.captureRequest( httpChannel );
			
			//hook to listen to the xhr response
			new XHRHandler( xhr );
			
			//map the call in progress with the xhr object
			callList.add( call, xhr );
		}
		catch( Exception e ){
			System.err.println( "Failed handleXHRRequest: " + e.getMessage() );
		}
	}
	
	/**
	 * The response for an XHR call has been detected. Grap the call in progress
	 * from the call list, capture the response information, and send an update
	 * message.
	 * 
	 * @param xhr
	 * @param error true if the handler detected an error (onError handler)
	 */
	protected void handleXHRResponse( nsIXMLHttpRequest xhr, boolean error ){
		
		MozXHRCall call = (MozXHRCall)callList.get( xhr );
		call.captureResponse( xhr, error );
		
		//this notifies the call list that it needs to notify its listeners that a call
		//object has changed
		callList.removeMap( xhr ); //no need to keep the map
		callList.update( call );
	}
	
	/**
	 * nsIWebProgressListener seem to be consistent in that there is a START and
	 * a STOP for every request. 
	 * 
	 * There are several issues:
	 * - It does not fire for XHR request
	 * - Image request come in as imgIRequest instances which have no access to
	 *   request/response headers.
	 *   
	 * Note:
	 * 	The Start is ignored because we are using the observer to detect start of
	 * request and possible response. The Stop remains because there might be
	 * cased that the Response topic is not fired for the observer and the STOP here
	 * does, giving the opportunity of capturing the response information.
	 */
	public void onStateChange(nsIWebProgress webProgress, nsIRequest request,
			long stateFlags, long status) {
		//System.err.println( "onStateChange " + this + " " + request.getName() );
		try{			
			//only care about HTTP request (should care about other requests such as file://)
			//nsIHttpChannel httpChannel = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );
			//System.err.println( httpChannel.getURI().getSpec() );
			if( (stateFlags & STATE_START) != 0 ){
				//System.err.println( "START " + httpChannel.getURI().getSpec() );
				
				//handleStartRequest( webProgress, request, stateFlags, status );
			}
			else if( (stateFlags & STATE_STOP) != 0 ){
				//System.err.println( "STOP " + httpChannel.getURI().getSpec() );
				
				handleStopRequest( webProgress, request, stateFlags, status );
			}
			
			return;
		}
		catch( Exception e ){
			
		}
//		
//		try{
//			imgIRequest imgRequest = (imgIRequest)request.queryInterface( imgIRequest.IMGIREQUEST_IID );
//			
//			if( (stateFlags & STATE_START) != 0 ){
//				System.err.println( "START Image" + request.getName() );
//			}
//			else if( (stateFlags & STATE_STOP) != 0 ){
//				System.err.println( "STOP Image" + request.getName() );
//			}
//			
////			nsIChannel channel = (nsIChannel)imgRequest.queryInterface( nsIChannel.NS_ICHANNEL_IID );
////			System.err.println( "got channel" );
//		}
//		catch( Exception e ){
//			
//		}
	}
	
	/*
	 * handle the start for a request
	 * 
	 * capture the request information, add to list with a mapping to the nsIRequest instance so that it can
	 * be retrieved once the response is detected.
	 */
	protected void handleStartRequest(nsIWebProgress webProgress, nsIRequest request, long stateFlags, long status ){
		//System.err.println( "START: " + request.getName() );
		
		//handle HTTP requests
		try{
			nsIHttpChannel httpRequest = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );

			MozHTTPCall call = new MozHTTPCall();
			
			call.captureRequest( httpRequest );
			
			callList.add( call, request );
			return;
		}
		catch( Exception e ){
			
		}
		
//		try{
//			imgIRequest imgRequest = (imgIRequest)request.queryInterface( imgIRequest.IMGIREQUEST_IID );
//			
//			
//			System.err.println("image");
//			//need to find the matching request from the list of request in the load group
//			nsILoadGroup group = imgRequest.getLoadGroup();
//			
//			nsIRequest rootRequest = group.getDefaultLoadRequest();
//			int activeCount = (int)group.getActiveCount();
//			
//			nsIRequest reqObj = null;
//			
//			nsISimpleEnumerator reqEnum = group.getRequests();
//			
//			while( reqEnum.hasMoreElements() ){
//				reqObj = (nsIRequest)reqEnum.getNext().queryInterface(nsIRequest.NS_IREQUEST_IID);
//				
//				System.out.print('\0');
//				
//			}
			
//			nsIChannel channel = (nsIChannel)imgRequest.queryInterface( nsIChannel.NS_ICHANNEL_IID );
//			
//			nsIHttpChannel httpRequest = (nsIHttpChannel)imgRequest.getDecoderObserver().queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );
//			HTTPCall call = new HTTPCall( httpRequest );
//			callList.addCall(call);
//			ongoingHTTPCalls.put( request, call );
//			return;
//		}
//		catch( Exception e ){
//			
//		}
	}
	
	/*
	 * handle the stop for a request
	 */
	protected void handleStopRequest(nsIWebProgress webProgress, nsIRequest request, long stateFlags, long status ){
		//System.err.println( "STOP: " + request.getName() );
		
		MozHTTPCall call = (MozHTTPCall)callList.get( request );
		if( call != null ){
			nsIHttpChannel httpRequest = (nsIHttpChannel)request.queryInterface( nsIHttpChannel.NS_IHTTPCHANNEL_IID );
			call.captureResponse( httpRequest );
			
			//this notifies the call list that it needs to notify its listeners that a call
			//object has changed
			callList.removeMap( request ); //no need to keep around the map
			callList.update( call );
		}
	}
	
	public void onLocationChange(nsIWebProgress webProgress,
			nsIRequest request, nsIURI location) {}

	public void onProgressChange(nsIWebProgress webProgress,
			nsIRequest request, int curSelfProgress, int maxSelfProgress,
			int curTotalProgress, int maxTotalProgress) {}

	public void onSecurityChange(nsIWebProgress webProgress,
			nsIRequest request, long state) {}

	public void onStatusChange(nsIWebProgress webProgress, nsIRequest request,
			long status, String message) {}

	public nsISupports queryInterface(String uuid) {
		return Mozilla.queryInterface( this, uuid ); 
	}
}
