/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/
package org.eclipse.atf.mozilla.ide.ui.netmon;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.atf.mozilla.ide.ui.MozIDEUIPlugin;
import org.eclipse.atf.mozilla.ide.ui.browser.IWebBrowser;
import org.eclipse.atf.mozilla.ide.ui.browser.views.IBrowserView;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.IHTTPRequest;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.IHTTPResponse;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.INetworkCall;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.INetworkCallList;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.IRequest;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.IResponse;
import org.eclipse.atf.mozilla.ide.ui.netmon.payload.IPayloadRenderStrategy;
import org.eclipse.atf.mozilla.ide.ui.netmon.payload.ITextPayloadRenderStrategy;
import org.eclipse.atf.mozilla.ide.ui.netmon.payload.ImagePayloadRenderStrategy;
import org.eclipse.atf.mozilla.ide.ui.netmon.payload.PayloadRenderStrategyFactory;
import org.eclipse.atf.mozilla.ide.ui.netmon.payload.SimpleTextPayloadRenderStrategy;
import org.eclipse.atf.mozilla.ide.ui.netmon.payload.UnsupportedContentStrategy;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.part.Page;
import org.eclipse.ui.progress.UIJob;

/**
 * Implementation of a Page that visualizes all the Network calls. It is implemented
 * so that it is independent of the platform (i.e. Mozilla...).
 * 
 * It shows a table with all the network calls that have been logged. For each call, a
 * panel can be opened that shows details of the Request and Response side of the call.
 * 
 * For HTTP calls, it will show a list of headers and body for each side of the call.
 * 
 * @author Gino Bustelo, Kevin Sawicki
 *
 */
public class NetworkMonitorPage extends Page implements INetworkMonitor, IBrowserView{
	
	public static final String VIEW_POINT = "org.eclipse.atf.mozilla.ide.ui.filter";
	public static final String FILTER_NODE = "filter";
	public static final String CLASS_ATTR = "class";
	public static final String ICON_ATTR = "icon";
	public static final String TOOLTIP_ATTR = "tooltip";

	protected IWebBrowser webBrowser;
	protected INetworkMonitorAdapter netMonAdapter;
	protected INetworkCallList callList;
	
	protected SashForm displayArea = null;
	protected TableViewer callsViewer = null;
	protected Table table = null;
	protected Label requestLabel = null;
	protected Label responseLabel = null;
		
	//call details
	protected SourceViewer requestText = null;	//request body
	protected SourceViewer responseText = null;	//response body
	protected TableViewer requestHeadersViewer = null;	//request headers
	protected TableViewer responseHeadersViewer = null;	//response headers
	
	protected Composite responseUnsupported;
	protected Composite responseImage;
	protected Label imageLabel;
	protected CTabItem responseBodyItem;
	
	// this color will be allocated by the system, and must not be disposed
	protected Color white = null;
	
	protected Action clearAction = null;
	protected Action collapseAction = null;
	protected Action scrollLockAction = null;
	
	protected Action filterXHR = null;
	protected Action filterHTTP = null;
	
	//Formatting details
	protected Button formatButton;

	
	//Copy Menu
	protected Menu menu;
	protected MenuItem copy;
	protected Clipboard clipboard;
		
	protected static final String STATE_COL = "state";
	protected static final String URL_COL = "url";
	protected static final String METHOD_COL = "method";
	protected static final String START_COL = "start";
	protected static final String STOP_COL = "stop";
	protected static final String ELAPSED_COL = "elapsed";
	protected String[] columnNames = {STATE_COL, URL_COL, METHOD_COL, START_COL, STOP_COL, ELAPSED_COL};
	protected List filters = new ArrayList();
	protected List actions = new ArrayList();
	
	//boolean to determine if new XHR calls are revealed by the viewer
	protected boolean scrollLock = false;
		
	protected ViewerFilter xhrFilter = new ViewerFilter() {
	
		public boolean select(Viewer viewer, Object parentElement, Object element) {
			//need to check the request side of the call to see if it is XHR
			INetworkCall call = (INetworkCall)element;
			IRequest req = call.getRequest();
			
			if( req instanceof IHTTPRequest )
				return ((IHTTPRequest)req).isXHR();
			else
				return false;
		}
	
	};
	
	protected ViewerFilter httpFilter = new ViewerFilter() {
	
		public boolean select(Viewer viewer, Object parentElement, Object element) {

			//need to check the request side is http but not XHR
			INetworkCall call = (INetworkCall)element;
			IRequest req = call.getRequest();
			
			if( req instanceof IHTTPRequest )
				return !((IHTTPRequest)req).isXHR();
			else
				return false;
		
		}
	
	};
			
	public void createControl(Composite parent) {
		
		//initialize the reference to the monitor adapter
		netMonAdapter = (INetworkMonitorAdapter)webBrowser.getAdapter( INetworkMonitorAdapter.class );
		
		//connect to the current call list (there might be calls already logged)
		callList = netMonAdapter.getCallList();
		callList.addListener( this );
		
		displayArea = new SashForm( parent, SWT.VERTICAL );
		createTable();
		createTableViewer();
		createRequestResponsePanel();
		
		//Set up copy menu for response text
		clipboard = new Clipboard(getSite().getShell().getDisplay());
		menu = new Menu(responseText.getControl());
		responseText.getControl().setMenu(menu);
		menu.setEnabled(true);
		copy = new MenuItem(menu,SWT.PUSH);
		copy.setText("Copy");
		copy.setImage(MozIDEUIPlugin.getDefault().getImage(MozIDEUIPlugin.COPYRESPONSE_ID));
		copy.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				try {
					Point p = responseText.getSelectedRange();
					int offset = p.x;
					int length = p.y;
					String content = responseText.getDocument().get(offset, length);
					if( content != null && content.length() > 0 ) {
						clipboard.setContents(new String[]{content}, new Transfer[]{TextTransfer.getInstance()});
					}
				} catch (BadLocationException e1) {}
			}
		
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		//set up copy menu for request text
		menu = new Menu(requestText.getControl());
		requestText.getControl().setMenu(menu);
		menu.setEnabled(true);
		copy = new MenuItem(menu,SWT.PUSH);
		copy.setText("Copy");
		copy.setImage(MozIDEUIPlugin.getDefault().getImage(MozIDEUIPlugin.COPYRESPONSE_ID));
		copy.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				try {
					Point p = requestText.getSelectedRange();
					int offset = p.x;
					int length = p.y;
					String content = requestText.getDocument().get(offset, length);
					if( content != null && content.length() > 0 ) {
						clipboard.setContents(new String[]{content}, new Transfer[]{TextTransfer.getInstance()});
					}
				} catch (BadLocationException e1) {}
			}
		
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		callsViewer.setContentProvider(new NetworkCallListContentProvider());
		callsViewer.setLabelProvider(new NetworkCallLabelProvider());
		callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.START_TIME));
		
		//Initially hide the request/response panel
		displayArea.setMaximizedControl(table);
		
		createActions();
		IToolBarManager toolBarManager = getSite().getActionBars().getToolBarManager();
		toolBarManager.add( new Separator() );
		

		toolBarManager.add( filterXHR );
		toolBarManager.add( filterHTTP );
		toolBarManager.add( new Separator() );
		
		
		toolBarManager.add( collapseAction );
		toolBarManager.add( scrollLockAction );
		toolBarManager.add( clearAction );
		
		callsViewer.setInput( callList );
	}
	
	protected void createTable() {
		
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION;

		table = new Table(displayArea, style);
						
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);		
				
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// 1st column 
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);		
		column.setText("");
		column.setWidth(25);
		column.addSelectionListener(new SelectionAdapter() {
	       	
			public void widgetSelected(SelectionEvent e) {
				NetworkCallSorter sorter = (NetworkCallSorter)callsViewer.getSorter();
				if(sorter != null && sorter.getColumnCriteria() == NetworkCallSorter.STATE) {
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.STATE, !sorter.isAscending()));
				}
				else {	
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.STATE));
				}
			}
		});
		
		// 2nd column 
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("URL");
		column.setWidth(300);
		column.addSelectionListener(new SelectionAdapter() {
	       	
			public void widgetSelected(SelectionEvent e) {
				NetworkCallSorter sorter = (NetworkCallSorter)callsViewer.getSorter();
				if(sorter != null && sorter.getColumnCriteria() == NetworkCallSorter.URL) {
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.URL, !sorter.isAscending()));
				}
				else {	
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.URL));
				}
			}
		});
						
		// 3rd column 
		column = new TableColumn(table, SWT.LEFT, 2);
		column.setText("Method");
		column.setWidth(75);
		column.addSelectionListener(new SelectionAdapter() {
	       	
			public void widgetSelected(SelectionEvent e) {
				NetworkCallSorter sorter = (NetworkCallSorter)callsViewer.getSorter();
				if(sorter != null && sorter.getColumnCriteria() == NetworkCallSorter.METHOD) {
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.METHOD, !sorter.isAscending()));
				}
				else {	
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.METHOD));
				}
			}
		});		
		
		// 4th column
		column = new TableColumn(table, SWT.LEFT, 3);
		column.setText("Start Time");
		column.setWidth(175);
		column.addSelectionListener(new SelectionAdapter() {
	       	
			public void widgetSelected(SelectionEvent e) {
				NetworkCallSorter sorter = (NetworkCallSorter)callsViewer.getSorter();
				if(sorter != null && sorter.getColumnCriteria() == NetworkCallSorter.START_TIME) {
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.START_TIME, !sorter.isAscending()));
				}
				else {	
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.START_TIME));
				}
			}
		});		
		
		// 5th column
		column = new TableColumn(table, SWT.LEFT, 4);
		column.setText("Stop Time");
		column.setWidth(175);
		column.addSelectionListener(new SelectionAdapter() {
	       	
			public void widgetSelected(SelectionEvent e) {
				NetworkCallSorter sorter = (NetworkCallSorter)callsViewer.getSorter();
				if(sorter != null && sorter.getColumnCriteria() == NetworkCallSorter.STOP_TIME) {
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.STOP_TIME, !sorter.isAscending()));
				}
				else {	
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.STOP_TIME));
				}
			}
		});		
		
		// 6th column
		column = new TableColumn(table, SWT.LEFT, 5);
		column.setText("Elapsed Time");
		column.setWidth(100);
		column.addSelectionListener(new SelectionAdapter() {
	       	
			public void widgetSelected(SelectionEvent e) {
				NetworkCallSorter sorter = (NetworkCallSorter)callsViewer.getSorter();
				if(sorter != null && sorter.getColumnCriteria() == NetworkCallSorter.ELAPSED_TIME) {
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.ELAPSED_TIME, !sorter.isAscending()));
				}
				else {	
					callsViewer.setSorter(new NetworkCallSorter(NetworkCallSorter.ELAPSED_TIME));
				}
			}
		});
		
		//7th column STATUS CODE
		column = new TableColumn(table, SWT.LEFT, 6);
		column.setText("Status Code");
		column.setWidth(100);
		
		table.addSelectionListener(new SelectionListener() {

			public void widgetSelected(SelectionEvent e) {
				callSelectionChanged();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
			
		});
		
		table.addMouseListener(new MouseListener() {

			public void mouseDoubleClick(MouseEvent e) {
				showCallInfo();
			}

			public void mouseDown(MouseEvent e) {}

			public void mouseUp(MouseEvent e) {}
			
		});
	}
	
	protected void createTableViewer() {
		
		callsViewer = new TableViewer(table); 
		callsViewer.setUseHashlookup(true);
		
		callsViewer.setColumnProperties(columnNames);

/*		// Create the cell editors
		CellEditor[] editors = new CellEditor[columnNames.length];

		// Column 1 : State (text)
		TextCellEditor textEditor = new TextCellEditor(table, SWT.READ_ONLY);
		((Text) textEditor.getControl()).setTextLimit(50);
		editors[0] = textEditor;

		// Column 2 : URL (text)
		textEditor = new TextCellEditor(table, SWT.READ_ONLY);
		((Text) textEditor.getControl()).setTextLimit(150);
		editors[1] = textEditor;

		// Column 3 : Method (text) 
		textEditor = new TextCellEditor(table, SWT.READ_ONLY);
		((Text) textEditor.getControl()).setTextLimit(75);
		editors[2] = textEditor;
		
		// Column 4 : Start Time (Text)
		textEditor = new TextCellEditor(table, SWT.READ_ONLY);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[3] = textEditor;
		
		// Column 5 : Stop Time (Text)
		textEditor = new TextCellEditor(table, SWT.READ_ONLY);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[4] = textEditor;
		
		// Column 6 : Elapsed Time (Checkbox)
		textEditor = new TextCellEditor(table, SWT.READ_ONLY);
		((Text) textEditor.getControl()).setTextLimit(100);
		editors[5] = textEditor;
				
		// Assign the cell editors to the viewer 
		callsViewer.setCellEditors(editors);
		
		// Set the cell modifier for the viewer
		callsViewer.setCellModifier(new XHRCellModifier());*/
	}
	
	protected void createRequestResponsePanel() {
		
		Composite composite = new Composite(displayArea, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.makeColumnsEqualWidth = true;
		layout.marginHeight = 1;
		layout.marginWidth = 1;
		layout.verticalSpacing = 1;
		
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(SWT.FILL,SWT.FILL, true, true));
		
		requestLabel = new Label(composite, SWT.NONE);
		requestLabel.setText("Request:");
		requestLabel.setLayoutData(new GridData(GridData.BEGINNING, GridData.BEGINNING, true, false));
		
		Composite comp2 = new Composite(composite, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginHeight = 1;
		layout.marginWidth = 1;
		layout.verticalSpacing = 1;
		layout.makeColumnsEqualWidth = false;
		comp2.setLayout(layout);
		comp2.setLayoutData(new GridData(SWT.FILL,SWT.FILL, false, false));
		responseLabel = new Label(comp2, SWT.NONE);
		responseLabel.setText("Response:");
		responseLabel.setLayoutData(new GridData(GridData.BEGINNING, GridData.BEGINNING, true, false));
		
		formatButton = new Button(comp2, SWT.CHECK );
		formatButton.setText("Format");
		formatButton.setVisible( false );
		formatButton.setSelection( false );

		formatButton.addSelectionListener( new SelectionListener() {
		
			/**
			 * If this button is visible, we can assume that the content is text.
			 */
			public void widgetSelected(SelectionEvent e) {
				IStructuredSelection selection = (IStructuredSelection) callsViewer.getSelection();
				INetworkCall call = (INetworkCall) selection.getFirstElement();
				
				IResponse res = call.getResponse();
				
				if( res instanceof IHTTPResponse ){
					PayloadRenderStrategyFactory strategyFactory = PayloadRenderStrategyFactory.getInstance();	
					
					//we can assume that if we are here, the content is text
					ITextPayloadRenderStrategy strategy = (ITextPayloadRenderStrategy)strategyFactory.getStrategy( (IHTTPResponse)res );
				
					SourceViewerConfiguration viewerConf = strategy.getConfiguration();
					
					responseText.unconfigure();
					
					if( viewerConf != null && formatButton.getSelection() ){
						responseText.configure( viewerConf );
					}
					
					responseText.refresh();
				}
			}
		
			public void widgetDefaultSelected(SelectionEvent e) {}
		
		});
		
		
		//insert a tab panel with Headers and Body tabs
		
		//REQUEST TAB FOLDER
		CTabFolder requestTabFolder = new CTabFolder(composite, SWT.BOTTOM | SWT.BORDER | SWT.FLAT );
		requestTabFolder.setLayoutData( new GridData(GridData.FILL, GridData.FILL, true, true) );
		
		CTabItem requestHeaderItem = new CTabItem( requestTabFolder, SWT.NONE );
		requestHeaderItem.setText( "Headers" );
		CTabItem requestBodyItem = new CTabItem( requestTabFolder, SWT.NONE );
		requestBodyItem.setText( "Body" );
		
		//request headers table
		requestHeadersViewer = createHeadersViewer( requestTabFolder, new HeaderContentProvider() );
		
		//request body text
		requestText = createBodyText( requestTabFolder );
		
		
		requestHeaderItem.setControl( requestHeadersViewer.getControl() );
		requestBodyItem.setControl( requestText.getControl() );
		
		requestTabFolder.setSelection( requestHeaderItem );
		
		//RESPONSE TAB FOLDER
		CTabFolder responseTabFolder = new CTabFolder(composite, SWT.BOTTOM | SWT.BORDER | SWT.FLAT);
		responseTabFolder.setLayoutData( new GridData(GridData.FILL, GridData.FILL, true, true) );
		
		CTabItem responseHeaderItem = new CTabItem( responseTabFolder, SWT.NONE );
		responseHeaderItem.setText( "Headers" );
		responseBodyItem = new CTabItem( responseTabFolder, SWT.NONE );
		responseBodyItem.setText( "Body" );
		
		//response headers table
		responseHeadersViewer = createHeadersViewer( responseTabFolder, new HeaderContentProvider() );
		
		//response body text
		responseText = createBodyText( responseTabFolder );
		
		responseImage = new ScrolledComposite( responseTabFolder, SWT.H_SCROLL | SWT.V_SCROLL );
		// this is a system color, and must not be disposed
		white = getSite().getShell().getDisplay().getSystemColor( SWT.COLOR_WHITE );
		responseImage.setBackground( white );
		
		//responseImage.setLayout( new FillLayout() );
		imageLabel = new Label( responseImage, SWT.NONE );
		((ScrolledComposite)responseImage).setContent( imageLabel );
		imageLabel.setText( "Image Here" );
		imageLabel.setSize( 100, 50 );
		
		responseUnsupported = new Composite( responseTabFolder, SWT.NONE );
		responseUnsupported.setLayout( new FillLayout() );
		
		responseImage.setBackground( white );
		
		Label unsupportedLabel = new Label( responseUnsupported, SWT.NONE );
		unsupportedLabel.setText( "Cannot display... Unsupported MIME type!" );
		
		//body = new Composite(responseTabFolder, SWT.NONE );
		
		responseHeaderItem.setControl( responseHeadersViewer.getControl() );
		//GINO: REDO FORMATING: responseBodyItem.setControl( body );
		responseBodyItem.setControl( responseText.getControl() );
		
		responseTabFolder.setSelection( responseHeaderItem );
		
	}
	
	/*
	 * Creates the Text widget used to show the body content of a Request
	 */
	protected SourceViewer createBodyText( Composite parent ){
		
		SourceViewer bodyText = new SourceViewer(parent,null,SWT.V_SCROLL | SWT.H_SCROLL);
		bodyText.setEditable(false);
		bodyText.setDocument(new Document(""));
		
		return bodyText;
	}
	
	/*
	 * Creates the table that is used to show the headers for either the request
	 * or response.
	 */
	protected TableViewer createHeadersViewer( Composite parent, HeaderContentProvider provider ){
		TableViewer headerViewer = new TableViewer( parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.HIDE_SELECTION );
		
		Table t = headerViewer.getTable();
		
		t.setLinesVisible(true);
		t.setHeaderVisible(false);
		
		
		// 1st column 
		TableColumn column = new TableColumn(t, SWT.LEFT | SWT.FILL, 0);		
		column.setText("Name");
		
		// 2nd column 
		column = new TableColumn(t, SWT.LEFT | SWT.FILL, 1);
		column.setText("Value");
		
		//Listener to set up the initial Widths of the table's columns
		t.addControlListener( new ControlAdapter(){

			public void controlResized(ControlEvent e) {
				try{
					Table resizedTable = (Table)e.getSource();
					
					int w = resizedTable.getClientArea().width;
					
					resizedTable.getColumn(0).setWidth( w/2 );
					resizedTable.getColumn(1).setWidth( w/2 );
					
					//remove the listener so that it is only done once
					resizedTable.removeControlListener( this );
				}
				catch( Exception exception ){
					//if anything 
				}
			}
			
			
		});
		
		
		
		headerViewer.setLabelProvider( new HeaderLabelProvider() );
		headerViewer.setContentProvider( provider );
		
		return headerViewer;
	}
	
	private void useContentFilter(ViewerFilter filter) {
		for( int i = 0; i < filters.size(); i++ ) {
			callsViewer.removeFilter((ViewerFilter)filters.get(i));
		}
		callsViewer.addFilter(filter);
	}
	
	private void useRequestFilter(ViewerFilter filter) {
		callsViewer.removeFilter(xhrFilter);
		callsViewer.removeFilter(httpFilter);
		callsViewer.addFilter(filter);
	}
	
	private void disableOtherActions(IAction action) {
		for( int i = 0; i < actions.size(); i++ ) {
			if( !actions.get(i).equals(action) ) {
				((IAction)actions.get(i)).setChecked(false);
			}
		}
	}
	
	protected void createActions() {
		IToolBarManager toolBarManager = getSite().getActionBars().getToolBarManager();
		
		IExtensionRegistry reg = Platform.getExtensionRegistry();
		IExtensionPoint ep = reg.getExtensionPoint(VIEW_POINT);
		IExtension[] extensions = ep.getExtensions();
		for(int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] ce = extensions[i].getConfigurationElements();
			for(int j = 0; j < ce.length; j++ ) {				
				try {
					if( ce[j].getName().equals(FILTER_NODE) ) {
						String filter = ce[j].getAttribute(CLASS_ATTR);
						String icon = ce[j].getAttribute(ICON_ATTR);
						String tooltip = ce[j].getAttribute(TOOLTIP_ATTR);
						if( filter != null && icon != null && tooltip != null ) {
							final ViewerFilter filterObj = (ViewerFilter)ce[j].createExecutableExtension(CLASS_ATTR);
							IAction action = new Action(null, Action.AS_CHECK_BOX) {
								
								private ViewerFilter filter = filterObj;
								
								public void run() {
									if( isChecked() ) {
										disableOtherActions(this);
										useContentFilter(filter);
									} else {
										callsViewer.removeFilter(filter);
									}
								}
							
							};
							action.setImageDescriptor(MozIDEUIPlugin.getImageDescriptor(icon));
							action.setToolTipText(tooltip);
							filters.add(filterObj);
							actions.add(action);
							toolBarManager.add(action);
						}
					}
				} catch (Exception e) {}
			}
		}
		
		filterXHR = new Action(null, Action.AS_CHECK_BOX){
		
			public void run() {
				if(isChecked()) {
					filterHTTP.setChecked(false);
					useRequestFilter(xhrFilter);
				} else {
					callsViewer.removeFilter(xhrFilter);
				}
			}
		
		};
		filterXHR.setImageDescriptor(MozIDEUIPlugin.getDefault().getImageDescriptorFromRegistry(MozIDEUIPlugin.FILTERXHR_ID));
		filterXHR.setToolTipText("Show XHR only");
		
		filterHTTP = new Action(null, Action.AS_CHECK_BOX){
			
			public void run() {
				if(isChecked()) {
					filterXHR.setChecked(false);
					useRequestFilter(httpFilter);
				} else {
					callsViewer.removeFilter(httpFilter);
				}
			}
		
		};
		filterHTTP.setToolTipText("Show HTTP only");
		filterHTTP.setImageDescriptor(MozIDEUIPlugin.getDefault().getImageDescriptorFromRegistry(MozIDEUIPlugin.FILTERHTTP_ID));
		
		collapseAction = new Action(null, Action.AS_CHECK_BOX){
			public void run(){
				if(isChecked()) {
					displayArea.setMaximizedControl(table);
				} else {
					displayArea.setMaximizedControl(null);
					
					//check the selection and maintain it visible
					revealSelection();
				}
			}
		};
			
		collapseAction.setImageDescriptor(MozIDEUIPlugin.getImageDescriptor("icons/xhrmon/collapse.gif"));
		collapseAction.setToolTipText("Hide the request/response content panel");
		collapseAction.setChecked(true);
		
		//scroll lock action
		scrollLockAction = new Action(null, Action.AS_CHECK_BOX){
			public void run(){
				scrollLock = isChecked();
			}
		};
			
		scrollLockAction.setImageDescriptor(MozIDEUIPlugin.getImageDescriptor("icons/xhrmon/scrolllock.gif"));
		scrollLockAction.setToolTipText("Lock/Unlock the scroll of the content pane");
		scrollLockAction.setChecked(false);
		
		
		clearAction = new Action(null, Action.AS_PUSH_BUTTON) {
			public void run() {
				callList.clear();
				callsViewer.refresh();
				clearRequestResponseArea();
			}
		};
		clearAction.setImageDescriptor(MozIDEUIPlugin.getDefault().getImageDescriptorFromRegistry(MozIDEUIPlugin.CLEAR_IMG_ID) );
		clearAction.setToolTipText("Clear the call list");
		
	}

	public Control getControl() {
		return displayArea;
	}

	public void setFocus() {
		callsViewer.getControl().setFocus();

	}
	
	public void dispose() {
		callList.removeListener( this ); 
		clipboard.dispose();
		
		super.dispose();
	}
	
	protected void clearRequestResponseArea() {
		clearRequestArea();
		clearResponseArea();
	}

	protected void clearResponseArea() {
		//set responseText as the visible control and set to ""
		responseText.getDocument().set("");
		responseBodyItem.setControl( responseText.getControl() );
		
		formatButton.setVisible( false );
		
		responseHeadersViewer.setInput( null );
		
		//no need to have this job running because the call list is now clear
		if( responseBodyGetterJob != null )
			responseBodyGetterJob.cancel();
		
	}

	protected void clearRequestArea() {
		//clear the body text for the request
		requestText.getDocument().set("");
		
		//clear the headers
		requestHeadersViewer.setInput( null );
	}

	/**
	 * This is a self scheduling Job that runs in the UI thread (because there is acess to UI
	 * components) and tries to get the body of the response. The Response object could
	 * be waiting for access to cache so the response is not guaranteed to be ready.
	 * 
	 * @author Gino Bustelo
	 *
	 */
	protected class ResponseBodyGetterJob extends UIJob{
		
		protected int triesLeft = 5;
		
		protected final static int DELAY = 250;
		
		protected IHTTPResponse response = null;
		
		public ResponseBodyGetterJob( IHTTPResponse response ) {
			super( "Response Body Getter" );
			this.response = response;
		}
		
		public IStatus runInUIThread(IProgressMonitor monitor) {
			
//			System.err.println( "ResponseBodyGetterJob: tries left <"+triesLeft+">..." );
			
			triesLeft--;
			
			Object body = response.getBody();
			
			if( !monitor.isCanceled() ){
				if( body != null ){
					
//					System.err.println( "ResponseBodyGetterJob: got body, setting..." );
					
					//set the text in the UI and that's it
					responseText.getDocument().set( body.toString() );
					
				}
				else{
					//reschedule if not reach max
					if( triesLeft > 0 ){
//						System.err.println( "ResponseBodyGetterJob: try again..." );
						schedule( DELAY );
					}
					else{
						//done trying
//						System.err.println( "ResponseBodyGetterJob: done trying." );
						responseText.getDocument().set( "Cache not responding!" );
					}
					
				}
			}
			
			return Status.OK_STATUS;
		}
		
	};
	
	//cached instance if need to cancel
	protected ResponseBodyGetterJob responseBodyGetterJob = null;
	
	protected void callSelectionChanged() {
		IStructuredSelection selection = (IStructuredSelection) callsViewer.getSelection();
		
		if( selection.isEmpty() || selection.size() > 1) {
			//clear the details area if there is nothing selected or more than one call selected
			clearRequestResponseArea();
			
		} else {
			
			//cancel the job for previous selection because there is no need to wait.
			if( responseBodyGetterJob != null )
				responseBodyGetterJob.cancel();
			
			
			INetworkCall call = (INetworkCall) selection.getFirstElement();
						
			//set information for Request side of call
			IRequest req = call.getRequest();
			requestHeadersViewer.setInput( req );
						
			if( req instanceof IHTTPRequest ){
				requestText.getDocument().set( ((IHTTPRequest)req).getBody() );
			}
			else{
				requestText.getDocument().set( "" );
			}
			
			//set response side of the call (Note that the call might be ongoing and response not available
			IResponse res = call.getResponse();
			
			if( res != null ){
				responseHeadersViewer.setInput( res );
			
				//render the body
				if( res instanceof IHTTPResponse ){
					//show the correct control based on the MIME
					
					PayloadRenderStrategyFactory strategyFactory = PayloadRenderStrategyFactory.getInstance();
					
					IPayloadRenderStrategy strategy = strategyFactory.getStrategy( (IHTTPResponse)res );
					
					//Handle Text Payload (html, css, js, xml...)
					if( strategy.getRenderType() == SimpleTextPayloadRenderStrategy.RENDER_TYPE ){
						
						
						//perform this in a job
						Object responseBody = ((IHTTPResponse)res).getBody();
						
						if( responseBody != null ){
							responseText.getDocument().set( responseBody.toString() );
						}
						/*
						 * Assume that there is cache access going on so launch a job to poll for the body
						 */
						else{
							responseText.getDocument().set( "Loading..." );
							
							responseBodyGetterJob = new ResponseBodyGetterJob( ((IHTTPResponse)res) );
							
							responseBodyGetterJob.setUser( false );							
							responseBodyGetterJob.setPriority( Job.INTERACTIVE );
							
							responseBodyGetterJob.schedule();
						}
						
						responseBodyItem.setControl( responseText.getControl() );
						
						//set the formatting settings
						ITextPayloadRenderStrategy textStrategy = (ITextPayloadRenderStrategy)strategy;
						SourceViewerConfiguration sourceConfig = textStrategy.getConfiguration();
						
						//if sourceConfig is null, then assume that this text can't be formatted
						if( sourceConfig == null ){
							formatButton.setVisible( false );
							responseText.unconfigure();
						}
						else{
							formatButton.setVisible( true );
							
							//need to check the selection status of the button
							if( formatButton.getSelection() ){
								responseText.unconfigure();
								responseText.configure( textStrategy.getConfiguration() );
							}
							else{
								responseText.unconfigure();
							}
						}
						
						responseText.refresh();
						
					}
					//Handle Image Payload (gif, png, jpg...)
					else if( strategy.getRenderType() == ImagePayloadRenderStrategy.RENDER_TYPE ){
						//read the image
						responseBodyItem.setControl( responseImage );
						
						//dispose the previous image
						if( imageLabel.getImage() != null ){
							imageLabel.getImage().dispose();
						}
						
						try {
							
							//@GINO: Need to cache the image data and not access the net every time
							//Should try to use the Browser's cache
							Image image = ImageDescriptor.createFromURL(new URL(call.getRequest().getURL())).createImage();
							imageLabel.setImage( image );
							imageLabel.setSize( image.getBounds().width, image.getBounds().height );
							
						} catch (MalformedURLException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
						
						formatButton.setVisible( false );

					}
					else if( strategy.getRenderType() == UnsupportedContentStrategy.RENDER_TYPE ){
						responseBodyItem.setControl( responseUnsupported );
						
						formatButton.setVisible( false );
					}
					else{
						clearResponseArea(); //there might not be a need for else
					}
					
					
				}
				else{
					//currently only supporting HTTP so this is here of future support
					responseText.getDocument().set("");
					responseBodyItem.setControl( responseText.getControl() );
					responseText.refresh();
				}
			}
			else{
				clearResponseArea();
			}			
		}
		
	}
	
	protected void showCallInfo() {
		IStructuredSelection selection = (IStructuredSelection) callsViewer.getSelection();
		if(selection.size() == 1 && displayArea.getMaximizedControl() != null) {
			displayArea.setMaximizedControl(null);
			collapseAction.setChecked(false);
			
			revealSelection();
		} else {
			//displayArea.setMaximizedControl(table);
			//collapseAction.setChecked(true);
		}
	}
	
	protected void revealSelection(){
		
		//if there is a selection, leave the selection visible
		IStructuredSelection selection = (IStructuredSelection) callsViewer.getSelection();
		if(selection != null && selection.size() == 1) {
			callsViewer.reveal( selection.getFirstElement() );
		}
	}
	
	private class NetworkCallListContentProvider implements IStructuredContentProvider {

		// Return the variables as an array of Objects
		public Object[] getElements(Object parent) {
			return callList.getAll();
		}

		public void dispose() {}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {}		
	}

	public void setWebBrowser(IWebBrowser documentContainer) {
		this.webBrowser = documentContainer;
	}

	public void notifyCallAdded(INetworkCall call) {
		callsViewer.add(call);
		
		if( !scrollLock )
			callsViewer.reveal( call );
		else{
			revealSelection();
		}		
	}

	public void notifyCallUpdated(INetworkCall call) {
		callsViewer.update(call, null);
		
		//@GINO: TODO: Probably need to check if it is the selection to update the details area
	}
}
