package org.eclipse.atf.mozilla.ide.ui.netmon.model.impl;

import java.io.ByteArrayInputStream;
import java.util.zip.GZIPInputStream;

import org.eclipse.atf.mozilla.ide.ui.netmon.model.IHTTPResponse;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.IHeader;
import org.mozilla.interfaces.nsIBinaryInputStream;
import org.mozilla.interfaces.nsICache;
import org.mozilla.interfaces.nsICacheEntryDescriptor;
import org.mozilla.interfaces.nsICacheListener;
import org.mozilla.interfaces.nsICacheService;
import org.mozilla.interfaces.nsICacheSession;
import org.mozilla.interfaces.nsIHttpChannel;
import org.mozilla.interfaces.nsIHttpHeaderVisitor;
import org.mozilla.interfaces.nsIInputStream;
import org.mozilla.interfaces.nsIScriptableInputStream;
import org.mozilla.interfaces.nsISupports;
import org.mozilla.xpcom.Mozilla;

/**
 * This class is used to extract the HTTPCall response side information from the
 * XPCOM couterpart.
 * 
 * @author Gino Bustelo
 *
 */
public class MozHTTPResponse extends MozHeaderContainer implements IHTTPResponse {

	protected String status;
	
	protected Object body = null;
	
	protected boolean awaitingCache = false;
	
	//reference to the call instance that this response if part of
	protected MozHTTPCall call = null;
	
	public MozHTTPResponse( nsIHttpChannel httpChannel, MozHTTPCall call ){
		this.call = call;

		status = httpChannel.getResponseStatus() + " "
					+ httpChannel.getResponseStatusText();

		extractHeaders(httpChannel);
	}
	
	public String getStatus() {
		return status;
	}

	public Object getBody() {

		if( body == null && !awaitingCache ){
			
			//try to load from cache
			nsICacheService cacheService = (nsICacheService)Mozilla.getInstance().getServiceManager().getServiceByContractID( "@mozilla.org/network/cache-service;1", nsICacheService.NS_ICACHESERVICE_IID );
				
			//Cache session id MUST be "HTTP" so that the entries saved by the HTTP channel are found
			nsICacheSession session = cacheService.createSession( "HTTP" , nsICache.STORE_ANYWHERE, true );
			session.setDoomEntriesIfExpired(false);
			
//			System.err.println( "Storage Enabled:" + session.isStorageEnabled() );
			
			nsICacheEntryDescriptor cacheEntry = null;
			
			try {
									
//				cacheEntry = session.openCacheEntry( call.getRequest().getURL(), nsICache.ACCESS_READ, true);
//				readCacheEntry( cacheEntry );
				
				//to prevent blocking, access to cache is done async
				session.asyncOpenCacheEntry( call.getRequest().getURL(), nsICache.ACCESS_READ, new nsICacheListener(){

					public void onCacheEntryAvailable(
							nsICacheEntryDescriptor descriptor,
							int accessGranted, long status) {

						readCacheEntry( descriptor );
						awaitingCache = false;
						
					}

					public nsISupports queryInterface(String uuid) {
						return Mozilla.queryInterface( this, uuid );
					}
					
				});
				
				awaitingCache = true;

			} catch( Exception e ) {
				e.printStackTrace();
			}
			finally{
				
				if( cacheEntry != null ){
					try{
						cacheEntry.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
			}
		}
		
		return body;
	}
	
	public void setBody( Object body ){
		this.body = body;
	}

	protected void visitHeaders(nsIHttpChannel httpChannel,
			nsIHttpHeaderVisitor visitor) {
		httpChannel.visitResponseHeaders( visitor );		
	}
	
	protected void readCacheEntry( nsICacheEntryDescriptor cacheEntry ) {

		IHeader contentEncHeader = getHeader("Content-Encoding");
		
		//GZIPed data needs to be read with a BinaryIS because the Scriptable one
		//returns String on the read and truncates the content.
		if( contentEncHeader != null && "gzip".equals(contentEncHeader.getValue()) ){
			nsIInputStream stream = null;
			nsIBinaryInputStream bIS = null;
			
			GZIPInputStream gin = null;
			try {
				
				if( cacheEntry.isStreamBased() ) {
					
					stream = cacheEntry.openInputStream(0);
					
					bIS = (nsIBinaryInputStream)Mozilla.getInstance().getComponentManager().createInstanceByContractID( "@mozilla.org/binaryinputstream;1", null, nsIBinaryInputStream.NS_IBINARYINPUTSTREAM_IID );
					bIS.setInputStream( stream );
					
					//parse the header for content length
					IHeader contentLengthHeader = getHeader("Content-Length");
					if( contentLengthHeader == null ){
						//if can't determine the content length, then ignore getting the body content.
						body = "";
						return;
					}
					
					int contentLength = Integer.parseInt( contentLengthHeader.getValue() );	

					byte [] bytes = new byte[contentLength];
					
					for (int i = 0; i < bytes.length; i++) {
						bytes[i] = (byte)bIS.read8();
					}
										
					gin = new GZIPInputStream( new ByteArrayInputStream(bytes) );
					
					byte[] t = new byte[0];
					byte[] BUFFER = new byte[4096];
					while ( gin.available() > 0 ) {
						int n = gin.read(BUFFER);
						if (n > 0) {
							byte[] temp = new byte[t.length + n];
							System.arraycopy(t, 0, temp, 0, t.length);
							System.arraycopy(BUFFER, 0, temp, t.length, n);
							t = temp;
						}
					}
					
					body = new String( t );
						
									
				}
			}
			catch( Exception e ) {
				body = "";
				e.printStackTrace();
			}
			finally{
				
				if( cacheEntry != null ){
					try{
						cacheEntry.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
				
				if( gin != null ){
					
					try {
						gin.close();
					} catch (Exception e) {
						//e.printStackTrace();
					}
					
				}
				
				if( stream != null ){
					try{
						stream.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
				
				if( bIS != null ){
					try{
						bIS.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
			}
		}
		
		//read text
		else{
			
			nsIInputStream stream = null;
			nsIScriptableInputStream scriptStream = null;
			try {
				
				if( cacheEntry.isStreamBased() ) {
					stream = cacheEntry.openInputStream(0);
					
					scriptStream = (nsIScriptableInputStream)Mozilla.getInstance().getComponentManager().createInstanceByContractID("@mozilla.org/scriptableinputstream;1", null, nsIScriptableInputStream.NS_ISCRIPTABLEINPUTSTREAM_IID );
					scriptStream.init(stream);
	
					StringBuffer contentBuf = new StringBuffer();
	
					long count;
					while(  (count = scriptStream.available()) > 0 ){//some left to read{
						contentBuf.append( scriptStream.read(count) );
					
					}
					body = contentBuf.toString();
				}
				
			} catch( Exception e ) {
				e.printStackTrace();
			}
			finally{
				
				if( cacheEntry != null ){
					try{
						cacheEntry.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
				
				if( stream != null ){
					try{
						stream.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
				
				if( scriptStream != null ){
					try{
						scriptStream.close();
					}
					catch( Exception e ){
						//e.printStackTrace();
					}
				}
			}
		}
	}

}
