package org.eclipse.atf.mozilla.ide.ui.netmon.model.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.atf.mozilla.ide.ui.netmon.INetworkMonitor;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.INetworkCall;
import org.eclipse.atf.mozilla.ide.ui.netmon.model.INetworkCallList;

/**
 * This instance holds the INetworkCalls that have been logged so far. It notifies the
 * registered listeners when calls are added and updated.
 * 
 * @author Gino Bustelo
 *
 */
public class NetworkCallListImpl implements INetworkCallList {

	protected Set listeners = new HashSet();
	
	/*
	 * Log of calls, This is what viewers will have access to
	 */
	protected ArrayList calls = new ArrayList();
	
	/*
	 * Calls are places on this map when they are first added and held here until
	 * the map is removed. This helps the platform specific (Mozilla) network sniffer 
	 * (MozNetworkMonitorAdapter) find the call object when the response notification 
	 * comes in.
	 */
	protected HashMap ongoingCalls = new HashMap();
	
	public void add(INetworkCall call, Object mapKey) {
		
		//add to our list
		calls.add( call );
		
		//create the mapping
		ongoingCalls.put( mapKey, call );
		
		//notify listeners
		fireCallAddedNotification( call );
	}
	
	public void clear() {
		calls.clear();
		ongoingCalls.clear();
	}

	public INetworkCall get(Object mapKey) {
		return (INetworkCall)ongoingCalls.get( mapKey );
	}

	public INetworkCall[] getAll() {
		INetworkCall [] callsArray = new INetworkCall[ calls.size() ];
		return (INetworkCall [])calls.toArray( callsArray );
	}
	
	public void removeMap(Object mapKey) {
		ongoingCalls.remove( mapKey );
	}

	/**
	 * This implementation just passes the notification along to the
	 * listening INetworkMonitors
	 */
	public void update(INetworkCall call) {
		fireCallUpdateNotification( call );
	}
	
	public void addListener(INetworkMonitor netMonitor) {
		listeners.add( netMonitor );
	}

	public void removeListener(INetworkMonitor netMonitor) {
		listeners.remove( netMonitor );
	}
	
	private void fireCallAddedNotification(INetworkCall call) {
		Iterator iterator = listeners.iterator();
		while (iterator.hasNext())
			((INetworkMonitor) iterator.next()).notifyCallAdded(call);
	}
	
	private void fireCallUpdateNotification(INetworkCall call) {
		Iterator iterator = listeners.iterator();
		while (iterator.hasNext())
			((INetworkMonitor) iterator.next()).notifyCallUpdated(call);
	}
}
