/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/

package org.eclipse.atf.ui.debug;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;

import org.eclipse.atf.mozilla.ide.debug.MozillaDebugPlugin;
import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.ui.UIPlugin;
import org.eclipse.atf.util.ErrorStatusUtilities;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.LaunchConfigurationDelegate;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.ServerPort;
import org.eclipse.wst.server.core.internal.IClient;
import org.eclipse.wst.server.core.internal.ServerPlugin;
import org.eclipse.wst.server.http.core.internal.HttpServer;
import org.eclipse.wst.server.ui.internal.Messages;


public class WebLaunchConfigurationDelegate extends LaunchConfigurationDelegate /*implements IServerListener*/ {

	protected String appPath;
	protected String projectName;
	protected IProject project;
	protected String applicationName;
	protected String moduleName;
	protected boolean isUrl;
	protected ILaunch launch;
	protected String serverMode = "run";
	protected String mode = "run";
	protected boolean useServerOption;
	private IServer server = null;
	
	public void launch(ILaunchConfiguration configuration, String mode,
			ILaunch launch, IProgressMonitor monitor) throws CoreException {
		
		this.launch = launch;
		this.mode = mode;
		int work = 100;
		monitor.beginTask("Launching web application", work);

		// TODO remove when multiple debug targets supported
		if (mode.equals("debug")){
			IDebugTarget[] targets = DebugPlugin.getDefault().getLaunchManager().getDebugTargets();
			for (int i = 0; i < targets.length; i++) {
				if (targets[i].getModelIdentifier().equals(MozillaDebugPlugin.DEBUG_MODEL_ID)) {
					if (!targets[i].isTerminated()){
						String errorMessage = "Multiple JavaScript Debug Sessions is not supported. " +
								"Terminate the active session before starting a new session.";
						IStatus status = new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								Status.ERROR, errorMessage, null);
						throw new CoreException (status);
					}
				}
			}
		} 
		
		appPath = configuration.getAttribute(ILaunchConfigurationConstants.APP_PATH, (String)null);
		applicationName = configuration.getAttribute(ILaunchConfigurationConstants.PROCESS_TYPE, (String)null);
		isUrl = configuration.getAttribute(ILaunchConfigurationConstants.IS_URL, false);
		useServerOption = configuration.getAttribute(ILaunchConfigurationConstants.SERVER_CONFIG_OPTION, true);
		if(isUrl) {
			launchApp();			
		} else if (!useServerOption) {
			launchApp();                         
		} else {
			projectName = configuration.getAttribute(ILaunchConfigurationConstants.PROJECT, (String)null);
			moduleName = configuration.getAttribute(ILaunchConfigurationConstants.WEB_MODULE, (String)null);
			project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
			retrieveAndLaunchServer(project, moduleName, serverMode, monitor);
		}
		
		monitor.worked(work/2);
		monitor.done();		

	}
	
	private void retrieveAndLaunchServer(IProject project, String moduleName, String mode, IProgressMonitor monitor) throws CoreException {
		server = ServerManager.getInstance().getServer(project, moduleName, mode, monitor);
		
		if(server == null) {
			/*final IStatus s = new Status(IStatus.ERROR,
									UIPlugin.PLUGIN_ID,
									IStatus.OK,
									"A server is not yet defined for the "+moduleName+" web module. To remedy this, open the \"Servers\" view, define a new server and add the "+moduleName+" module to the new server. ",
									null);
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					ErrorDialog.openError(
							UIPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
							"An unexpected error has occurred",
							"Error launching web application",
							s);
				}
			});*/
			return;
					
		}
			
		launchApp();
		
	}
	
	private void launchApp() throws CoreException {   
		
		URL url = null;
		if(isUrl) {
			try {
				url = new URL(appPath);
			} catch(MalformedURLException e) {
				throw new CoreException(
						new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								IStatus.OK, "Unable to parse URL:" + appPath, e));
			}
			Job mozillaJob;

			if("run".equals(mode)) {
				mozillaJob = new MozillaRunJob(applicationName, null, null, launch, url);
			}
			else {
				mozillaJob = new MozillaDebugJob(applicationName, null, null, launch, url);
			}
			mozillaJob.schedule();
			
		} else if (!useServerOption) {

			projectName = launch.getLaunchConfiguration().getAttribute(ILaunchConfigurationConstants.PROJECT, (String)null);
			project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
			IFile file = project.getFile(FlexibleProjectUtils.getWebContentPath(project,appPath));
//			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(FlexibleProjectUtils.getWebContentPath(project,appPath));
			if (!file.exists()){
				throw new CoreException(
						new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								IStatus.OK, "File doesn't exists in workspace: " + appPath, null));
			}
			String systemPath = "file://" + file.getLocation().toPortableString();
			try {
				url = new URL(systemPath);
			} catch(MalformedURLException e) {
				throw new CoreException(
						new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								IStatus.OK, "Malformed URL:" + systemPath, e));
			}
			
			Job mozillaJob;
			if("run".equals(mode)) {
				mozillaJob = new MozillaRunJob(applicationName, null, null, launch, url);
			}
			else {
				mozillaJob = new MozillaDebugJob(applicationName, null, null, launch, url);
			}
			mozillaJob.schedule();
			
		} else {
			url = getURL(moduleName, appPath);
			IModule[] modules = server.getModules();
			IModule module = null;
			for(int i=0; i<modules.length; i++) {
				if(modules[i].getName().equals(moduleName)) {
					module = modules[i];
					break;
				}
			}

			if (module != null) {
				Job mozillaJob;
				if ("run".equals(mode)) {
					mozillaJob = new MozillaRunJob(applicationName, server, new IModule[] {module}, launch, url);				
				} else {
					mozillaJob = new MozillaDebugJob(applicationName, server, new IModule[] {module}, launch, url);
				}
	
				//TODO: only do this for locally hosted projects?
				ServerManager.getInstance().startServer(server, "run", mozillaJob, getActiveWorkbenchWindow().getShell());
			} else {
				throw new CoreException(
						new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
								IStatus.OK, "The selected file can not be launched because the project has not been added to a server.", null));
			}
		}
	}
	
		
	/**
	 * Returns the launchable clients for the given server and launchable
	 * object.
	 *
	 * @param server org.eclipse.wst.server.core.IServer
	 * @param launchable
	 * @param launchMode String
	 * @return an array of clients
	 */
	public static IClient[] getClients(IServer server, Object launchable, String launchMode) {
		ArrayList list = new ArrayList();
		IClient[] clients = ServerPlugin.getClients();
		if (clients != null) {
			int size = clients.length;
			for (int i = 0; i < size; i++) {
				if (clients[i].supports(server, launchable, launchMode))
					list.add(clients[i]);
			}
		}
		
		IClient[] clients2 = new IClient[list.size()];
		list.toArray(clients2);
		return clients2;
	}
	
	/**
	 * Open a message dialog.
	 * 
	 * @param shell
	 * @param message
	 * @return a dialog return constant
	 */
	protected int openWarningDialog(Shell shell, String message) {
		MessageDialog dialog = new MessageDialog(shell, Messages.errorDialogTitle, null,
			message,	MessageDialog.WARNING, new String[] {Messages.dialogModeWarningRestart,
			Messages.dialogModeWarningContinue, IDialogConstants.CANCEL_LABEL}, 0);
		return dialog.open();
	}
	
		
	private URL getURL(String module, String appPath) throws CoreException {
		
		URL url = null;
		int port = 8080;
		
		if(server != null) {
			if(server.getServerType().getId().equals(HttpServer.ID)) {
				port = server.getServerPorts(null)[0].getPort();
				String host = server.getHost();
				String contextRoot = FlexibleProjectUtils.getServerContextRoot(project);
				if( contextRoot == null ) {
					contextRoot = module;
				}
				String urlPrefix = ((HttpServer)server.loadAdapter(HttpServer.class, null)).getURLPrefix();
				String urlPath = !urlPrefix.equals("") ? "/"+urlPrefix : "";
				urlPath += !contextRoot.equals("") ? "/"+contextRoot+"/"+appPath : "/"+appPath;
				try {
					// Special case Mozilla's handling of port 80
					if (port == 80 || port == -1) {
						url = new URL("http", host, urlPath);
					}else{
						url = new URL("http", host, port, urlPath);
					}
				} catch(MalformedURLException e) {
					throw new CoreException(
							new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
									IStatus.ERROR, "Unable to parse URL", e));
				}
			} else {
				ServerPort[] serverPorts = server.getServerPorts(null);
				for(int i=0; i<serverPorts.length; i++) {
					if(serverPorts[i].getProtocol().equals("http")) {
						port = serverPorts[i].getPort();
						break;
					}
				}
				try {
					url = new URL("http", "localhost", port, "/" + module + "/" + appPath);
				}
				catch(MalformedURLException e) {
					throw new CoreException(
							new Status(IStatus.ERROR, UIPlugin.PLUGIN_ID,
									IStatus.OK, "Unable to parse URL", e));
				}
			}
		}
		
		return url;
	}
	
	private void showError(String title, String message, Exception e) {
		
		MultiStatus multiStatus = ErrorStatusUtilities.createMultiStatusFromException(e, this.getClass(), UIPlugin.PLUGIN_ID, message);
		ErrorDialog.openError(null, title, message, multiStatus);	
				
	}
	
	private IWorkbenchWindow getActiveWorkbenchWindow() {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if(window == null) {
			IWorkbenchWindow[] windows = PlatformUI.getWorkbench().getWorkbenchWindows();
			if(windows.length > 0)
				window = windows[0];
		}
		return window;
	}
	
}
