/*******************************************************************
 *
 * Licensed Materials - Property of IBM
 * 
 * AJAX Toolkit Framework 6-28-496-8128
 * 
 * (c) Copyright IBM Corp. 2006 All Rights Reserved.
 * 
 * U.S. Government Users Restricted Rights - Use, duplication or 
 * disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *******************************************************************/

package org.eclipse.atf.ui.debug;

import org.eclipse.atf.mozilla.ide.debug.preferences.JSDebugPreferenceNames;
import org.eclipse.atf.project.FlexibleProjectUtils;
import org.eclipse.atf.ui.UIPlugin;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.model.WorkbenchLabelProvider;

public class WebLaunchConfigurationTab extends AbstractLaunchConfigurationTab {

	protected Text pathField;
	protected Label appLabel;
	protected Label browserLabel;
	protected String applicationType = "Mozilla";
	protected String[] requiredNatures = new String[0];
	private String appPath = "";
	private Button appButton;
	private Button urlTypeButton;
	private Button fileTypeButton;
	private Button serverOptionButton;
	private DebugPreferencesGroup debugPrefs;
	
	protected boolean isURL;
	protected boolean useServerConfig;
	private String filePath = "", urlPath = "";
			
	private class WebLaunchLabelProvider extends WorkbenchLabelProvider {

		protected String decorateText(String input, Object element) {
			if(element instanceof IFile) {
				IFile file = (IFile)element;
				String filename = file.getName();
				return filename;
			}
			return input;
		}
	}
	
	public WebLaunchConfigurationTab(String applicationType, String[] requiredNatures, String mode) {
		super();
		this.applicationType = applicationType;
		this.requiredNatures = requiredNatures;
	}	
	
	public boolean isValid(ILaunchConfiguration launchConfig) {
		return getErrorMessage() == null;
	}

	public boolean canSave() {
		return getErrorMessage() == null;
	}

	public void createControl(Composite parent) {
		Composite tab = new Composite(parent, SWT.NONE );
		tab.setFont(parent.getFont());
		tab.setLayout(new GridLayout());
		tab.setLayoutData(new GridData(GridData.FILL_BOTH));
		setControl(tab);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);

		Group webGroup = new Group(tab, SWT.NONE);
		webGroup.setFont(parent.getFont());
		gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		webGroup.setLayout(gridLayout);
		webGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		webGroup.setText("Debug Target:");

		Composite radioGroup = new Composite(webGroup, SWT.NONE);
		radioGroup.setLayout(new RowLayout());
		fileTypeButton = new Button(radioGroup, SWT.RADIO);
		fileTypeButton.setFont(parent.getFont());
		fileTypeButton.setText("Project file"); //$NON-NLS-1$
		urlTypeButton = new Button(radioGroup, SWT.RADIO);
		urlTypeButton.setFont(parent.getFont());
		urlTypeButton.setText("URL"); //$NON-NLS-1$
		
		urlTypeButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				if (urlTypeButton.getSelection()) {
					if (!isURL) {
						isURL = true;
						appButton.setVisible(false);
						serverOptionButton.setVisible(false);
						filePath = pathField.getText();
						if(urlPath != null) 
							pathField.setText(urlPath);
					} 
				}
				updateWidgetEnablements();
				updateLaunchConfigurationDialog();
			}
		});
		fileTypeButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				if (fileTypeButton.getSelection()) {
					if (isURL) {
						isURL = false;
						appButton.setVisible(true);
						serverOptionButton.setVisible(true);
						urlPath = pathField.getText();
						if(filePath != null)
							pathField.setText(filePath);
					}
				}
				updateWidgetEnablements();
				updateLaunchConfigurationDialog();
			}
		});
		
		Composite temp = new Composite(webGroup, SWT.NULL);
		gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		temp.setLayout(gridLayout);
		temp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		pathField = new Text(temp, SWT.SINGLE | SWT.BORDER);
		data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);
		pathField.setLayoutData(data);
		pathField.setFont(parent.getFont());
		pathField.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent evt) {
				appPath = pathField.getText();
				updateWidgetEnablements();
				updateLaunchConfigurationDialog();
				
			}
		});
		
		appButton = createPushButton(temp, "Browse...", null);
		appButton.setVisible(!isURL);
		appButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				handleAppButtonSelected();
				updateWidgetEnablements();
			}
		});
		
		Composite serverOptionGroup = new Composite(webGroup, SWT.NONE);
		serverOptionGroup.setLayout(new RowLayout());				
		serverOptionButton = new Button(serverOptionGroup, SWT.CHECK);
		serverOptionButton.setFont(parent.getFont());
		serverOptionButton.setText("Load from Server");
		String toolTip = "Determines whether a server (Apache, Tomcat, etc,) should be used to run the project. \n" + 
						 "If the checkbox is set, then the project will be loaded from a server with a URL like \n" +
						 "http://hostname/projectName/fileName. If the checkbox is not set, the project is run \n" +
						 "using a file URL like file://c:/<eclipseInstallDirectory>/projectName/fileName.";
		serverOptionButton.setToolTipText(toolTip);
		serverOptionButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent evt) {
				updateLaunchConfigurationDialog();
			}
		});
	
		debugPrefs = new DebugPreferencesGroup(tab, this,  SWT.NONE);		
	}
	
	protected void updateWidgetEnablements() {
		
		appButton.setEnabled(true);
		setErrorMessage(null);
		
		if(isURL) {
			if(pathField.getText().equals("")) {
				setErrorMessage("A valid URL must be specified");
			} 
		} else if(pathField.getText().equals("") || !isValidPath(pathField.getText())) {
				setErrorMessage("A valid workspace file must be specified");
		} 
	}
	
	protected void handleAppButtonSelected() {
		
		// Figure out the right initial directory for the file dialog, based
		// on the project field
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		ApplicationFileSelectionDialog dialog = new ApplicationFileSelectionDialog(
											getShell(), 
											new WebLaunchLabelProvider(),
											"Web application file selection",
											"Select a target web application file from the project's contents",
											projects,
											requiredNatures,
											false);
		dialog.setInput(projects);
		dialog.setAllowMultiple(false);
//		dialog.setSorter(new ResourceSorter(ResourceSorter.TYPE));
		dialog.open();
		if(dialog.getReturnCode() == Window.OK) {
			Object[] targetFiles = dialog.getResult();
			IFile targetFile = (IFile)targetFiles[0];
			if(targetFile != null) {
				pathField.setText(targetFile.getProject().getName()+"/"+targetFile.getProjectRelativePath());
				updateLaunchConfigurationDialog();
			}
		}
	}
	
	protected IPath computeDefaultDirectory(IPath projectPath) {
		return projectPath;
	}
	
	protected boolean isValidProject(String projName) {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projName);
		if(project.exists())
			return true;
		return false;
	}
	
	protected boolean isValidPath(String path) {
		try {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(path));
			if(file.exists()) {
				return true;
			}
		} catch( Exception e ) {}
		return false;
	}

	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		isURL = false;
		configuration.setAttribute(JSDebugPreferenceNames.OVERRIDE_PREFERENCES, false);
	}

	public void initializeFrom(ILaunchConfiguration config) {
		try {
			//URL type
			debugPrefs.initialize(config);
			
			isURL =
				config.getAttribute(
					ILaunchConfigurationConstants.IS_URL,
					false);
			urlTypeButton.setSelection(isURL);
			//project
			String projectName =
				config.getAttribute(
					ILaunchConfigurationConstants.PROJECT,
					"");
			//web module
			String webModule =
				config.getAttribute(
					ILaunchConfigurationConstants.WEB_MODULE,
					"");
			
			//App file path
			appPath = 
				config.getAttribute(
						ILaunchConfigurationConstants.APP_PATH,
						"");
			
			
			if(isURL) {
				pathField.setText(appPath);
			} else {
				if( !config.getAttribute(ILaunchConfigurationConstants.FULL_PATH, "").equals("") ) {
					pathField.setText(config.getAttribute(ILaunchConfigurationConstants.FULL_PATH, ""));
				} else {
					String fullAppPath = getFullAppPath(projectName, webModule, new Path(appPath));
					pathField.setText(fullAppPath == null ? "" : projectName+"/"+fullAppPath);
				}
			}
			
			//File type
			fileTypeButton.setSelection(!isURL);
			appButton.setVisible(!isURL);

			//App file path
			useServerConfig = 
				config.getAttribute(
						ILaunchConfigurationConstants.SERVER_CONFIG_OPTION,
						true);
			
			serverOptionButton.setVisible(!isURL);
			serverOptionButton.setSelection(useServerConfig);
			
			updateWidgetEnablements();
			
		} catch (CoreException ce) {
			UIPlugin.log(ce);
		}
	}

	public void performApply(ILaunchConfigurationWorkingCopy config) {
			Path path = new Path(pathField.getText());
			String projectName = path.segment(0);
			config.setAttribute(
				ILaunchConfigurationConstants.PROJECT,
				projectName);
			config.setAttribute(
				ILaunchConfigurationConstants.WEB_MODULE,
				getModuleName(projectName, new Path(appPath)));
			
			
			if( isURL ){
				config.setAttribute(ILaunchConfigurationConstants.APP_PATH, getAppPath(null, new Path(appPath)));
			}
			else{
				try {
					IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(appPath));
					String wcRelativePath = FlexibleProjectUtils.getWebContentRelativePath(file);
					config.setAttribute( ILaunchConfigurationConstants.APP_PATH, wcRelativePath );
				} catch( Exception e ) {}
			}
			config.setAttribute(
				ILaunchConfigurationConstants.PROCESS_TYPE, 
				getName());
			config.setAttribute(
				ILaunchConfigurationConstants.IS_URL,
				urlTypeButton.getSelection());
			config.setAttribute(ILaunchConfigurationConstants.FULL_PATH, pathField.getText());
			
			config.setAttribute(ILaunchConfigurationConstants.SERVER_CONFIG_OPTION, serverOptionButton.getSelection());
					
			debugPrefs.apply(config);
	}
	
	protected String getModuleName(String project, IPath projectRelativePath) {
		if(project == null || project.equals("") || projectRelativePath == null || projectRelativePath.segmentCount() == 0)
			return null;
		return FlexibleProjectUtils.getModuleName(project, projectRelativePath);
	}
	
	protected String getAppPath(String project, IPath projectRelativePath) {
		if(isURL)
			return projectRelativePath.toString();
		if(project == null || project.equals("") || projectRelativePath == null || projectRelativePath.segmentCount() == 0)
			return null;
		IProject p = ResourcesPlugin.getWorkspace().getRoot().getProject(project);
		if(p == null)
			return null;
		IFile f = p.getFile(projectRelativePath);
		if(f == null)
			return null;
		return FlexibleProjectUtils.getWebContentRelativePath(f);
	}
	
	protected String getFullAppPath(String project, String modulename, IPath webContentRelativePath) {
		if(project == null || project.equals(""))
			return null;
		IProject p = ResourcesPlugin.getWorkspace().getRoot().getProject(project);
		if(p == null)
			return null;
		return FlexibleProjectUtils.getProjectRelativePath(project, modulename, webContentRelativePath);
	}

	public String getName() {
		return applicationType + " Configuration";
	}
	
	public void update() {
		updateLaunchConfigurationDialog();
	}
		

}
