package org.eclipse.atf.ui.properties;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.atf.adapter.IWebResourceLocator;
import org.eclipse.atf.project.ProjectUtils;
import org.eclipse.atf.runtime.IRuntime;
import org.eclipse.atf.runtime.IRuntimeContainer;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.atf.runtime.RuntimeInstance;
import org.eclipse.atf.runtime.RuntimeManager;
import org.eclipse.atf.runtime.installer.FileCopyRuntimeInstaller;
import org.eclipse.atf.runtime.installer.IRuntimeInstaller;
import org.eclipse.atf.runtime.validator.IRuntimeValidator;
import org.eclipse.atf.runtime.version.IVersionFinder;
import org.eclipse.atf.ui.runtime.ToolkitSelectionField;
import org.eclipse.atf.ui.runtime.standins.RuntimeContainerStandin;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchPropertyPage;
import org.eclipse.ui.dialogs.PropertyPage;

public class InstalledToolkitsProperties extends PropertyPage implements
		IWorkbenchPropertyPage
{
	private ToolkitSelectionField selector;
	private ToolkitsInputProvider inputProvider;
	private IProject project;
	protected ArrayList toolkitsToAdd;
	protected ArrayList toolkitsToDelete;

	// status of toolkit installation
	public static final int BACKUP = 0;
	public static final int INSTALL = 1;
	public static final int DELETE = 2;

	public InstalledToolkitsProperties()
	{
		super();

		inputProvider = new ToolkitsInputProvider( new RuntimeContainerStandin() );
		noDefaultAndApplyButton();
		
		int length = inputProvider.getRuntimeInstances().length;
		toolkitsToAdd = new ArrayList( length );
		toolkitsToDelete = new ArrayList( length );
	}

	protected Control createContents( Composite parent )
	{
		final Composite composite = new Composite( parent, SWT.NONE );

		// create layout for control
		GridLayout gridLayout = new GridLayout();
		composite.setLayout( gridLayout );

		// explanatory text
		Label label = new Label( composite, SWT.NONE );
		label.setText( "Select the JavaScript toolkit to use in this project." );

		// create toolkit selection table and buttons
		selector = new ToolkitSelectionField( false, SWT.CHECK )
		{
			protected void setStatus( IStatus status )
			{
				if ( status.getSeverity() == IStatus.OK )
				{
					setErrorMessage( null );
					setValid( true );
					return;
				}
				
				// severity is ERROR
				setErrorMessage( status.getMessage() );
				setValid( false );
			}
		};
		selector.createControl( composite );

		GridData data = new GridData( SWT.FILL, SWT.FILL, true, true );
		selector.setLayoutData( data );

		Object[] installedToolkits = getInstalledToolkits();

		// set input
		selector.setInput( inputProvider );

		// enable checkmark next to currently installed toolkit
		if ( installedToolkits != null )
		{
			selector.setCheckedToolkits( installedToolkits );
		}

		// listen for changes in checked toolkits
		selector.addCheckStateListener( new ICheckStateListener()
		{
			public void checkStateChanged( CheckStateChangedEvent event )
			{
				Object element = event.getElement();
				if ( event.getChecked() )
				{
					// element is checked (to be added)
					if ( ! toolkitsToDelete.remove( element ) )
						toolkitsToAdd.add( element );
				}
				else
				{
					// element is unchecked (to be deleted)
					if ( ! toolkitsToAdd.remove( element ) )
						toolkitsToDelete.add( element );
				}
			}
		});
		return composite;
	}

	/**
	 * Returns the an object that represents the toolkit ("runtime") that is
	 * installed in the project. The minimum matching criteria are the toolkit
	 * type and version, but the code will also try to map the "pretty name" for
	 * the toolkit instance.
	 * 
	 * If none of the stored toolkits matches that installed in the project,
	 * then this method returns a minimal object representing the project
	 * toolkit.
	 * 
	 * @return object representing toolkit installed in project
	 */
	private Object[] getInstalledToolkits()
	{
		// retrieve info on toolkit installed in project
		project = (IProject) getElement().getAdapter( IResource.class );
		Object[] installedToolkits = null;
		try
		{
			installedToolkits = ProjectUtils.loadSettings( project );
		}
		catch ( CoreException e )
		{
		}

		if ( installedToolkits == null )
			return null;

		// see if the toolkit installed in the project is one that is listed
		// in preferences
		ArrayList toolkits = new ArrayList( installedToolkits.length );
		IRuntimeInstance[] prefToolkits = inputProvider.getRuntimeInstances();

		for ( int i = 0; i < installedToolkits.length; i++ )
		{
			IRuntimeInstance installedTK = (IRuntimeInstance) installedToolkits[ i ];
			IRuntimeInstance match = null;
			
			boolean found = false;
			for ( int j = 0; j < prefToolkits.length; j++ )
			{
				IRuntimeInstance prefTK = prefToolkits[ j ];
				
				if ( ! installedTK.getType().getId().equals( prefTK.getType().getId() ))
					continue;
				
				if ( installedTK.getVersion().equals( prefTK.getVersion() ) )
				{
					// Only consider this a "match" if the prefs toolkit
					// is valid.  Otherwise, show the project toolkit as
					// a separate item.
					boolean isValid = true;
			    	IRuntimeValidator validator = prefTK.getType().getValidator();
			    	if ( validator != null )
			    	{
			    		IStatus validationStatus = validator.validate( prefTK );
			    		
						if ( ! validationStatus.isOK() )
							isValid = false;
			    	}

			    	if ( isValid )
			    	{
						// Found a listed toolkit of same type and version as that
						// installed in project. If they share the same name, then
						// return this one immediately. If not, save this toolkit
						// while we see if another listed toolkit has the same name.
						if ( installedTK.getName().equals( prefTK.getName() ) )
						{
							// Project toolkit matches prefs toolkit, which
							// is valid.
							toolkits.add( prefTK );
							found = true;
							break;
						}
	
						if ( match == null )
							match = prefTK;
			    	}
				}
			}
			
			if ( found )
				continue;

			// found toolkit that matches type & version, but not name
			if ( match != null )
			{
				toolkits.add( match );
				continue;
			}

			// User doesn't have any toolkits in preferences that match that
			// installed in project. So add a new entry, with location set
			// to project relative path.
			String location = null;
			IWebResourceLocator adapter = (IWebResourceLocator) project.getAdapter( IWebResourceLocator.class );
			if ( adapter != null )
			{
				IContainer webContentRoot = adapter.getWebResourceContainer();
				IRuntimeInstaller installer = installedTK.getType().getInstaller();
				IPath installPath = installer.getProjectInstallPath( installedTK );
				IPath fullPath = webContentRoot.getFullPath().append( installPath );
				location = "platform:/resource" + fullPath.toPortableString();
			}
			
			if ( location == null )
			{
				// XXX what do we do if there is no adapter?
				location = "[Project: " + project.getName() + "]";
			}
			
			( (RuntimeInstance) installedTK ).setLocation( location );
			inputProvider.addExtraToolkit( installedTK );
			toolkits.add( installedTK );
		}

		return toolkits.toArray();
	}

	public boolean performOk()
	{
		final Object[] checkedElements = selector.getCheckedElements();

		if ( (toolkitsToAdd != null && ! toolkitsToAdd.isEmpty()) ||
				(toolkitsToDelete != null && ! toolkitsToDelete.isEmpty()) )
		{
			// create new job to install new toolkits and delete the old
			Job changeToolkitsJob = new Job( "Modify Installed Toolkits" )
			{
				protected IStatus run( IProgressMonitor monitor )
				{
					monitor.beginTask( "Modify Installed Toolkits", 100 );
	
					// XXX need to implement cancelling
	
					// XXX need to handle error during install/uninstall.
					//     roll back to previous state.

					try
					{
						RuntimeManager manager = RuntimeManager.getInstance();

						// store the settings in the project
						monitor.subTask( "Saving settings..." );
						ProjectUtils.saveSettings( project, checkedElements );
						monitor.worked( 5 );

						if ( toolkitsToDelete != null && ! toolkitsToDelete.isEmpty() )
						{
							SubProgressMonitor sub = new SubProgressMonitor( monitor, 45 );
							manager.uninstallRuntimes( toolkitsToDelete.toArray(), project, sub );
						}
	
						if ( toolkitsToAdd != null && ! toolkitsToAdd.isEmpty() )
						{
							SubProgressMonitor sub = new SubProgressMonitor( monitor, 45 );
							manager.installRuntimes( toolkitsToAdd.toArray(), project, sub );
						}
					}
					catch ( CoreException e )
					{
						return new Status( IStatus.ERROR, "org.eclipse.atf.ui",
								"Failed to install/uninstall toolkits" );
					}
	
					monitor.done();
					return new Status( IStatus.OK, "org.eclipse.atf.ui",
							"New toolkits installed fine" );
				}
	
			};
			changeToolkitsJob.setUser( true );
			changeToolkitsJob.schedule();
		}

		return super.performOk();
	}

}

/**
 * Wraps the given IRuntimeContainer and adds our temporary toolkit instance to
 * the list of instances that are saved by the specific toolkit type.
 */
class ToolkitsInputProvider extends RuntimeContainerStandin
{
	private IRuntimeContainer container;
	private ArrayList extraInstances = null;

	public ToolkitsInputProvider( IRuntimeContainer container )
	{
		this.container = container;
	}

	public IRuntime getRuntime( String type )
	{
		IRuntime runtime = container.getRuntime( type );
		if ( extraInstances == null )
			return runtime;

		ArrayList runtimeInstances = new ArrayList();
		Iterator iterator = extraInstances.iterator();
		while ( iterator.hasNext() )
		{
			IRuntimeInstance instance = (IRuntimeInstance) iterator.next();
			if ( instance.getType().getId().equals( type ) )
				runtimeInstances.add( instance );
		}
		
		if ( runtimeInstances.isEmpty() )
			return runtime;
		
		// XXX If 'runtime' is null, then the toolkit type for 'extraInstance'
		// is not supported. Need to put up error message/dialog.

		// Asking for same type as our extra instance. We need to add the
		// instance to the instance iterator of the runtime.
		return new RuntimeWithExtraInstances( runtime, runtimeInstances.toArray() );
	}

	public IRuntime[] getRuntimes()
	{
		IRuntime [] runtimes = container.getRuntimes();
		
		//if there are no extras then return what is already configured
		if( extraInstances == null )
			return runtimes;
		
		for (int i = 0; i < runtimes.length; i++) {
			
			//find the runtime type that mathches this instance and wrap it

			/*
			 * XXX Need to support the case where the extraInstance's Runtime type is not available
			 * 
			 * see comment on getRuntime() method
			 */

			runtimes[ i ] = getRuntime( runtimes[ i ].getId() );
		}
		
		return runtimes;
	}

	public void addExtraToolkit( IRuntimeInstance runtimeInstance )
	{
		if ( extraInstances == null )
			extraInstances = new ArrayList();
		extraInstances.add( runtimeInstance );
	}
}

/**
 * Wraps the given toolkit ("runtime") and adds our temporary toolkit instance
 * to its list of instances.
 */
class RuntimeWithExtraInstances implements IRuntime
{
	private IRuntime runtime;
	private Object[] extraInstances;

	public RuntimeWithExtraInstances( IRuntime runtime, Object[] extraInstances )
	{
		this.runtime = runtime;
		this.extraInstances = extraInstances;
	}

	public String getId()
	{
		return runtime.getId();
	}

	public IRuntimeInstaller getInstaller()
	{
		return runtime.getInstaller();
	}

	public boolean isUser() {
		return runtime.isUser();
	}
	
	/*
	 * Adds the extra instance to the list of instances
	 * 
	 * (non-Javadoc)
	 * @see org.eclipse.atf.runtime.IRuntime#getRuntimeInstances()
	 */
	public IRuntimeInstance[] getRuntimeInstances()
	{
		IRuntimeInstance [] instances = runtime.getRuntimeInstances();
		IRuntimeInstance [] instancesPlusExtra =
			new IRuntimeInstance[ instances.length + extraInstances.length ];
		
		System.arraycopy( instances, 0, instancesPlusExtra, 0, instances.length );
		System.arraycopy( extraInstances, 0, instancesPlusExtra, instances.length, extraInstances.length );

		return instancesPlusExtra;
	}

	public String getName()
	{
		return runtime.getName();
	}

	public IRuntimeValidator getValidator()
	{
		return runtime.getValidator();
	}

	public IVersionFinder getVersionFinder()
	{
		return runtime.getVersionFinder();
	}

	public boolean isAllowUserInstances()
	{
		return runtime.isAllowUserInstances();
	}
}