package org.eclipse.atf.ui.runtime;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.atf.runtime.IRuntime;
import org.eclipse.atf.runtime.IRuntimeInstance;
import org.eclipse.atf.runtime.validator.IRuntimeValidator;
import org.eclipse.atf.runtime.version.IVersionFinder;
import org.eclipse.atf.ui.runtime.standins.RuntimeContainerStandin;
import org.eclipse.atf.ui.runtime.standins.RuntimeInstanceStandin;
import org.eclipse.atf.ui.runtime.standins.RuntimeStandin;
import org.eclipse.atf.ui.util.StatusInfo;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

public class AddRuntimeInstanceDialog extends StatusDialog {

	//private static final String DEFAULT_TYPE_TEXT = "<Select a Runtime Type>";
	
	protected Combo typeCombo = null;
	protected Text nameText = null;
	protected Text versionText = null;
	protected Text locationText = null;
	
	//validation status
	protected IStatus [] validationStatus = new IStatus[4];
	protected static final int TYPE_STATUS = 0;
	protected static final int NAME_STATUS = 1;
	protected static final int VERSION_STATUS = 2;
	protected static final int LOCATION_STATUS = 3;
	
	//Strings
	private static final String TYPE_LABEL_TEXT = "Type:";
	private static final String NAME_LABEL_TEXT = "Name:";
	private static final String VERSION_LABEL_TEXT = "Version:";
	private static final String LOCATION_LABEL_TEXT = "Location:";
	private static final String BROWSE_BUTTON_TEXT = "Browse...";
	
	private static final String NEW_RUNTIME_BUTTON_TEXT = "New...";
	private static final String NEW_RUNTIME_DIALOG_TITLE = "New AJAX Toolkit Type";
	
	//instance for editing
	protected RuntimeContainerStandin runtimeContainerStandin = null;
	protected RuntimeInstanceStandin editRI = null;
	
	protected IRuntime [] runtimes;
	
	public AddRuntimeInstanceDialog(Shell parent, RuntimeContainerStandin runtimeContainerStandin, RuntimeInstanceStandin editRI ) {
		super(parent);
		
		this.runtimeContainerStandin = runtimeContainerStandin;
		this.runtimes = runtimeContainerStandin.getRuntimes();
		
		this.editRI = editRI;
		
	}
	
	public void create() {
		super.create();
		nameText.setFocus();
		
		/*
		 * Force an update because the superclass would not allow to start with errors
		 */
		updateStatusLine();
	}

	protected Control createDialogArea(Composite parent) {
		
		Composite displayArea = (Composite)super.createDialogArea(parent);
		((GridLayout)displayArea.getLayout()).numColumns= 3;
		
		/*
		 * Runtime Type selection Combo
		 */
		Label typeLabel = new Label( displayArea, SWT.LEFT | SWT.WRAP );
		typeLabel.setFont( displayArea.getFont() );
		typeLabel.setText( TYPE_LABEL_TEXT );
		
		GridData layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		typeLabel.setLayoutData(layoutData);
		
		typeCombo = new Combo( displayArea, SWT.READ_ONLY );

		typeCombo.setFont( displayArea.getFont() );
		
		for (int i = 0; i < runtimes.length; i++) {
			IRuntime r = runtimes[i];
			typeCombo.add( r.getName() );
		}
		
		if( editRI != null ){
			int idx = typeCombo.indexOf( editRI.getType().getName() );
			typeCombo.select( idx );
			typeCombo.setEnabled( false );
		}
		
		typeCombo.addModifyListener( new ModifyListener(){

			public void modifyText(ModifyEvent e) {
				validationStatus[TYPE_STATUS] = validateType();
				
				//name validation is tied to the type
				validationStatus[NAME_STATUS] = validateName();
				
				//must validate location also
				validationStatus[LOCATION_STATUS] = validateLocation();
				updateStatusLine();
				
				updateVersionField();
			}
			
		});
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.widthHint = convertWidthInCharsToPixels(50);
		typeCombo.setLayoutData(layoutData);
		
		//new Runtime button
		Button newRuntimeButton = new Button( displayArea, SWT.PUSH );
		newRuntimeButton.setText( NEW_RUNTIME_BUTTON_TEXT );
		newRuntimeButton.setFont( parent.getFont() );
		setButtonLayoutData( newRuntimeButton );
		
		newRuntimeButton.addListener( SWT.Selection, new Listener() {
			public void handleEvent( Event event )
			{
				AddRuntimeInstanceDialog.this.handleNewRuntimeButton();
			}
		});
		
		//separator
		Label separator = new Label( displayArea, SWT.SEPARATOR | SWT.SHADOW_NONE );
		layoutData = new GridData();
		layoutData.horizontalSpan = 3;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.heightHint = 10;
		separator.setLayoutData(layoutData);
		
		/*
		 * Runtime Instance name
		 */
		Label nameLabel = new Label( displayArea, SWT.LEFT | SWT.WRAP );
		nameLabel.setFont( displayArea.getFont() );
		nameLabel.setText( NAME_LABEL_TEXT );
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		nameLabel.setLayoutData(layoutData);
		
		nameText = new Text( displayArea, SWT.SINGLE | SWT.BORDER );
		nameText.setFont( displayArea.getFont() );
		nameText.setTextLimit( 30 );
		
		if( editRI != null ){
			nameText.setText( editRI.getName() );
		}
		
		nameText.addModifyListener( new ModifyListener(){

			public void modifyText(ModifyEvent e) {
				validationStatus[NAME_STATUS] = validateName();
				updateStatusLine();
			}
		});
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 2;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.widthHint = convertWidthInCharsToPixels(50);
		nameText.setLayoutData(layoutData);
		
		/*
		 * Runtime Instance version
		 */
		Label versionLabel = new Label( displayArea, SWT.LEFT | SWT.WRAP );
		versionLabel.setFont( displayArea.getFont() );
		versionLabel.setText( VERSION_LABEL_TEXT );
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		versionLabel.setLayoutData(layoutData);
		
		versionText = new Text( displayArea, SWT.SINGLE | SWT.BORDER );
		versionText.setFont( displayArea.getFont() );
		versionText.setTextLimit( 20 );
		
		if( editRI != null ){
			versionText.setText( editRI.getVersion() );
		}
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 2;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.widthHint = convertWidthInCharsToPixels(50);
		versionText.setLayoutData(layoutData);
		
		/*
		 * Runtime Instance location
		 */
		Label locationLabel = new Label( displayArea, SWT.LEFT | SWT.WRAP );
		locationLabel.setFont( displayArea.getFont() );
		locationLabel.setText( LOCATION_LABEL_TEXT );
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		locationLabel.setLayoutData(layoutData);
		
		locationText = new Text( displayArea, SWT.SINGLE | SWT.BORDER );
		locationText.setFont( displayArea.getFont() );

		if( editRI != null ){
			
			try{
				URL url = new URL( editRI.getLocation() );
				
				locationText.setText( url.getPath() );
			}
			catch( MalformedURLException e ){
				locationText.setText( "" );
				e.printStackTrace();
			}
		}
		
		locationText.addModifyListener( new ModifyListener(){

			public void modifyText(ModifyEvent e) {
				validationStatus[LOCATION_STATUS] = validateLocation();
				updateStatusLine();
				updateVersionField();
			}
		});
		
		layoutData = new GridData();
		layoutData.horizontalSpan = 1;
		layoutData.horizontalAlignment = GridData.FILL;
		layoutData.grabExcessHorizontalSpace = true;
		layoutData.widthHint = convertWidthInCharsToPixels(50);
		locationText.setLayoutData(layoutData);
		
		Button locationBrowseButton = new Button( displayArea, SWT.PUSH );
		locationBrowseButton.setFont( displayArea.getFont() );
		setButtonLayoutData( locationBrowseButton );
		locationBrowseButton.setText( BROWSE_BUTTON_TEXT );
		locationBrowseButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event evt) {
				handleBrowseButton();
			}
		});
				
		//do the intial validation pass
		initValidationStatus();
		updateStatusLine();
		updateVersionField();
		
		return displayArea;
	}	
	
	protected void initValidationStatus(){
		
		if( editRI == null ){
			validationStatus[TYPE_STATUS] = validateType();		
		}
		else{
			validationStatus[TYPE_STATUS] = new StatusInfo(); //ok
		}
		
		validationStatus[NAME_STATUS] = validateName();	
		validationStatus[VERSION_STATUS] = new StatusInfo(); //ok (not validated for now)
		validationStatus[LOCATION_STATUS] = validateLocation();	
		
	}
	
	/*
	 * Sends the worst status to the status line
	 */
	protected void updateStatusLine() {
		IStatus max= null;
		for (int i= 0; i < validationStatus.length; i++) {
			IStatus curr= validationStatus[i];
			if (curr.matches(IStatus.ERROR)) {
				updateStatus(curr);
				return;
			}
			if (max == null || curr.getSeverity() > max.getSeverity()) {
				max= curr;
			}
		}
		updateStatus(max);
	}
	
	/*
	 * Update the value and READ_ONLY state of the Version field
	 */
	protected void updateVersionField(){
		if( validationStatus[TYPE_STATUS].isOK() ){
			
			RuntimeStandin runtime = getSelectedRuntime();
			final IVersionFinder versionFinder = runtime.getVersionFinder();
			
			if( versionFinder != null ){
				versionText.setEditable( false ); //no editing allowed if version is auto-detected
				
				//only set the version on the UI if the location is valid
				if( validationStatus[LOCATION_STATUS].isOK() ){
					
					/*
					 * Create a temporary instance that can be used to calculate the version
					 */
					final RuntimeInstanceStandin newRi = new RuntimeInstanceStandin();
					String location = locationText.getText();
					try {
						newRi.setLocation( (new File(location)).toURL().toString() );
						
						final String[] ret = new String[1]; 
						BusyIndicator.showWhile(getShell().getDisplay(), new Runnable() {
				            public void run() {
				                ret[0] = versionFinder.findVersion(newRi);
				            }
				        });
						if( ret[0] != null )
							versionText.setText( ret[0] );
						else
							versionText.setText( "" );
						
					} catch (MalformedURLException e) {
						e.printStackTrace();
					}
					
				}
				else{
					versionText.setText( "" );
					
				}
			}
			else{
				versionText.setEditable(true);
			}
			
		}
		else{
			//do not allow edits until a valid runtime type is selected
			versionText.setEditable(false);
		}
	}
	
	protected void handleNewRuntimeButton()
	{
		AddRuntimeTypeDialog dialog = new AddRuntimeTypeDialog( getShell(), runtimeContainerStandin, null );
		dialog.setTitle( NEW_RUNTIME_DIALOG_TITLE );
		
		if (dialog.open() != Window.OK) {
			return;
		}
		
		/*
		 * A new Runtime defined by the user is available... need to add it to the list
		 */
		typeCombo.removeAll();
		
		//get new set of runtimes and assume something was added if OK was pressed
		//also try to select the just added runtime
		int latestAddedIdx = 0;
		runtimes = this.runtimeContainerStandin.getRuntimes();
		for (int i = 0; i < runtimes.length; i++) {

			if( runtimes[i] == this.runtimeContainerStandin.getLatestRuntimeAdded() )
				latestAddedIdx = i;
			typeCombo.add( runtimes[i].getName() );
		}
		
		typeCombo.select( latestAddedIdx );
	}
	
	protected void okPressed() {
		
		//is an edit
		if( editRI != null ){
			String name = nameText.getText();
			String version = versionText.getText();
			String location = locationText.getText();
			
			editRI.setName( name );
			editRI.setVersion( version );
			try {
				editRI.setLocation( (new File(location)).toURL().toString() );
			} catch (MalformedURLException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			
		}
		//is a new instance
		else{
			RuntimeStandin selectedRuntime = getSelectedRuntime();
			String name = nameText.getText();
			String version = versionText.getText();
			String location = locationText.getText();
			
			try{
				RuntimeInstanceStandin newRi = new RuntimeInstanceStandin();
				newRi.setName( name );
				newRi.setVersion( version );
				newRi.setLocation( (new File(location)).toURL().toString() );
				selectedRuntime.addInstance( newRi );
			}
			catch( MalformedURLException e ){
				e.printStackTrace();
			}
		}
		
		super.okPressed();
	}
		
	
	protected IStatus validateName(){
		StatusInfo status = new StatusInfo();
				
		String name = nameText.getText();
		
		if( name == null || name.equals("") ){
			status.setError( "Name must be set" );
			return status;
		}
		
		//check to see if the name is not already in use.
		//if we are editing and the name has not changed, there is no need for this check
		if( editRI == null || !editRI.getName().equals(name) ){
			IRuntime selectedRuntime = getSelectedRuntime();
			
			/*
			 * Can only match the name if a Runtime is selected, otherwise
			 * validation returns a warning (there should be an error status because
			 * the type is not set)
			 */
			if( selectedRuntime != null ){
				
				IRuntimeInstance [] instances = selectedRuntime.getRuntimeInstances();
				boolean exists = false;
				for (int i = 0; i < instances.length; i++) {
					exists = instances[i].getName().equals(name);
					if( exists )
						break;
				}
				
				if( exists )
					status.setError( "A Toolkit with the name already exists" );
			}
			else{
				status.setWarning( "Name uniqueness cannot be confirmed because Type is not set." );
			}
		}
		
		return status;
	}
	
	
	
	protected IStatus validateType(){
		//just make sure that a type is selected
		StatusInfo status = new StatusInfo();
		
		if( typeCombo.getSelectionIndex() == -1 )
			status.setError( "Type must be set" );
		
		//check if the selected Runtime alows the user to add an instance
		else{
			RuntimeStandin rStandin = getSelectedRuntime();
			if( !rStandin.isAllowUserInstances() )
				status.setError( "The selected Toolkit type does not allow users to add new instances" );
		}
			
		return status;
	}
	
	protected IStatus validateLocation(){
		
		StatusInfo status = new StatusInfo();
		String location = locationText.getText();
		if( location == null || location.equals("") )
			status.setError( "Location must be set" );
		
		else{
			RuntimeStandin selRuntime = getSelectedRuntime();
			if( selRuntime != null ){
				//validate that the location is valid for runtime
				//an invalid location will be displayed as a warning
				final IRuntimeValidator validator = selRuntime.getValidator();
				
				//create a temporary instance that holds the location for validation
				final RuntimeInstanceStandin newRi = new RuntimeInstanceStandin();
				try {
					newRi.setLocation( (new File(location)).toURL().toString() );
					
					if( validator != null ){
						final IStatus[] ret = new IStatus[1]; 
						BusyIndicator.showWhile(getShell().getDisplay(), new Runnable() {
				            public void run() {
				                ret[0] = validator.validate(newRi);
				            }
				        });
						
						//if not valid
						if( !ret[0].isOK() )
							status.setError( "Invalid Toolkit location: " + ret[0].getMessage() );
					}
					
				} catch (MalformedURLException e) {
					e.printStackTrace();
				}
			}			
		}
		return status;
	}

	protected void handleBrowseButton(){
		DirectoryDialog dialog= new DirectoryDialog(getShell());
		dialog.setFilterPath( locationText.getText());
		dialog.setMessage( "Select the root folder of the AJAX Toolkit installation:" ); 
		String newPath= dialog.open();
		if (newPath != null) {
			locationText.setText(newPath);
		}
	}
	
	/*
	 * Return the Runtime type that matches the selection in the combo
	 */
	private RuntimeStandin getSelectedRuntime(){
		RuntimeStandin type = null;
		
		if( editRI != null ){
			//shortcut because when editing an instance, the type is not editable
			type = (RuntimeStandin)editRI.getType();
		}
		else{
			
			int selectedTypeIdx = typeCombo.getSelectionIndex();
			
			//only do the match if there is a selection in the combo
			if( selectedTypeIdx >= 0 ){
				
				for (int i = 0; i < runtimes.length; i++) {
					type = (RuntimeStandin)runtimes[i];
					if( i == selectedTypeIdx )
						break;
				}
			
			}
		}
		
		return type;
	}

}
