/*******************************************************************************
 * Copyright (c) 2006 Zend Corporation and IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.php.internal.debug.core.zend.model;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.php.internal.debug.core.model.PHPDebugElement;

/**
 * A PHP debugger threaded.
 */
public class PHPThread extends PHPDebugElement implements IThread {

    /**
     * Breakpoints this thread is suspended at or <code>null</code> if none.
     */
    private IBreakpoint[] fBreakpoints;

    /**
     * Whether this thread is stepping
     */
    private boolean fStepping = false;

    /**
     * Constructs a new thread for the given target
     * 
     * @param target
     *            VM
     */
    public PHPThread(PHPDebugTarget target) {
        super(target);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IThread#getStackFrames()
     */
    public IStackFrame[] getStackFrames() throws DebugException {
        if (isSuspended()) {
            return ((PHPDebugTarget) getDebugTarget()).getStackFrames();
        } else {
            return new IStackFrame[0];
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IThread#hasStackFrames()
     */
    public boolean hasStackFrames() throws DebugException {
        return isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IThread#getPriority()
     */
    public int getPriority() throws DebugException {
        return 0;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IThread#getTopStackFrame()
     */
    public IStackFrame getTopStackFrame() throws DebugException {
        IStackFrame[] frames = getStackFrames();
        if (frames.length > 0) {
            return frames[0];
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IThread#getName()
     */
    public String getName() throws DebugException {
        return "PHPthread";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IThread#getBreakpoints()
     */
    public IBreakpoint[] getBreakpoints() {
        if (fBreakpoints == null) {
            return new IBreakpoint[0];
        }
        return fBreakpoints;
    }

    /**
     * Sets the breakpoints this thread is suspended at, or <code>null</code>
     * if none.
     * 
     * @param breakpoints
     *            the breakpoints this thread is suspended at, or
     *            <code>null</code> if none
     */
    protected void setBreakpoints(IBreakpoint[] breakpoints) {
        fBreakpoints = breakpoints;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ISuspendResume#canResume()
     */
    public boolean canResume() {
        return isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ISuspendResume#canSuspend()
     */
    public boolean canSuspend() {
        return !isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ISuspendResume#isSuspended()
     */
    public boolean isSuspended() {
        return getDebugTarget().isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ISuspendResume#resume()
     */
    public void resume() throws DebugException {
        setStepping(false);
        getDebugTarget().resume();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ISuspendResume#suspend()
     */
    public void suspend() throws DebugException {
        setStepping(false);
        getDebugTarget().suspend();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#canStepInto()
     */
    public boolean canStepInto() {
        return isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#canStepOver()
     */
    public boolean canStepOver() {
        return isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#canStepReturn()
     */
    public boolean canStepReturn() {
        return isSuspended();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#isStepping()
     */
    public boolean isStepping() {
        return fStepping;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#stepInto()
     */
    public void stepInto() throws DebugException {
        setStepping(true);
        ((PHPDebugTarget) getDebugTarget()).stepInto();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#stepOver()
     */
    public void stepOver() throws DebugException {
        setStepping(true);
        ((PHPDebugTarget) getDebugTarget()).stepOver();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IStep#stepReturn()
     */
    public void stepReturn() throws DebugException {
        setStepping(true);
        ((PHPDebugTarget) getDebugTarget()).stepReturn();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ITerminate#canTerminate()
     */
    public boolean canTerminate() {
        return !isTerminated();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ITerminate#isTerminated()
     */
    public boolean isTerminated() {
        return getDebugTarget().isTerminated();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.ITerminate#terminate()
     */
    public void terminate() throws DebugException {
        getDebugTarget().terminate();
    }

    /**
     * Sets whether this thread is stepping
     * 
     * @param stepping
     *            whether stepping
     */
    public void setStepping(boolean stepping) {
        fStepping = stepping;
    }

    // Future method for desplaying error from the debugger client.
    public Object getError() {
        return null;
    }
}
