/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceCorrelator.java,v 1.2 2005/02/25 22:17:10 hleung Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace;

import org.eclipse.hyades.collection.correlation.BaseCorrelatorDataImpl;
import org.eclipse.hyades.collection.correlation.exceptions.InsufficientBufferLengthException;


/**
 * This is the trace correlator holding information about each method invocation.
 *  
 * @author Richard Duggan, Qiyan Li
 */
public class TraceCorrelator extends BaseCorrelatorDataImpl {

    /**
     * the class of the invoked object
     */
    private Class clazz;

    /**
     * the invoked object
     */
    private int object;

    /**
     * the name of the invoked method
     */
    private String method;

    /**
     * the classes of the parameters to the invoked method
     */
    private Class[] parmClasses;

    /**
     * the values of the parameters to the invoked method
     */
    private Object[] parmValues;

    /**
     * the names of the parameters to the invoked method
     */
    private String[] parmNames;

    /**
     * the return class of the invoked method
     */
    private Class returnClass;

    /**
     * the return value of the invoked method
     */
    private Object returnValue;

    /**
     * the UUID of the agent on the invoked side
     */
    private String agentIdRef;

    /**
     * the UUID of the process on the invoked side
     */
    private String processIdRef;

    /**
     * the UUID of the node on the invoked side
     */
    private String nodeIdRef;

    /**
     * the thread ID on the invoked side
     */
    private int threadIdRef;


    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#readBinary(byte[], int, int)
     */
    public int readBinary(byte[] buffer, int offset, int length) throws InsufficientBufferLengthException {

        /* Read the counters first. */
        offset = super.readBinary(buffer, offset, length);

        /* Only fields in the sub class are serialized.  Fields from the super class are not set
           because they are not serialized. */
        int stringLength = (int) convertBytesToInteger(buffer, offset);
        offset += 4;
        agentIdRef = new String(buffer, offset, stringLength);
        offset += stringLength;
        stringLength = (int) convertBytesToInteger(buffer, offset);
        offset += 4;
        processIdRef = new String(buffer, offset, stringLength);;
        offset += stringLength;
        stringLength = (int) convertBytesToInteger(buffer, offset);
        offset += 4;
        nodeIdRef = new String(buffer, offset, stringLength);;
        offset += stringLength;
        threadIdRef = (int) convertBytesToInteger(buffer, offset);
        offset += 4;

        return offset;
    }


    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#writeBinary(byte[], int, int)
     */
    public int writeBinary(byte[] buffer, int offset, int length) throws InsufficientBufferLengthException {

        offset = super.writeBinary(buffer, offset, length);

        byte[] byteAgentIdRef = agentIdRef.getBytes();
        byte[] byteProcessIdRef = processIdRef.getBytes();
        byte[] byteNodeIdRef = nodeIdRef.getBytes();
        int totalLength = byteAgentIdRef.length + byteProcessIdRef.length + byteNodeIdRef.length + 16;
        if (totalLength > length - offset) {
            throw new InsufficientBufferLengthException(offset + totalLength);
        }
        convertIntegerToBytes(byteAgentIdRef.length, buffer, offset);
        offset += 4;
        System.arraycopy(byteAgentIdRef, 0, buffer, offset, byteAgentIdRef.length);
        offset += byteAgentIdRef.length;
        convertIntegerToBytes(byteProcessIdRef.length, buffer, offset);
        offset += 4;
        System.arraycopy(byteProcessIdRef, 0, buffer, offset, byteProcessIdRef.length);
        offset += byteProcessIdRef.length;
        convertIntegerToBytes(byteNodeIdRef.length, buffer, offset);
        offset += 4;
        System.arraycopy(byteNodeIdRef, 0, buffer, offset, byteNodeIdRef.length);
        offset += byteNodeIdRef.length;
        convertIntegerToBytes(threadIdRef, buffer, offset);
        offset += 4;
        return offset;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.BaseCorrelatorDataImpl#invalidate()
     */
    public void invalidate() {
        super.invalidate();
        clazz = null;
        object = 0;
        method = null;
        parmClasses = null;
        parmValues = null;
        parmNames = null;
        returnClass = null;
        returnValue = null;
        agentIdRef = null;
        processIdRef = null;
        nodeIdRef = null;
        threadIdRef = 0;
    }

    /**
     * Sets a common set of parameters for an invocation.
     * 
     * @param object        the object on which the invocation was made
     * @param method        the name of the method
     * @param parmClasses   an ordered list of parameter classes
     * @param parmValues    an ordered list of parameter values
     * @param returnClass   the return type of the method, if any
     */
    public void setInvocation(int object, Class clazz, String method,
        Class[] parmClasses, Object[] parmValues, Class returnClass) {

        this.clazz =  clazz;
        this.object = object;
        this.method = method;
        this.parmClasses = parmClasses;
        this.parmValues = parmValues;
        if (returnClass != null) {
            this.returnClass = returnClass;
        }
    }


    public String getInvokedMethod() {
        return method;
    }


    public String getAgentIdRef() {
        return agentIdRef;
    }


    public String getProcessIdRef() {
        return processIdRef;
    }


    public void setAgentIdRef(String agentIdRef) {
        this.agentIdRef = agentIdRef;
    }


    public void setProcessIdRef(String processIdRef) {
        this.processIdRef = processIdRef;
    }


    public int getThreadIdRef() {
        return threadIdRef;
    }


    public void setThreadIdRef(int threadIdRef) {
        this.threadIdRef = threadIdRef;
    }


    public Class getClazz() {
        return clazz;
    }


    public String getMethod() {
        return method;
    }


    public String[] getParmNames() {
        return parmNames;
    }


    public Object[] getParmValues() {
        return parmValues;
    }


    public void setClazz(Class clazz) {
        this.clazz = clazz;
    }


    public void setMethod(String method) {
        this.method = method;
    }


    public void setParmNames(String[] parmNames) {
        this.parmNames = parmNames;
    }


    public void setParmValues(Object[] parmValues) {
        this.parmValues = parmValues;
    }
    

    public Class getReturnClass() {
        return returnClass;
    }


    public Object getReturnValue() {
        return returnValue;
    }


    public void setReturnValue(Object returnValue) {
        this.returnValue = returnValue;
    }


    public int getObject() {
        return object;
    }

    public Class[] getParmClasses() {
        return parmClasses;
    }


    public void setObject(int object) {
        this.object = object;
    }


    public void setParmClasses(Class[] parmClasses) {
        this.parmClasses = parmClasses;
    }


    public void setReturnClass(Class returnType) {
        this.returnClass = returnType;
    }


    public String getNodeIdRef() {
        return nodeIdRef;
    }


    public void setNodeIdRef(String nodeIdRef) {
        this.nodeIdRef = nodeIdRef;
    }
}
