/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecordParameterDef.java,v 1.2 2005/02/25 22:17:10 hleung Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace.util;

/**
 * This is the record for a parameter definition.  For up-to-date descriptions of the tag attributes,
 * see Event Specification for the Java Profiler at <code>http://www.eclipse.org/hyades</code>.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class RecordParameterDef {

    /**
     * the number of records to provide a unique object ID
     */
    private static int _numberRecords = 0;

    /**
     * the optional name of this variable
     */
    private final int _parameterId;

    /**
     * the optional name of this variable
     */
    private String name = null;

    /**
     * the Java class of this parameter definition
     */
    private final Class _clazz;

    /**
     * the Java method to which this parameter definition belongs to
     */
    private final int _methodIdRef;

    /**
     * the position of this parameter in the parameter list with the index of the first parameter being 0
     */
    private final int _index;


    /**
     * Create a record for a parameter definition with a unique ID.
     *
     * @param clazz         the Java class of the parameter 
     * @param methodIdRef   the unique ID distinguishing the method
     * @param index         the index position of the parameter in the parameter list
     */
    public RecordParameterDef(Class clazz, int methodIdRef, int index) {

        /* Set the mandatory fields. */
        _clazz = clazz;
        _methodIdRef = methodIdRef;
        _index = index;

        /* Obtain a unique ID for this object. */
        synchronized (this.getClass()) {
            _parameterId = ++_numberRecords;
        }
    }


    /**
     * Converts this record into a string.
     * 
     * @return  this record as a string. 
     */
    public String toString() {

        return "<parameterDef parameterId=\"" + _parameterId + "\" type=\"" + Utilities.getJniNotation(_clazz) +
            "\" methodIdRef=\"" + _methodIdRef + "\" index=\"" + _index + "\"" +
            (name != null ? " name=\"" + name + "\"" : "") + "/>";
    }


    public String getName() {
        return name;
    }


    public void setName(String name) {
        this.name = name;
    }


    public Class getClazz() {
        return _clazz;
    }


    public int getIndex() {
        return _index;
    }


    public int getMethodIdRef() {
        return _methodIdRef;
    }


    public int getParameterId() {
        return _parameterId;
    }
}
